package pt.digitalis.dif.dataintegration;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.map.CaseInsensitiveMap;
import org.hibernate.Session;
import pt.digitalis.dif.dataintegration.events.publishers.DISetEventCategory;
import pt.digitalis.dif.dataintegration.events.publishers.DISetEventPublisher;
import pt.digitalis.dif.dataintegration.events.subscribers.DISetEventSubscriber;
import pt.digitalis.dif.dem.managers.impl.model.DIFRepositoryFactory;
import pt.digitalis.dif.dem.managers.impl.model.IDataIntegrationService;
import pt.digitalis.dif.dem.managers.impl.model.data.DataIntegrationField;
import pt.digitalis.dif.dem.managers.impl.model.data.DataIntegrationRecord;
import pt.digitalis.dif.dem.managers.impl.model.data.DataIntegrationSet;
import pt.digitalis.dif.events.api.IEventPublisher;
import pt.digitalis.dif.events.api.IEventSubscriber;
import pt.digitalis.dif.events.exceptions.EventException;
import pt.digitalis.dif.events.exceptions.EventSubscriptionExistsException;
import pt.digitalis.dif.events.impl.EventsManager;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.dif.model.dataset.Query;
import pt.digitalis.dif.model.dataset.SortMode;
import pt.digitalis.dif.model.utils.AbstractBeanAttributes;
import pt.digitalis.dif.model.utils.AbstractBeanAttributesDefinition;
import pt.digitalis.utils.config.IConfigurations;

import java.lang.reflect.InvocationTargetException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * The type Data integration manager.
 */
public class DataIntegrationManager
{

    /** The Instance. */
    private static DataIntegrationManager instance = null;

    /** The Data integration service. */
    private static IDataIntegrationService dataIntegrationService =
            DIFIoCRegistry.getRegistry().getImplementation(IDataIntegrationService.class);

    /** The Fields by root record. */
    Map<Long, Map<String, DataIntegrationField>> fieldsByRootRecord = new CaseInsensitiveMap();

    /** The Data integration set id. */
    private Long dataIntegrationSetId;

    /** The Fields by record. */
    private Map<Long, List<DataIntegrationField>> fieldsByRecord = null;

    /**
     * Get instance data integration manager.
     *
     * @param dataIntegrationSetId the data integration set id
     */
    public DataIntegrationManager(Long dataIntegrationSetId)
    {
        this.dataIntegrationSetId = dataIntegrationSetId;
    }

    /**
     * Gets most recent set.
     *
     * @param entityType the entity type
     * @param entityId   the entity id
     *
     * @return the most recent set
     *
     * @exception DataSetException the data set exception
     */
    public static DataIntegrationSet getMostRecentSet(String entityType, String entityId) throws DataSetException
    {
        Query<DataIntegrationSet> query = dataIntegrationService.getDataIntegrationSetDataSet().query();
        query.in(DataIntegrationSet.FK().dataIntegrationRecords().ENTITYTYPE(), entityType);
        query.in(DataIntegrationSet.FK().dataIntegrationRecords().ENTITYID(), entityId);
        query.sortBy(DataIntegrationSet.Fields.ID, SortMode.DESCENDING);

        return query.singleValue();
    }

    /**
     * Translate object to data integration data integration set.
     *
     * @param entityId the entity id
     * @param obj      the obj
     *
     * @return the data integration set
     *
     * @exception InvocationTargetException the invocation target exception
     * @exception IllegalAccessException    the illegal access exception
     * @exception NoSuchMethodException     the no such method exception
     */
    public static DataIntegrationSet translateObjectToDataIntegration(String entityId, Object obj)
            throws InvocationTargetException, IllegalAccessException, NoSuchMethodException
    {
        DataIntegrationSet dis = new DataIntegrationSet();
        dis.setCreationDate(new Timestamp(new Date().getTime()));

        String entityType = obj.getClass().getSimpleName();

        DataIntegrationRecord record = DataIntegrationManager.translateObjectToFields(entityType, entityId, "U", obj);

        Set<DataIntegrationRecord> records = new HashSet<DataIntegrationRecord>();
        records.add(record);
        dis.setDataIntegrationRecords(records);

        return dis;
    }

    /**
     * Translate object to fields data integration record.
     *
     * @param entityType  the entity type
     * @param entityId    the entity id
     * @param operationId the operation id
     * @param obj         the obj
     *
     * @return the data integration record
     *
     * @exception InvocationTargetException the invocation target exception
     * @exception IllegalAccessException    the illegal access exception
     * @exception NoSuchMethodException     the no such method exception
     */
    public static DataIntegrationRecord translateObjectToFields(String entityType, String entityId, String operationId,
            Object obj) throws InvocationTargetException, IllegalAccessException, NoSuchMethodException
    {
        return translateObjectToFields(entityType, entityId, operationId, obj, new ArrayList<Integer>());
    }

    /**
     * Translate object to record set.
     *
     * @param entityType       the entity type
     * @param entityId         the entity id
     * @param operationId      the operation id
     * @param obj              the obj
     * @param processedClasses the processed classes
     *
     * @return the set
     *
     * @exception InvocationTargetException the invocation target exception
     * @exception IllegalAccessException    the illegal access exception
     * @exception NoSuchMethodException     the no such method exception
     */
    private static DataIntegrationRecord translateObjectToFields(String entityType, String entityId, String operationId,
            Object obj, List<Integer> processedClasses)
            throws InvocationTargetException, IllegalAccessException, NoSuchMethodException
    {
        DataIntegrationRecord newRecord = new DataIntegrationRecord();
        newRecord.setEntityType(entityType);
        newRecord.setEntityId(entityId);
        newRecord.setOperationId(operationId);

        Set<DataIntegrationField> fields = new HashSet<DataIntegrationField>();

        if (obj instanceof Collection)
        {

            int count = 0;
            String objectListClass = null;
            for (Object listObj : (Collection) obj)
            {
                objectListClass = listObj.getClass().getSimpleName();
                if (listObj.getClass().getCanonicalName().startsWith("java.lang"))
                {
                    DataIntegrationField field = new DataIntegrationField();
                    field.setFieldId(objectListClass + "[" + count + "]");
                    field.setValue(listObj.toString());
                    field.setDataIntegrationRecordByDataIntegrationRecordId(newRecord);

                    fields.add(field);
                }
                else
                {
                    DataIntegrationRecord newRecordList = DataIntegrationManager
                            .translateObjectToFields(objectListClass, count + "", operationId, listObj,
                                    processedClasses);

                    DataIntegrationField field = new DataIntegrationField();
                    field.setFieldId(objectListClass + "[" + count + "]");
                    field.setValue(DataIntegrationFieldType.LIST.getId());
                    field.setDataIntegrationRecordByDataIntegrationRecordId(newRecord);
                    field.setDataIntegrationRecordByRelationRecordId(newRecordList);
                    fields.add(field);
                }
                count++;
            }
        }
        else
        {

            Map<String, Object> map = PropertyUtils.describe(obj);
            boolean isBeanAttributes = obj instanceof AbstractBeanAttributes;

            processedClasses.add(System.identityHashCode(obj));

            for (Map.Entry entry : map.entrySet())
            {
                if (!entry.getKey().toString().equals("class") && (!isBeanAttributes || (isBeanAttributes &&
                                                                                         !(entry.getValue() instanceof AbstractBeanAttributesDefinition))))
                {
                    DataIntegrationField field = new DataIntegrationField();
                    field.setFieldId(entry.getKey().toString());
                    field.setDataIntegrationRecordByDataIntegrationRecordId(newRecord);

                    if (!entry.getKey().toString().equals("class") && (entry.getValue() == null ||
                                                                       entry.getValue().getClass().getCanonicalName()
                                                                               .startsWith("java.lang")))
                    {
                        if (entry.getValue() != null)
                        {
                            field.setValue(entry.getValue().toString());
                        }
                    }
                    else
                    {
                        try
                        {
                            if (!processedClasses.contains(System.identityHashCode(entry.getValue())))
                            {
                                DataIntegrationRecord newRecordObj =
                                        translateObjectToFields(entry.getValue().getClass().getSimpleName(),
                                                entry.getValue().getClass().getSimpleName() + "Element", operationId,
                                                entry.getValue(), processedClasses);

                                field.setValue(entry.getValue().getClass().getSimpleName());
                                field.setDataIntegrationRecordByDataIntegrationRecordId(newRecord);
                                field.setDataIntegrationRecordByRelationRecordId(newRecordObj);
                            }
                        }
                        catch (Exception e)
                        {
                            //Do nothing
                        }
                    }
                    fields.add(field);
                }
            }
        }
        newRecord.setDataIntegrationFieldsForDataIntegrationRecordId(fields);

        return newRecord;
    }

    /**
     * Init events.
     *
     * @exception EventException the event exception
     */
    public static void initEvents() throws EventException
    {
        IEventPublisher publisher =
                EventsManager.getInstance().getPublisherByID(DISetEventPublisher.class.getSimpleName());
        IEventSubscriber subscriber =
                EventsManager.getInstance().getSubscriberByID(DISetEventSubscriber.class.getSimpleName());
        try
        {
            EventsManager.getInstance()
                    .subscribeEvent(publisher, DISetEventCategory.DI_SET_AVAILABLE.getId(), subscriber);
        }
        catch (EventSubscriptionExistsException e)
        {
            // All is fine. Nothing to do!
        }
    }

    /**
     * Insert data integration set data integration set.
     *
     * @param processId the process id
     * @param entityId  the entity id
     * @param obj       the obj
     *
     * @return the data integration set
     *
     * @exception DataSetException          the data set exception
     * @exception IllegalAccessException    the illegal access exception
     * @exception NoSuchMethodException     the no such method exception
     * @exception InvocationTargetException the invocation target exception
     */
    public static DataIntegrationSet insertDataIntegrationSet(String processId, String entityId, Object obj)
            throws DataSetException, IllegalAccessException, NoSuchMethodException, InvocationTargetException
    {
        DataIntegrationSet dis = DataIntegrationManager.translateObjectToDataIntegration(entityId, obj);

        return insertDataIntegrationSet(dis, processId);
    }

    /**
     * Insert data integration set data integration set.
     *
     * @param dis       the dis
     * @param processId the process id
     *
     * @return the data integration set
     *
     * @exception DataSetException the data set exception
     * @exception DataSetException the data set exception
     */
    private static DataIntegrationSet insertDataIntegrationSet(DataIntegrationSet dis, String processId)
            throws DataSetException
    {
        boolean wasActive = DIFRepositoryFactory.openTransaction();

        /** The Configs. */
        IConfigurations configs = DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class);

        try
        {
            Session session = DIFRepositoryFactory.getSession();

            Set<DataIntegrationRecord> recordsToInsert = dis.getDataIntegrationRecords();
            dis.setDataIntegrationRecords(null);
            dis.setProcessId(processId);
            dis.setConfigId(configs.getGeneralPrefix());

            session.save(dis);

            Set<DataIntegrationRecord> newRecords = new HashSet<DataIntegrationRecord>();
            for (DataIntegrationRecord record : recordsToInsert)
            {
                record = insertDataIntegrationRecord(session, dis, record);
                record.setRoot(true);

                if (record != null)
                {
                    newRecords.add(record);
                }
            }
            if (!wasActive)
            {
                DIFRepositoryFactory.getSession().getTransaction().commit();
            }
            return dis;
        }
        catch (DataSetException e)
        {
            e.printStackTrace();

            if (!wasActive)
            {
                DIFRepositoryFactory.getSession().getTransaction().rollback();
            }
            throw e;
        }
    }

    /**
     * Insert data integration record data integration record.
     *
     * @param session the session
     * @param dis     the dis
     * @param record  the record
     *
     * @return the data integration record
     *
     * @exception DataSetException the data set exception
     */
    private static DataIntegrationRecord insertDataIntegrationRecord(Session session, DataIntegrationSet dis,
            DataIntegrationRecord record) throws DataSetException
    {

        Set<DataIntegrationField> fieldsToInsert = record.getDataIntegrationFieldsForDataIntegrationRecordId();
        if (fieldsToInsert != null && fieldsToInsert.size() > 0)
        {
            record.setDataIntegrationSet(dis);
            record.setDataIntegrationFieldsForDataIntegrationRecordId(null);

            session.save(record);

            Set<DataIntegrationField> newFields = new HashSet<DataIntegrationField>();
            for (DataIntegrationField field : fieldsToInsert)
            {
                if (field.getDataIntegrationRecordByRelationRecordId() != null)
                {
                    DataIntegrationRecord newRecord = insertDataIntegrationRecord(session, dis,
                            field.getDataIntegrationRecordByRelationRecordId());

                    field.setDataIntegrationRecordByRelationRecordId(newRecord);
                }

                field.setDataIntegrationRecordByDataIntegrationRecordId(record);
                session.save(field);

                newFields.add(field);
            }

            return record;
        }
        else
        {
            return null;
        }
    }

    /**
     * Gets field.
     *
     * @param record    the record
     * @param fieldPath the field path
     *
     * @return the field
     *
     * @exception DataSetException the data set exception
     */
    public DataIntegrationField getField(DataIntegrationRecord record, String fieldPath) throws DataSetException
    {
        if (!fieldsByRootRecord.containsKey(record.getId()))
        {
            Map<String, DataIntegrationField> fields = this.getFields("", record);
            fieldsByRootRecord.put(record.getId(), fields);
        }

        return fieldsByRootRecord.get(record.getId()).get(fieldPath);
    }

    /**
     * Gets field value.
     *
     * @param record    the record
     * @param fieldPath the field path
     *
     * @return the field value
     *
     * @exception DataSetException the data set exception
     */
    public String getFieldValue(DataIntegrationRecord record, String fieldPath) throws DataSetException
    {
        String result = null;
        DataIntegrationField field = this.getField(record, fieldPath);
        if (field != null)
        {
            result = field.getValue();
        }

        return result;
    }

    /**
     * Gets fields.
     *
     * @param fieldParentName the field parent name
     * @param record          the record
     *
     * @return the fields
     *
     * @exception DataSetException the data set exception
     * @exception DataSetException the data set exception
     */
    public Map<String, DataIntegrationField> getFields(String fieldParentName, DataIntegrationRecord record)
            throws DataSetException
    {
        if (fieldParentName != "")
        {
            fieldParentName += ".";
        }
        Map<String, DataIntegrationField> fields = new HashMap<String, DataIntegrationField>();
        List<DataIntegrationField> fieldsList = this.getFieldsByRecords().get(record.getId());
        for (DataIntegrationField field : fieldsList)
        {
            if (field.getDataIntegrationRecordByRelationRecordId() != null)
            {
                fields.putAll(this.getFields(fieldParentName + field.getFieldId(),
                        field.getDataIntegrationRecordByRelationRecordId()));
            }

            fields.put(fieldParentName + field.getFieldId(), field);
        }

        return fields;
    }

    /**
     * Gets fields by records.
     *
     * @return the fields by records
     *
     * @exception DataSetException the data set exception
     */
    private Map<Long, List<DataIntegrationField>> getFieldsByRecords() throws DataSetException
    {
        if (fieldsByRecord == null)
        {
            Query<DataIntegrationField> queryAllFields =
                    dataIntegrationService.getDataIntegrationFieldDataSet().query();
            queryAllFields.equals(DataIntegrationField.FK().dataIntegrationRecordByDataIntegrationRecordId()
                    .dataIntegrationSet().ID(), this.dataIntegrationSetId.toString());

            fieldsByRecord = new HashMap<Long, List<DataIntegrationField>>();

            for (DataIntegrationField field : queryAllFields.asList())
            {
                List<DataIntegrationField> listFields =
                        fieldsByRecord.get(field.getDataIntegrationRecordByDataIntegrationRecordId().getId());
                if (listFields == null)
                {
                    listFields = new ArrayList<DataIntegrationField>();
                }
                listFields.add(field);

                fieldsByRecord.put(field.getDataIntegrationRecordByDataIntegrationRecordId().getId(), listFields);
            }
        }

        return fieldsByRecord;
    }

    /**
     * Gets record fields.
     *
     * @param recordId the record id
     *
     * @return the record fields
     *
     * @exception DataSetException the data set exception
     */
    public List<DataIntegrationField> getRecordFields(Long recordId) throws DataSetException
    {
        return this.getFieldsByRecords().get(recordId);
    }

    /**
     * Gets root records.
     *
     * @return root records
     *
     * @exception DataSetException the data set exception
     */
    public List<DataIntegrationRecord> getRootRecords() throws DataSetException
    {
        Query<DataIntegrationRecord> queryRootRecord = dataIntegrationService.getDataIntegrationRecordDataSet().query();
        queryRootRecord
                .equals(DataIntegrationRecord.FK().dataIntegrationSet().ID(), this.dataIntegrationSetId.toString());
        queryRootRecord.equals(DataIntegrationRecord.Fields.ROOT, Boolean.TRUE.toString());

        return queryRootRecord.asList();
    }
}

