/**
 * 2014, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dem.managers.impl;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import net.sf.json.JSONNull;
import net.sf.json.JSONObject;
import net.sf.json.JSONSerializer;
import net.sf.json.JsonConfig;
import net.sf.json.processors.JsonBeanProcessor;
import pt.digitalis.dif.controller.interfaces.IModelManager;
import pt.digitalis.dif.dem.interfaces.ICustomFormDefinition;
import pt.digitalis.dif.dem.managers.impl.model.IPersistentFormsService;
import pt.digitalis.dif.dem.managers.impl.model.data.PersistentForm;
import pt.digitalis.dif.dem.objects.CustomFormDefinition;
import pt.digitalis.dif.dem.objects.FeatureState;
import pt.digitalis.dif.dem.objects.FormFieldCustomization;
import pt.digitalis.dif.dem.objects.messages.Message;
import pt.digitalis.dif.dem.objects.messages.MessageList;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.common.collections.CaseInsentiveArrayList;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 29/01/2014
 */
public class CustomFormManagerDBImpl extends CustomFormManagerDummyImpl {

    /** T if the DIFDatabaseRepository is available */
    private boolean databasePersistent = false;

    /**
     * Default constructor
     */
    public CustomFormManagerDBImpl()
    {
        IModelManager difDatabaseRepositoryImpl = DIFIoCRegistry.getRegistry().getImplementation(IModelManager.class,
                DIFRepositoryDBModelManager.MODEL_ID);

        try
        {
            this.databasePersistent = difDatabaseRepositoryImpl.isEnabled();
        }
        catch (InternalFrameworkException e)
        {
            this.databasePersistent = false;
        }
    }

    /**
     * @see pt.digitalis.dif.dem.managers.impl.CustomFormManagerDummyImpl#getConfiguration(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    @Override
    public CustomFormDefinition getConfiguration(String stageID, String formName, String businessConfigurationID)
    {
        stageID = stageID.toLowerCase();
        formName = formName.toLowerCase();

        if (StringUtils.isBlank(businessConfigurationID))
            businessConfigurationID = null;

        CustomFormDefinition customFormDef = (CustomFormDefinition) super.getConfiguration(stageID, formName,
                businessConfigurationID);

        if (this.databasePersistent)
        {
            String persistentConfigID = stageID + ":" + formName
                    + (businessConfigurationID == null ? "" : ":" + businessConfigurationID);

            IPersistentFormsService dbService = DIFIoCRegistry.getRegistry()
                    .getImplementation(IPersistentFormsService.class);
            try
            {
                PersistentForm persistentForm = dbService.getPersistentFormDataSet().query()
                        .equals(PersistentForm.Fields.FORMID, persistentConfigID).singleValue();

                if (persistentForm != null)
                {
                    // Parameter/field exclusions
                    if (StringUtils.isNotBlank(persistentForm.getDisabledFields()))
                        customFormDef.setExcludedParameters(new CaseInsentiveArrayList(
                                Arrays.asList(persistentForm.getDisabledFields().split(","))));

                    Map<String, FormFieldCustomization> customizations = new HashMap<String, FormFieldCustomization>();

                    // Parameter/field customization
                    if (StringUtils.isNotBlank(persistentForm.getFieldDefs()))
                    {
                        JSONObject fieldCustomizationsJSONObj = (JSONObject) JSONSerializer
                                .toJSON(persistentForm.getFieldDefs());

                        for (Object fieldIDObj: fieldCustomizationsJSONObj.keySet())
                        {
                            String fieldID = fieldIDObj.toString();
                            FormFieldCustomization customization = new FormFieldCustomization(fieldID);
                            JSONObject customizationJSONObj = fieldCustomizationsJSONObj.getJSONObject(fieldID);

                            // Mandatory value
                            if (customizationJSONObj.containsKey("mandatory"))
                            {
                                FeatureState mandatory = FeatureState
                                        .valueOf(customizationJSONObj.getString("mandatory"));

                                customization.setMandatory(mandatory);
                            }

                            // Messages
                            if (customizationJSONObj.containsKey("messages"))
                            {
                                JSONObject messagesJSONObj = customizationJSONObj.getJSONObject("messages");
                                MessageList messages = new MessageList();

                                for (Object languageIDObj: messagesJSONObj.keySet())
                                {
                                    String languageID = languageIDObj.toString();
                                    @SuppressWarnings("unchecked")
                                    Map<String, Object> messagesMap = (Map<String, Object>) JSONObject
                                            .toBean(messagesJSONObj.getJSONObject(languageID), Map.class);

                                    for (Entry<String, Object> message: messagesMap.entrySet())
                                    {
                                        if (message.getValue() != null)
                                        {
                                            Class<?> messageValueClass = message.getValue().getClass();

                                            if (messageValueClass != JSONNull.class
                                                    && StringUtils.isNotBlank(message.getValue().toString()))
                                                messages.addMessageTranslation(message.getKey(), languageID,
                                                        new Message(message.getValue().toString()));
                                        }
                                    }
                                }

                                customization.setMessages(messages);
                            }

                            customizations.put(fieldID, customization);
                        }
                    }

                    customFormDef.setCustomizedParameters(customizations);
                }
            }
            catch (DataSetException e)
            {
                e.printStackTrace();
            }
        }

        return customFormDef;
    }

    /**
     * Creates a {@link JsonConfig} instance that knows how to convert {@link FormFieldCustomization} objects
     * 
     * @return the JSONConfig with the proper {@link JsonBeanProcessor} registered
     */
    private JsonConfig getJSONConfigForFormFieldCustomizationSupport()
    {
        // Define the JSON processor for our class
        JsonConfig jsonConfig = new JsonConfig();
        jsonConfig.registerJsonBeanProcessor(FormFieldCustomization.class, new JsonBeanProcessor() {

            public JSONObject processBean(Object bean, JsonConfig jsonConfig)
            {
                if (!(bean instanceof FormFieldCustomization))
                {
                    return new JSONObject(true); // marks the object as null
                }
                FormFieldCustomization custom = (FormFieldCustomization) bean;
                JSONObject result = new JSONObject();

                if (custom.getMandatory() != FeatureState.DEFAULT)
                    result.element("mandatory", custom.getMandatory());

                // Has messages... convert them...
                if (custom.getMessages() != null && !custom.getMessages().getMessageIDs().isEmpty())
                {
                    Map<String, Map<String, String>> messagesByLanguage = new HashMap<String, Map<String, String>>();

                    // Determine existing languages from the messages
                    for (String messageID: custom.getMessages().getMessageIDs())
                    {
                        for (String languageID: custom.getMessages().getMessageTranslations(messageID).getTranslations()
                                .keySet())
                        {
                            Message message = custom.getMessages().getMessageTranslations(messageID).getTranslations()
                                    .get(languageID);

                            if (message != null && StringUtils.isNotBlank(message.getMessage()))
                            {
                                if (!messagesByLanguage.containsKey(languageID))
                                    messagesByLanguage.put(languageID, new HashMap<String, String>());

                                messagesByLanguage.get(languageID).put(messageID, message.getMessage());
                            }
                        }
                    }

                    result.element("messages", messagesByLanguage);
                }

                return result;
            }
        });

        return jsonConfig;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.impl.CustomFormManagerDummyImpl#updateConfiguration(pt.digitalis.dif.dem.interfaces.ICustomFormDefinition)
     */
    @Override
    public void updateConfiguration(ICustomFormDefinition definition)
    {
        if (this.databasePersistent)
        {
            String persistentConfigID = definition.getStageID() + ":" + definition.getFormName()
                    + (definition.getBusinessConfigurationID() == null ? ""
                            : ":" + definition.getBusinessConfigurationID());

            IPersistentFormsService dbService = DIFIoCRegistry.getRegistry()
                    .getImplementation(IPersistentFormsService.class);
            try
            {
                PersistentForm persistentForm = dbService.getPersistentFormDataSet().query()
                        .equals(PersistentForm.Fields.FORMID, persistentConfigID).singleValue();

                if (persistentForm == null)
                {
                    persistentForm = new PersistentForm();
                    persistentForm.setFormId(persistentConfigID);
                }

                // Parameter/field exclusions
                persistentForm.setDisabledFields(
                        CollectionUtils.listToCommaSeparatedString(definition.getExcludedParameters()));

                // Parameter/field mandatory & messages customization
                if (!definition.getCustomizedParameters().isEmpty())
                {
                    String fieldDefs = JSONObject.fromObject(definition.getCustomizedParameters(),
                            getJSONConfigForFormFieldCustomizationSupport()).toString();

                    persistentForm.setFieldDefs(fieldDefs);
                }

                if (persistentForm.getId() == null)
                    dbService.getPersistentFormDataSet().insert(persistentForm);
                else
                    dbService.getPersistentFormDataSet().update(persistentForm);
            }
            catch (DataSetException e)
            {
                e.printStackTrace();
            }
        }
        else
            super.updateConfiguration(definition);
    }
}
