/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.security.objects;

import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.utils.ObjectFormatter;

/**
 * Represents a grant on a resource for a given entity (user or group).
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created Dec 10, 2007
 */
public class ACLEntry {

    /**
     * Converts an entity name to the appropriate Entity.
     * 
     * @param entityName
     *            the entity name
     * @return the entity
     */
    static public Entity getEntityTypeFromName(String entityName)
    {

        Entity result = null;

        entityName = entityName.toLowerCase();

        if (entityName.equals(Entity.VALIDATOR.getName().toLowerCase()))
            result = Entity.VALIDATOR;
        else if (entityName.equals(Entity.PROVIDER.getName().toLowerCase()))
            result = Entity.PROVIDER;
        else if (entityName.equals(Entity.APPLICATION.getName().toLowerCase()))
            result = Entity.APPLICATION;
        else if (entityName.equals(Entity.SERVICE.getName().toLowerCase()))
            result = Entity.SERVICE;
        else if (entityName.equals(Entity.STAGE.getName().toLowerCase()))
            result = Entity.STAGE;

        return result;
    }

    /**
     * Converts an Entity to it's type name.
     * 
     * @param entity
     *            the entity
     * @return the entity type name
     */
    static public String getTypeNameFromEntity(Entity entity)
    {
        String result = null;

        if (entity.equals(Entity.VALIDATOR))
            result = Entity.VALIDATOR.getName();
        else if (entity.equals(Entity.PROVIDER))
            result = Entity.PROVIDER.getName();
        else if (entity.equals(Entity.APPLICATION))
            result = Entity.APPLICATION.getName();
        else if (entity.equals(Entity.SERVICE))
            result = Entity.SERVICE.getName();
        else if (entity.equals(Entity.STAGE))
            result = Entity.STAGE.getName();
        else
            result = "";

        return result.toLowerCase();
    }

    /** The id of the resource (entity). */
    private String entityID;

    /** The type of resource to grant access to. */
    private Entity entityType;

    /** If it's a group privilege holds the group ID. */
    private String groupID;

    /** Is this ACL entry default? */
    private boolean isDefault;

    /** Is this ACL entry enabled? */
    private boolean isEnabled;

    /** Is the accessed id public? */
    private boolean publicAccess;

    /** If it's a user privilege holds the user ID. */
    private String userID;

    /** Default ctor. */
    public ACLEntry()
    {
        this.userID = null;
        this.groupID = null;
        this.entityType = null;
        this.entityID = null;
        this.publicAccess = false;
        this.isEnabled = true;
        this.isDefault = false;
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj)
    {
        ACLEntry entry = (ACLEntry) obj;
        boolean same = true;

        // Validate ownership...
        if (isUserACL())
            same = same && this.getUserID().equals(entry.getUserID());
        else if (isGroupACL())
            same = same && this.getGroupID().equals(entry.getGroupID());
        // else, they can still be equal ('same' is T)

        // Validate object
        if (this.entityID != null && entry.entityID != null)
            same = same && this.getEntityID().equals(entry.getEntityID());

        if (this.entityType != null && entry.entityType != null)
            same = same && this.getEntityType().equals(entry.getEntityType());

        same = same && this.isDefault() == entry.isDefault();
        same = same && this.isEnabled() == entry.isEnabled();
        same = same && this.isPublicAccess() == entry.isPublicAccess();

        return same;
    }

    /**
     * Returns the ID of the entity associated with this entity.
     * 
     * @return the entityID
     */
    public String getEntityID()
    {
        return entityID;
    }

    /**
     * Returns the entity type.
     * 
     * @return the entityType
     */
    public Entity getEntityType()
    {
        return entityType;
    }

    /**
     * Returns the group ID.
     * 
     * @return the groupID
     */
    public String getGroupID()
    {
        return groupID;
    }

    /**
     * Returns the user ID.
     * 
     * @return the userID
     */
    public String getUserID()
    {
        return userID;
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode()
    {

        final int ODD_PRIME_FACTOR = 37;
        final int ODD_PRIME_TRUE = 1231;
        final int ODD_PRIME_FALSE = 1237;

        int result = 17;

        if (this.isUserACL())
            result = ODD_PRIME_FACTOR * result + this.userID.hashCode();
        else if (this.isGroupACL())
            result = ODD_PRIME_FACTOR * result + this.groupID.hashCode();

        result = ODD_PRIME_FACTOR * result + (publicAccess ? ODD_PRIME_TRUE : ODD_PRIME_FALSE);
        result = ODD_PRIME_FACTOR * result + (this.entityID == null ? 0 : this.entityID.hashCode());
        result = ODD_PRIME_FACTOR * result + (this.entityType == null ? 0 : this.entityType.hashCode());
        result = ODD_PRIME_FACTOR * result + (isDefault ? ODD_PRIME_TRUE : ODD_PRIME_FALSE);
        result = ODD_PRIME_FACTOR * result + (isEnabled ? ODD_PRIME_TRUE : ODD_PRIME_FALSE);

        return result;
    }

    /**
     * Checks if this is a default entry.
     * 
     * @return T if this entry is default, F otherwise
     */
    public boolean isDefault()
    {
        return this.isDefault;
    }

    /**
     * Checks if this entry is enabled.
     * 
     * @return T if this entry is enabled, F otherwise
     */
    public boolean isEnabled()
    {
        return this.isEnabled;
    }

    /**
     * Checks if it is a group entry.
     * 
     * @return T if it is a group grant
     */
    public boolean isGroupACL()
    {
        return (this.groupID != null);
    }

    /**
     * Checks if the entry has public access.
     * 
     * @return T if it's a public access entry, F otherwise
     */
    public boolean isPublicAccess()
    {
        return publicAccess;
    }

    /**
     * Checks if it is a user entry.
     * 
     * @return T if it is a user grant
     */
    public boolean isUserACL()
    {
        return (this.userID != null);
    }

    /**
     * Sets a new value for 'isDefault' field
     * 
     * @param isDefault
     *            the new default value
     */
    public void setDefault(boolean isDefault)
    {
        this.isDefault = isDefault;
    }

    /**
     * Sets a new value for 'isEnabled' field
     * 
     * @param isEnabled
     *            the new 'isEnabled' value
     */
    public void setEnabled(boolean isEnabled)
    {
        this.isEnabled = isEnabled;
    }

    /**
     * Sets the entity ID.
     * 
     * @param entityID
     *            the entityID to set
     */
    public void setEntityID(String entityID)
    {
        this.entityID = entityID;
    }

    /**
     * Sets the entity type.
     * 
     * @param entityType
     *            the entityType to set
     */
    public void setEntityType(Entity entityType)
    {
        this.entityType = entityType;
    }

    /**
     * Sets the group ID.
     * 
     * @param groupID
     *            the groupID to set
     */
    public void setGroupID(String groupID)
    {
        this.groupID = groupID;
        this.userID = null;
        this.publicAccess = false;
    }

    /**
     * Sets the ACL entry as public.
     */
    public void setPublicAccess()
    {
        this.publicAccess = true;
        this.userID = null;
        this.groupID = null;
    }

    /**
     * Sets the user ID.
     * 
     * @param userID
     *            the userID to set
     */
    public void setUserID(String userID)
    {
        this.userID = userID;
        this.groupID = null;
        this.publicAccess = false;
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();
        formatter.addItemIfNotNull("User ID", userID);
        formatter.addItemIfNotNull("Group ID", groupID);
        formatter.addItemIfNotNull("Public Access", publicAccess);
        formatter.addItemIfNotNull("Entity Type", entityType);
        formatter.addItemIfNotNull("Entity ID", entityID);
        formatter.addItemIfNotNull("Default", isDefault);
        formatter.addItemIfNotNull("Enabled", isEnabled);

        return formatter.getFormatedObject();
    }

}
