/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters.types;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.imageio.ImageIO;

import org.imgscalr.Scalr;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.objects.DIFSessionConstants;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterError;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorList;
import pt.digitalis.dif.dem.objects.parameters.errors.ParameterErrorType;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryException;
import pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager;
import pt.digitalis.dif.utils.logging.DIFLogger;

/**
 * /** This class will define a numeric Parameter.
 * <p>
 * It will hold information relative to the Parameter value, ID key and validation constraints.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Nov 23, 2007
 */
public class DocumentParameter extends AbstractParameter<DocumentRepositoryEntry> {

    /** the ID if a companion hidden field that includes the document repository ID when a document is resubmitted */
    private static final String DOCUMENT_ID_HIDDEN_FIELD = "_docid";
    /** The list of supported classes to define in the concrete implementations */
    @SuppressWarnings("serial")
    final static private List<String> supportedClasses = new ArrayList<String>() {

        {
            add(DocumentRepositoryEntry.class.getCanonicalName());
        }
    };

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getSupportedClasses()
     */
    public List<String> getSupportedClasses()
    {
        return supportedClasses;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsBigDecimal(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    public BigDecimal getValueAsBigDecimal(IDIFContext context) throws ParameterException
    {
        // Impossible conversion. Return an exception
        throw new ParameterException(getMessages().getMessages(context.getLanguage()).get("invalidUsage"), this);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsBoolean(IDIFContext)
     */
    public boolean getValueAsBoolean(IDIFContext context) throws ParameterException
    {
        // Impossible conversion. Return an exception
        throw new ParameterException(getMessages().getMessages(context.getLanguage()).get("invalidUsage"), this);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDate(IDIFContext)
     */
    public Date getValueAsDate(IDIFContext context) throws ParameterException
    {
        // Impossible conversion. Return an exception
        throw new ParameterException(getMessages().getMessages(context.getLanguage()).get("invalidUsage"), this);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#getValueAsDocument(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    @Override
    public DocumentRepositoryEntry getValueAsDocument(IDIFContext context) throws ParameterException
    {
        return getValue(context);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsDouble(IDIFContext)
     */
    public Double getValueAsDouble(IDIFContext context) throws ParameterException
    {
        // Impossible conversion. Return an exception
        throw new ParameterException(getMessages().getMessages(context.getLanguage()).get("invalidUsage"), this);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#getValueAsLong(IDIFContext)
     */
    public Long getValueAsLong(IDIFContext context) throws ParameterException
    {
        // Impossible conversion. Return an exception
        throw new ParameterException(getMessages().getMessages(context.getLanguage()).get("invalidUsage"), this);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#getValueAsString(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    @Override
    public String getValueAsString(IDIFContext context) throws ParameterException
    {
        // Impossible conversion. Return an exception
        throw new ParameterException(getMessages().getMessages(context.getLanguage()).get("invalidUsage"), this);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#isNumeric()
     */
    public boolean isNumeric()
    {
        return false;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#isStringSetterSupported()
     */
    @Override
    public boolean isStringSetterSupported()
    {
        return false;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#refreshParameterValue(pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    @Override
    public ParameterErrorList refreshParameterValue(IStageInstance stageInstance)
    {
        ParameterErrorList errorList = super.refreshParameterValue(stageInstance);

        if (!this.hasValue())
        {
            Object previousDocumentID = stageInstance.getContext().getRequest()
                    .getParameter(this.getId() + DOCUMENT_ID_HIDDEN_FIELD);
            if (previousDocumentID != null)
            {
                try
                {
                    this.setValue(DIFIoCRegistry.getRegistry().getImplementation(IDocumentRepositoryManager.class)
                            .getDocument(Long.parseLong(previousDocumentID.toString())), stageInstance);
                    if (hasValue())
                    {
                        for (ParameterError error: errorList.getErrorList())
                        {
                            if (error.getErrorType() == ParameterErrorType.MISSING)
                            {
                                errorList.getErrorList().remove(error);
                                break;
                            }
                        }
                    }
                }
                catch (Exception e)
                {
                    // If any error occurs do nothing. If the parameter is mandatory the missing error will be
                    // maintained
                    e.printStackTrace();
                }
            }
        }

        if (stageInstance.getContext().getRequest().getAttribute(HTTPConstants.UPLOAD_FILE_SIZE_ERROR + this.getId()) != null)
        {
            ParameterError pError = new ParameterError(
                    getMessages().getMessages(stageInstance.getContext().getLanguage()).get("fileUploadMaxSize")
                            + " "
                            + stageInstance.getContext().getSession()
                                    .getAttribute(DIFSessionConstants.MAX_DOCUMENT_SIZE), ParameterErrorType.OTHER);
            pError.setForceShowError(true);

            errorList.addError(pError);

            // Error reported. Discard the ChAL warning...
            stageInstance.getContext().getRequest().addAttribute(HTTPConstants.UPLOAD_FILE_SIZE_ERROR, null);
        }

        return errorList;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter#setValue(java.lang.Object,
     *      pt.digitalis.dif.dem.interfaces.IStageInstance, boolean)
     */
    @Override
    public ParameterErrorList setValue(DocumentRepositoryEntry value, IStageInstance stageInstance,
            boolean initializationInProgress)
    {
        // Resize functionality
        if (value != null && value.getBytes() != null && value.getBytes().length > 0
                && stageInstance.getContext().getRequest().getParameter(this.getId() + "_resize") != null)
        {
            // Resize issued
            String resizeParams[] = ((String) stageInstance.getContext().getRequest()
                    .getParameter(this.getId() + "_resize")).split("x");
            Integer width = Integer.parseInt(resizeParams[0]);
            Integer height = resizeParams.length > 1 ? Integer.parseInt(resizeParams[1]) : width;

            ByteArrayInputStream bais = new ByteArrayInputStream(value.getBytes());
            BufferedImage image;
            try
            {
                image = ImageIO.read(bais);
                bais.close();

                if (image != null && (image.getHeight() > height || image.getWidth() > width))
                {
                    BufferedImage resizedImage;
                    // Resize the image to the conventioned size
                    resizedImage = Scalr.resize(image, Scalr.Method.AUTOMATIC, Scalr.Mode.AUTOMATIC, width, height);

                    // Convert it back to byte[] in eventImage for saving
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    ImageIO.write(resizedImage, value.getMimeType(), baos);
                    baos.flush();
                    value.setBytes(baos.toByteArray());
                    baos.close();
                }
            }
            catch (IOException e)
            {
                DIFLogger.getLogger().warn("Could not resize the image: " + value.getFileName());
                e.printStackTrace();
            }
        }

        // Delete functionality
        if (stageInstance.getContext().getRequest().getParameter(this.getId() + "_delete") != null)
        {
            Object docID = stageInstance.getContext().getRequest()
                    .getParameter(this.getId() + DOCUMENT_ID_HIDDEN_FIELD);

            if (docID != null)
            {
                // TODO: Implement the deletion for images
                // TODO: Implement the deletion for files
                value = new DocumentRepositoryEntry();
                value.setId(Long.parseLong(docID.toString()));

                try
                {
                    value = DIFIoCRegistry.getRegistry().getImplementation(IDocumentRepositoryManager.class)
                            .getDocument(Long.parseLong(docID.toString()));
                }
                catch (NumberFormatException e)
                {
                    e.printStackTrace();
                }
                catch (DocumentRepositoryException e)
                {
                    e.printStackTrace();
                }

                if (value != null)
                    value.setDeleteRequested(true);
            }
        }

        return super.setValue(value, stageInstance, initializationInProgress);
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.IParameter#setValueFromString(java.lang.String,
     *      pt.digitalis.dif.dem.interfaces.IStageInstance, boolean)
     */
    public ParameterErrorList setValueFromString(String value, IStageInstance stageInstance,
            boolean initializationInProgress)
    {
        ParameterErrorList list = new ParameterErrorList(this, value);
        ParameterError error = new ParameterError(getMessages().getMessages(stageInstance.getContext().getLanguage())
                .get("invalidUsage"), ParameterErrorType.OTHER);
        list.addError(error);

        return list;
    }
}
