/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.startup;

import java.io.InputStream;
import java.util.Properties;

import pt.digitalis.dif.dem.objects.messages.MessagesLocation;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.ioc.IoCImplementations;
import pt.digitalis.utils.ioc.ModuleParser;

/**
 * This class represents the basic startup configuration DIF needs. In this phase it can't rely on several
 * features/resources since they have not yet been started Note: There is no support for changes in this configuration
 * in runtime. Only by updating the configuration file and then restarting the DIF will this be reflected. This is a
 * design decision since these parameters define several resources that are mandatory for the startup of the DIF itself
 * and set the ground for all its infrastructure, like, for instance the IoC controller.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Oct 2, 2007
 */
public class DIFStartupConfiguration {

    /**
     * The file name of the properties file that contains the startup parameters. Note: It's called dif2 for the
     * eventuality of the existence of dif 1.x instances in the same App. Server. this way they will not collide.
     */
    static final public String CONFIGURATION_FILE = "dif2.properties";

    /** The DEMO Mode property */
    static private Boolean demoMode;

    /** The developer Mode property */
    static private Boolean developerMode;

    /** The IoC engine to use: Property "iocimplementation" */
    static private IoCImplementations iocImplementation;

    /** The Log Level: Property "loglevel" */
    static private LogLevel logLevel;

    /** The Messages Location: Property "messagelocation" */
    static private MessagesLocation messagesLocation;

    /** The Module Parsing method: Property "moduleparser" */
    static private ModuleParser moduleParser;

    /** The "print stack trace" property. */
    static private Boolean printStackTrace = true;

    /** The testing Mode property */
    static private Boolean testingMode;

    /** Static initializer */
    static
    {
        Properties parameters = new Properties();
        try
        {
            /*
             * IMPLEMENTATION NOTE: Must use the Thread context class loader since inside Surefire the system
             * classLoader does not have all classpath entries
             */

            // Load the configuration file from the classPath
            InputStream configsFile = Thread.currentThread().getContextClassLoader()
                    .getResourceAsStream(CONFIGURATION_FILE);

            if (configsFile != null)
            {

                parameters.load(configsFile);

                iocImplementation = IoCImplementations.getImplementationByName(loadProperty(parameters,
                        "iocimplementation", IoCImplementations.GUICE.toString()));

                logLevel = LogLevel.getLevelByName(loadProperty(parameters, "loglevel", LogLevel.WARN.toString()));
                developerMode = loadProperty(parameters, "developerMode", false);
                demoMode = loadProperty(parameters, "demoMode", false);
                testingMode = loadProperty(parameters, "testingMode", false);

                moduleParser = ModuleParser.getParsingMethodByName(loadProperty(parameters, "moduleparser",
                        ModuleParser.FAST.toString()));
                messagesLocation = MessagesLocation.getMessageLocationByName(loadProperty(parameters,
                        "messagelocation", MessagesLocation.MESSAGES_FOLDER.toString()));
                printStackTrace = loadProperty(parameters, "printStackTrace", true);
            }
            else
                setDefaultConfigurations();

        }
        catch (Exception e)
        {
            setDefaultConfigurations();
            DIFLogger.getLogger().warn("dif2.properties unavailable! Reverting to defaults...");
        }
    }

    /**
     * Inspector for the 'demoMode' attribute.
     * 
     * @return the demoMode value
     */
    public static Boolean getDemoMode()
    {
        return demoMode;
    }

    /**
     * Inspector for the 'developerMode' attribute.
     * 
     * @return the developerMode value
     */
    public static Boolean getDeveloperMode()
    {
        return developerMode;
    }

    /**
     * @return the iocImplementation
     */
    static public IoCImplementations getIoCImplementation()
    {
        return iocImplementation;
    }

    /**
     * @return the logLevel
     */
    static public LogLevel getLogLevel()
    {
        return logLevel;
    }

    /**
     * @return the messagesLocation
     */
    static public MessagesLocation getMessagesLocation()
    {
        return messagesLocation;
    }

    /**
     * @return the moduleParser
     */
    static public ModuleParser getModuleParser()
    {
        return moduleParser;
    }

    /**
     * Inspector for the 'testingMode' attribute.
     * 
     * @return the testingMode value
     */
    public static Boolean getTestingMode()
    {
        return testingMode;
    }

    /**
     * Inspector for the 'printStackTrace' attribute.
     * 
     * @return the printStackTrace attribute value
     */
    static public Boolean isPrintStackTrace()
    {
        return printStackTrace;
    }

    /**
     * @return T is the instance is in production mode. Production mode is the default mode when none of the other modes
     *         are active (development, testing or demo)
     */
    public static boolean isProductionMode()
    {
        return !getDemoMode() && !getDeveloperMode() && !getTestingMode();
    }

    /**
     * Overloads the 'loadProperty' method for boolean values.
     * 
     * @param prop
     *            the Properties to search
     * @param key
     *            the key to search
     * @param defaultValue
     *            the default value when not found
     * @return the read value or the default if not found
     */
    static Boolean loadProperty(Properties prop, String key, boolean defaultValue)
    {
        String value = prop.getProperty(key);

        if (value == null)
            return defaultValue;
        else
        {
            if (value.toLowerCase().equals("false") || value.toLowerCase().equals("f"))
                return false;
            else if (value.toLowerCase().equals("true") || value.toLowerCase().equals("t"))
                return true;
            else
                throw new RuntimeException("Invalid configuration value for property '" + key
                        + "'! Unable to proceed...\nValid values are: 'true', 'T', 'false' and 'F'.");
        }
    }

    /**
     * Searches properties for a given key
     * 
     * @param prop
     *            the Properties to search
     * @param key
     *            the key to search
     * @param defaultValue
     *            the default value when not found
     * @return the read value or the default if not found
     */
    static String loadProperty(Properties prop, String key, String defaultValue)
    {
        String value = prop.getProperty(key);

        if (value == null)
            return defaultValue;
        else
            return value;
    }

    /** */
    static private void setDefaultConfigurations()
    {
        iocImplementation = IoCImplementations.GUICE;
        logLevel = LogLevel.WARN;
        developerMode = false;
        demoMode = false;
        testingMode = false;
        moduleParser = ModuleParser.FAST;
        messagesLocation = MessagesLocation.MESSAGES_FOLDER;
        printStackTrace = true;
    }

    /**
     * Modifier for the 'demoMode' attribute.
     * 
     * @param demoMode
     *            the new demoMode value to set
     */
    public static void setDemoMode(Boolean demoMode)
    {
        DIFStartupConfiguration.demoMode = demoMode;
    }

    /**
     * Modifier for the 'developerMode' attribute.
     * 
     * @param developerMode
     *            the new developerMode value to set
     */
    public static void setDeveloperMode(Boolean developerMode)
    {
        DIFStartupConfiguration.developerMode = developerMode;
    }

    /**
     * Modifier for the 'logLevel' attribute.
     * 
     * @param logLevel
     *            the new logLevel value to set
     */
    public static void setLogLevel(LogLevel logLevel)
    {
        DIFStartupConfiguration.logLevel = logLevel;
    }

    /**
     * Modifier for the 'testingMode' attribute.
     * 
     * @param testingMode
     *            the new testingMode value to set
     */
    public static void setTestingMode(Boolean testingMode)
    {
        DIFStartupConfiguration.testingMode = testingMode;
    }
}
