/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.extensions.cms;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentItemNotFoundException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentItemWithDuplicateNameAndParentNodeException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentManagerException;
import pt.digitalis.dif.utils.extensions.cms.exception.InvalidParentNodeException;
import pt.digitalis.dif.utils.extensions.cms.exception.InvalidPathException;
import pt.digitalis.dif.utils.extensions.cms.exception.NoAccessException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeNotFoundException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeWithDuplicatePathException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeWithNodesException;

/**
 * Base implementation of a {@link IContentManager}
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/10/25
 */
public abstract class AbstractContentManager implements IContentManager {

    /** */
    protected static char SEPARATOR = '\\';

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#addContent(pt.digitalis.dif.utils.extensions.cms.ContentItem)
     */
    public ContentItem addContent(ContentItem content) throws ContentManagerException
    {

        if (content.getUser() == null)
        {
            throw new NoAccessException("The Content User attribute must be filled to complete the insert");
        }

        if (content.getParentNodeId() != null)
        {
            try
            {
                if (!nodeExists(content.getParentNodeId(), content.getUser()))
                    throw new InvalidParentNodeException();
            }
            catch (Exception e)
            {
                throw new InvalidParentNodeException(e);
            }

            if (!hasNodeAccessUser(content.getParentNodeId(), content.getUser()))
                throw new NoAccessException("User doesn't have access to the parent node");
        }

        return persistContentInRepository(content, content.getUser());
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#addNode(pt.digitalis.dif.utils.extensions.cms.Node)
     */
    public Node addNode(Node node) throws ContentManagerException
    {
        if (node == null)
            throw new NodeNotFoundException();

        if (node.getUser() == null)
        {
            throw new NoAccessException("The Node User attribute must be filled to complete the insert");
        }

        String fullPath = determineFullPath(node);

        if (!validadeNameAndPath(fullPath, true))
            throw new InvalidPathException("Invalid path: " + fullPath);

        node.setFullPathName(fullPath);

        if (node.getParentNodeId() != null)
            if (!hasNodeAccessUser(node.getParentNodeId(), node.getUser()))
                throw new NoAccessException("User doesn't have access to the parent node");

        return persistNodeInRepository(node, node.getUser());
    }

    /**
     * Check the access in a list of content items
     * 
     * @param listToCheck
     *            the content list to check
     * @param user
     *            the user who's searching
     * @return the list with the contents with access
     */
    protected List<ContentItem> checkContentAccessInList(List<ContentItem> listToCheck, IDIFUser user)
    {
        List<ContentItem> reviewedList = new ArrayList<ContentItem>();

        for (ContentItem content: listToCheck)
        {
            try
            {
                if (hasContentAccessUser(content, user))
                    reviewedList.add(content);
            }
            catch (ContentManagerException e)
            {
                // never should occur
            }
        }

        return reviewedList;
    }

    /**
     * Check the access in a list of nodes
     * 
     * @param listToCheck
     *            the node list to check
     * @param user
     *            the user who's searching
     * @return the list with the contents with access
     */
    protected List<Node> checkNodeAccessInList(List<Node> listToCheck, IDIFUser user)
    {
        List<Node> reviewedList = new ArrayList<Node>();

        for (Node node: listToCheck)
        {
            try
            {
                if (hasNodeAccessUser(node, user))
                    reviewedList.add(node);
            }
            catch (ContentManagerException e)
            {
                // never should occur
            }
        }

        return reviewedList;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#deleteContent(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean deleteContent(String id, IDIFUser user) throws ContentManagerException
    {
        if (!hasContentAccessUser(id, user))
            throw new NoAccessException("User doesn't have access to content");

        return deleteContentInRepository(id, user);
    }

    /**
     * Deletes an existing content from the repository
     * 
     * @param id
     *            the id of the content to delete
     * @param user
     *            the user id that is deleting the content
     * @return T if all went well
     * @throws ContentManagerException
     */
    protected abstract boolean deleteContentInRepository(String id, IDIFUser user) throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#deleteNode(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean deleteNode(Long id, IDIFUser user) throws ContentManagerException
    {
        return deleteNode(id, user, false);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#deleteNode(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser, boolean)
     */
    public boolean deleteNode(Long id, IDIFUser user, boolean cascadeDelete) throws ContentManagerException
    {
        if (!hasNodeAccessUser(id, user))
            throw new NoAccessException("User doesn't have access to node");

        return deleteNodeInRepository(id, user, cascadeDelete);
    }

    /**
     * Deletes an existing node from the repository. <br>
     * Will launch an exception if the node is not empty
     * 
     * @param nodeId
     *            the node to delete
     * @param user
     *            the user that's deleting the node
     * @param cascadeDelete
     *            if T will delete all inner nodes and content, if F will launch an exception if the node is not empty
     * @return T if all went well
     * @throws NodeNotFoundException
     * @throws NodeWithNodesException
     * @throws ContentManagerException
     */
    protected abstract boolean deleteNodeInRepository(Long nodeId, IDIFUser user, boolean cascadeDelete)
            throws NodeNotFoundException, NodeWithNodesException, ContentManagerException;

    /**
     * Determines the full path String base in the Node chain
     * 
     * @param node
     *            node to convert
     * @return the node converted
     * @throws ContentManagerException
     */
    protected String determineFullPath(Node node) throws ContentManagerException
    {
        if (node.getParentNodeId() == null)
            return SEPARATOR + node.getName();
        else
        {
            String fullPathFromRep = getParentFullPathInRepository(node);

            if (!validadeNameAndPath(fullPathFromRep, true))
                throw new InvalidPathException("Invalid path: " + fullPathFromRep);

            return fullPathFromRep + SEPARATOR + node.getName();
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getContentByDescription(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public List<ContentItem> getContentByDescription(String description, IDIFUser user) throws ContentManagerException
    {
        return checkContentAccessInList(getContentByDescriptionInRepository(description, user), user);
    }

    /**
     * Searches content items by description
     * 
     * @param description
     *            the content id
     * @param user
     *            the user who's searching
     * @return the content list
     * @throws ContentManagerException
     */
    protected abstract List<ContentItem> getContentByDescriptionInRepository(String description, IDIFUser user)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getContentById(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public ContentItem getContentById(String id, IDIFUser user) throws ContentManagerException
    {
        ContentItem contentItem = getContentByIdInRepository(id, user);
        if (!hasContentAccessUser(contentItem, user))
            throw new NoAccessException("User has no access to content item");

        return contentItem;
    }

    /**
     * Searches an existing content by it's unique identifier
     * 
     * @param id
     *            the content id
     * @param user
     *            the user who's searching
     * @return the content
     * @throws ContentItemNotFoundException
     * @throws NoAccessException
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    protected abstract ContentItem getContentByIdInRepository(String id, IDIFUser user)
            throws ContentItemNotFoundException, NoAccessException, NodeNotFoundException, ContentManagerException;

    /**
     * Searches an existing content by it's unique identifier, without access validation
     * 
     * @param id
     *            the content id
     * @param user
     *            current user
     * @return the content
     * @throws ContentManagerException
     */
    protected ContentItem getContentByIdNoPermissions(String id, IDIFUser user) throws ContentManagerException
    {
        return getContentByIdNoPermissionsInRepository(id, user);
    }

    /**
     * Searches an existing content by it's unique identifier, without access validation
     * 
     * @param id
     *            the content id
     * @param user
     *            current user
     * @return the content
     * @throws ContentItemNotFoundException
     * @throws ContentManagerException
     */
    protected abstract ContentItem getContentByIdNoPermissionsInRepository(String id, IDIFUser user)
            throws ContentItemNotFoundException, ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getContentByName(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public List<ContentItem> getContentByName(String name, IDIFUser user) throws ContentManagerException
    {
        return checkContentAccessInList(getContentByNameInRepository(name, user), user);
    }

    /**
     * Searches content items by name
     * 
     * @param name
     *            the content id
     * @param user
     *            the user who's searching
     * @return the content list
     * @throws ContentManagerException
     */
    protected abstract List<ContentItem> getContentByNameInRepository(String name, IDIFUser user)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getContentByParentNode(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public List<ContentItem> getContentByParentNode(Long nodeId, IDIFUser user) throws ContentManagerException
    {
        // just checks if exists
        getNodeByIdNoPermissions(nodeId, user);

        return checkContentAccessInList(getContentByParentNodeInRepository(nodeId, user), user);
    }

    /**
     * Searches content items by parent node
     * 
     * @param nodeId
     *            the parent id
     * @param user
     *            the user who's searching
     * @return the content list
     * @throws ContentManagerException
     */
    protected abstract List<ContentItem> getContentByParentNodeInRepository(Long nodeId, IDIFUser user)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getContentFromNodePathByName(java.lang.String,
     *      java.lang.String, pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public ContentItem getContentFromNodePathByName(String nodeFullPath, String name, IDIFUser user)
            throws ContentManagerException
    {
        ContentItem contentItem = getContentFromNodePathByNameRepository(nodeFullPath, name, user);
        if (!hasContentAccessUser(contentItem, user))
            throw new NoAccessException("User has no access to content item");

        return contentItem;
    }

    /**
     * Searches content items by name and node path on the repository
     * 
     * @param nodeFullPath
     *            the node fullPath from the content will be retrieve
     * @param name
     *            the content id
     * @param user
     *            the user who's searching
     * @return the content object
     * @throws ContentManagerException
     */
    protected abstract ContentItem getContentFromNodePathByNameRepository(String nodeFullPath, String name,
            IDIFUser user) throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getContentItemACL(java.lang.String)
     */
    public List<ACLEntry> getContentItemACL(String id) throws ContentManagerException
    {
        return getContentItemACLInRepository(id);
    }

    /**
     * get node ACL
     * 
     * @param id
     *            the content item ID
     * @return the ACL
     * @throws ContentManagerException
     */
    protected abstract List<ACLEntry> getContentItemACLInRepository(String id) throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getNodeACL(java.lang.Long)
     */
    public List<ACLEntry> getNodeACL(Long id) throws ContentManagerException
    {
        return getNodeACLInRepository(id);
    }

    /**
     * get node ACL
     * 
     * @param id
     *            the node ID
     * @return the ACL
     * @throws ContentManagerException
     */
    protected abstract List<ACLEntry> getNodeACLInRepository(Long id) throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getNodeById(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public Node getNodeById(Long id, IDIFUser user) throws ContentManagerException
    {
        Node node = getNodeByIdInRepository(id, user);
        if (!hasNodeAccessUser(node, user))
            throw new NoAccessException("User does not have access to the Node");

        return node;
    }

    /**
     * Searches a node by unique identifier
     * 
     * @param id
     *            the node ID
     * @param user
     *            the user who's searching
     * @return a list of all existing root nodes
     * @throws ContentManagerException
     */
    protected abstract Node getNodeByIdInRepository(Long id, IDIFUser user) throws ContentManagerException;

    /**
     * Searches a node by unique identifier. Permission checks are ignored
     * 
     * @param nodeID
     *            the node ID
     * @param user
     *            the currentUser
     * @return a list of all existing root nodes
     * @throws ContentManagerException
     */
    protected Node getNodeByIdNoPermissions(Long nodeID, IDIFUser user) throws ContentManagerException
    {
        Node node = getNodeByIdNoPermissionsInRepository(nodeID, user);

        if (node == null)
            throw new NodeNotFoundException(nodeID);

        return node;
    }

    /**
     * Searches for a given node. Permissions are not checked
     * 
     * @param id
     *            the node ID
     * @param user
     *            current user
     * @return a list of all existing root nodes
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    protected abstract Node getNodeByIdNoPermissionsInRepository(Long id, IDIFUser user) throws NodeNotFoundException,
            ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getNodeByPath(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public Node getNodeByPath(String fullPath, IDIFUser user) throws ContentManagerException
    {
        Node node = getNodeByPathInRespository(fullPath, user);
        if (user != null)
            if (!hasNodeAccessUser(node, user))
                throw new NoAccessException("User doesn't have access to the Node");

        return node;
    }

    /**
     * Searches a node by path
     * 
     * @param fullPath
     *            the full path of the node the node ID
     * @param user
     *            the user who's searching
     * @return a list of all existing root nodes
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentManagerException
     */
    protected abstract Node getNodeByPathInRespository(String fullPath, IDIFUser user) throws NodeNotFoundException,
            NoAccessException, ContentManagerException;

    /**
     * Searches a node by path. Permission checks are ignored
     * 
     * @param nodePath
     *            the node path
     * @param user
     *            the currentUser
     * @return a list of all existing root nodes
     * @throws ContentManagerException
     */
    protected Node getNodeByPathNoPermissions(String nodePath, IDIFUser user) throws ContentManagerException
    {
        return getNodeByPathInRespository(nodePath, user);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getNodesByDescription(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public List<Node> getNodesByDescription(String description, IDIFUser user) throws ContentManagerException
    {
        return checkNodeAccessInList(getNodesByDescriptionInRepository(description, user), user);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getNodesByDescription(java.lang.String,
     *      java.lang.String, pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public List<Node> getNodesByDescription(String basePathToSearch, String description, IDIFUser user)
            throws ContentManagerException
    {
        return checkNodeAccessInList(getNodesByDescriptionInRepository(basePathToSearch, description, user), user);
    }

    /**
     * Searches nodes by description
     * 
     * @param description
     *            the description of the node to search
     * @param user
     *            the user who's searching
     * @return a list of all existing nodes with the description
     * @throws ContentManagerException
     */
    protected abstract List<Node> getNodesByDescriptionInRepository(String description, IDIFUser user)
            throws ContentManagerException;

    /**
     * Searches nodes by description
     * 
     * @param basePathToSearch
     *            the path to search from
     * @param description
     *            the name of the node to search
     * @param user
     *            the user who's searching
     * @return a list of all existing nodes with the description
     * @throws ContentManagerException
     */
    protected abstract List<Node> getNodesByDescriptionInRepository(String basePathToSearch, String description,
            IDIFUser user) throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getNodesByName(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public List<Node> getNodesByName(String name, IDIFUser user) throws ContentManagerException
    {
        return checkNodeAccessInList(getNodesByNameInRepository(name, user), user);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getNodesByName(java.lang.String, java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public List<Node> getNodesByName(String basePathToSearch, String name, IDIFUser user)
            throws ContentManagerException
    {
        return checkNodeAccessInList(getNodesByNameInRepository(basePathToSearch, name, user), user);
    }

    /**
     * Searches nodes by name
     * 
     * @param name
     *            the name of the node to search
     * @param user
     *            the user who's searching
     * @return a list of nodes with the same name. May exist with same name in different parent nodes
     * @throws ContentManagerException
     */
    protected abstract List<Node> getNodesByNameInRepository(String name, IDIFUser user) throws ContentManagerException;

    /**
     * Searches nodes by name
     * 
     * @param basePathToSearch
     *            the path to search from
     * @param name
     *            the name of the node to search
     * @param user
     *            the user who's searching
     * @return a list of all existing root nodes. May exist with same name in different parent nodes
     * @throws ContentManagerException
     */
    protected abstract List<Node> getNodesByNameInRepository(String basePathToSearch, String name, IDIFUser user)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getNodesByParentNode(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public List<Node> getNodesByParentNode(Long nodeId, IDIFUser user) throws ContentManagerException
    {
        return checkNodeAccessInList(getNodesByParentNodeInRepository(nodeId, user), user);
    }

    /**
     * Searches nodes by parent node
     * 
     * @param nodeId
     *            the parent node id
     * @param user
     *            the user who's searching
     * @return the content list
     * @throws ContentManagerException
     */
    protected abstract List<Node> getNodesByParentNodeInRepository(Long nodeId, IDIFUser user)
            throws ContentManagerException;

    /**
     * Searches nodes by parent node. Permission checks are ignored.
     * 
     * @param nodeId
     *            the parent node id
     * @param user
     *            current user
     * @return the content list
     * @throws ContentManagerException
     */
    protected List<Node> getNodesByParentNodeNoPermissions(Long nodeId, IDIFUser user) throws ContentManagerException
    {
        getNodeByIdNoPermissions(nodeId, user);

        return getNodesByParentNodeNoPermissionsInRepository(nodeId, user);
    }

    /**
     * Searches nodes by parent node. Permission checks are ignored.
     * 
     * @param nodeId
     *            the parent node id
     * @param user
     *            current user
     * @return the content list
     * @throws ContentManagerException
     */
    protected abstract List<Node> getNodesByParentNodeNoPermissionsInRepository(Long nodeId, IDIFUser user)
            throws ContentManagerException;

    /**
     * Gets the parent full path
     * 
     * @param node
     *            node to determine full path
     * @return the node converted
     * @throws InvalidParentNodeException
     * @throws ContentManagerException
     */
    protected abstract String getParentFullPathInRepository(Node node) throws InvalidParentNodeException,
            ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#getRootNodes(pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public List<Node> getRootNodes(IDIFUser user) throws ContentManagerException
    {
        return checkNodeAccessInList(getRootNodesInRepository(user), user);
    }

    /**
     * Searches root nodes
     * 
     * @param user
     *            the user who's searching
     * @return a list of all existing root nodes
     * @throws ContentManagerException
     */
    protected abstract List<Node> getRootNodesInRepository(IDIFUser user) throws ContentManagerException;

    /**
     * Grants group access to content item
     * 
     * @param contentId
     *            the id of the content to grant access
     * @param groupId
     *            the group to grant access
     * @return T if access was granted, F otherwise.
     * @throws ContentManagerException
     */
    public boolean grantContentAccessToGroup(String contentId, String groupId) throws ContentManagerException
    {
        ContentItem contentItem = this.getContentByIdNoPermissions(contentId, null);

        if (contentItem == null)
            throw new ContentItemNotFoundException(contentId);

        return grantContentAccessToGroupInRepository(contentItem, groupId);
    }

    /**
     * Grants group access to content item
     * 
     * @param contentItemParam
     *            the content to grant access
     * @param groupId
     *            the group to grant access
     * @return T if access was granted, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean grantContentAccessToGroupInRepository(ContentItem contentItemParam, String groupId)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#grantContentAccessToUser(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean grantContentAccessToUser(String contentId, IDIFUser user) throws ContentManagerException
    {
        ContentItem contentItem = getContentByIdNoPermissions(contentId, user);

        if (contentItem == null)
            throw new ContentItemNotFoundException(contentId);

        return grantContentAccessToUserInRepository(contentItem, user);
    }

    /**
     * Grants user access to content item
     * 
     * @param contentItemParam
     *            the content to grant access
     * @param user
     *            the group to grant access
     * @return T if access was granted, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean grantContentAccessToUserInRepository(ContentItem contentItemParam, IDIFUser user)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#grantNodeAccessToGroup(java.lang.Long,
     *      java.lang.String)
     */
    public boolean grantNodeAccessToGroup(Long nodeId, String groupId) throws ContentManagerException
    {
        Node node = getNodeByIdNoPermissions(nodeId, null);

        return grantNodeAccessToGroupInRepository(node, groupId);
    }

    /**
     * Grants group access to node
     * 
     * @param nodeParam
     *            the node to grant access
     * @param groupId
     *            the group to grant access
     * @return T if access was granted, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean grantNodeAccessToGroupInRepository(Node nodeParam, String groupId)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#grantNodeAccessToUser(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean grantNodeAccessToUser(Long nodeId, IDIFUser user) throws ContentManagerException
    {
        Node node = this.getNodeByIdNoPermissions(nodeId, user);

        return grantNodeAccessToUserInRepository(node, user);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#grantNodeAccessToUser(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean grantNodeAccessToUser(String nodePath, IDIFUser user) throws ContentManagerException
    {
        Node node = getNodeByPathNoPermissions(nodePath, user);
        return grantNodeAccessToUser(node.getId(), user);
    }

    /**
     * Grants user access to node
     * 
     * @param nodeParam
     *            the node to grant access
     * @param user
     *            the group to grant access
     * @return T if access was granted, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean grantNodeAccessToUserInRepository(Node nodeParam, IDIFUser user)
            throws ContentManagerException;

    /**
     * Checks if group has access to the content
     * 
     * @param contentItem
     *            the content to check access
     * @param groupId
     *            the group to check access
     * @return T if group has access, F otherwise.
     * @throws ContentManagerException
     */
    public boolean hasContentAccessGroup(ContentItem contentItem, String groupId) throws ContentManagerException
    {
        Node node = getNodeByIdNoPermissions(contentItem.getParentNodeId(), null);

        if (node.isPublic())
            return true;

        Set<String> groups = new HashSet<String>();
        groups.add(groupId);

        if (hasContentAccessGroupsInRepository(contentItem.getId(), groups))
            return true;
        else
            return hasNodeAccessGroups(contentItem.getParentNodeId(), groups);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#hasContentAccessGroup(java.lang.String,
     *      java.lang.String)
     */
    public boolean hasContentAccessGroup(String contentId, String groupId) throws ContentManagerException
    {
        ContentItem contentItem = getContentByIdNoPermissions(contentId, null);

        return hasContentAccessGroup(contentItem, groupId);
    }

    /**
     * Checks if one of the group in the list has access to the content
     * 
     * @param contentItem
     *            the content to check access
     * @param groups
     *            the group to check access
     * @return T if group has access, F otherwise.
     * @throws ContentManagerException
     */
    public boolean hasContentAccessGroups(ContentItem contentItem, Set<String> groups) throws ContentManagerException
    {
        Node node = getNodeByIdNoPermissions(contentItem.getParentNodeId(), null);

        if (node.isPublic())
            return true;

        if (hasContentAccessGroupsInRepository(contentItem.getId(), groups))
            return true;
        else
            return hasNodeAccessGroups(contentItem.getParentNodeId(), groups);
    }

    /**
     * Checks if one of the groups has access to the content
     * 
     * @param contentId
     *            the id of the content to check access
     * @param groups
     *            the group list to check access
     * @return T if group has access, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean hasContentAccessGroupsInRepository(String contentId, Set<String> groups)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#hasContentAccessUser(pt.digitalis.dif.utils.extensions.cms.ContentItem,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean hasContentAccessUser(ContentItem contentItem, IDIFUser user) throws ContentManagerException
    {
        boolean result = false;
        if (hasNodeAccessUser(contentItem.getParentNodeId(), user))
        {
            result = true;
        }

        if (!result && hasContentAccessUserInRepository(contentItem.getId(), user))
            result = true;

        try
        {
            if (!result && hasContentAccessGroups(contentItem, user.getGroupIDs()))
                result = true;
        }
        catch (IdentityManagerException e)
        {
            throw new ContentItemNotFoundException(e);
        }

        return result;

    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#hasContentAccessUser(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean hasContentAccessUser(String contentId, IDIFUser user) throws ContentManagerException
    {
        ContentItem contentItem = getContentByIdNoPermissions(contentId, user);

        if (contentItem == null)
            throw new ContentItemNotFoundException(contentId);

        return hasContentAccessUser(contentItem, user);

    }

    /**
     * Checks if group has access to thentent
     * 
     * @param contentId
     *            the id of the content to check access
     * @param user
     *            the group to check access
     * @return T if user has access, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean hasContentAccessUserInRepository(String contentId, IDIFUser user)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#hasNodeAccessGroups(java.lang.Long, java.util.Set)
     */
    public boolean hasNodeAccessGroups(Long nodeId, Set<String> groups) throws ContentManagerException
    {
        Node node = getNodeByIdNoPermissions(nodeId, null);

        if (node.isPublic())
            return true;

        if (hasNodeAccessGroupsInRepository(nodeId, groups))
            return true;

        if (node.getParentNodeId() != null)
            return hasNodeAccessGroups(node.getParentNodeId(), groups);

        return false;
    }

    /**
     * Checks if one of the groups has access to the node
     * 
     * @param nodeId
     *            the id of the node to check access
     * @param groups
     *            the group list to check access
     * @return T if group has access, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean hasNodeAccessGroupsInRepository(Long nodeId, Set<String> groups)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#hasNodeAccessUser(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean hasNodeAccessUser(Long nodeId, IDIFUser user) throws ContentManagerException
    {
        Node node = getNodeByIdNoPermissions(nodeId, user);

        return hasNodeAccessUser(node, user);
    }

    /**
     * Checks if user has access to the node
     * 
     * @param node
     *            the node to check access
     * @param user
     *            the user to check access
     * @return T if user has access, F otherwise.
     * @throws ContentManagerException
     */
    public boolean hasNodeAccessUser(Node node, IDIFUser user) throws ContentManagerException
    {
        boolean result = false;
        if (node.isPublic())
            result = true;

        if (!result && hasNodeAccessUserInRespository(node.getId(), user))
            result = true;

        if (!result)
        {
            try
            {
                result = hasNodeAccessGroups(node.getId(), user.getGroupIDs());
            }
            catch (IdentityManagerException e)
            {
                new ContentManagerException(e);
            }
        }

        if (!result && node.getParentNodeId() != null)
        {
            Node parentNode = getNodeByIdNoPermissions(node.getParentNodeId(), null);
            result = hasNodeAccessUser(parentNode, user);
        }

        return result;
    }

    /**
     * Checks if user has access to the node
     * 
     * @param nodeId
     *            the id of the node to check access
     * @param user
     *            the group to check access
     * @return T if user has access, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean hasNodeAccessUserInRespository(Long nodeId, IDIFUser user)
            throws ContentManagerException;

    /**
     * Updates a content item
     * 
     * @param contentParam
     *            the content to update in the repository
     * @param user
     *            current user
     * @return the updated content
     * @throws ContentItemNotFoundException
     * @throws ContentItemWithDuplicateNameAndParentNodeException
     * @throws ContentManagerException
     */
    protected abstract ContentItem mergeContentInRepository(ContentItem contentParam, IDIFUser user)
            throws ContentItemNotFoundException, ContentItemWithDuplicateNameAndParentNodeException,
            ContentManagerException;

    /**
     * Updates a node
     * 
     * @param nodeParam
     *            the node to update in the repository
     * @param user
     *            current user
     * @return the updated content
     */
    protected abstract Node mergeNodeInRepository(Node nodeParam, IDIFUser user);

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#moveContent(java.lang.String, java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean moveContent(String contentID, Long destinationNodeId, IDIFUser user) throws ContentManagerException
    {
        ContentItem contentItem = getContentById(contentID, user);

        if (contentItem == null)
            throw new ContentItemNotFoundException(contentID);

        Node node = getNodeByIdNoPermissions(destinationNodeId, user);

        if (!hasContentAccessUser(contentItem, user))
        {
            throw new NoAccessException("User has no access to the content");
        }

        if (!hasNodeAccessUser(node, user))
        {
            throw new NoAccessException("User has no access to the destination node");
        }

        moveContentInRepository(contentItem, node, user);

        return true;
    }

    /**
     * Moves content to another node
     * 
     * @param contentParam
     *            the content to update in the repository
     * @param nodeParam
     *            the node to move to
     * @param user
     *            the current user
     * @return the updated content
     * @throws ContentItemNotFoundException
     * @throws ContentManagerException
     */
    protected abstract ContentItem moveContentInRepository(ContentItem contentParam, Node nodeParam, IDIFUser user)
            throws ContentItemNotFoundException, ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#moveNode(java.lang.Long, java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean moveNode(Long nodeID, Long destinationNodeId, IDIFUser user) throws ContentManagerException
    {
        Node node = getNodeByIdNoPermissions(nodeID, user);

        Node newParentNode = getNodeByIdNoPermissions(destinationNodeId, user);

        if (!hasNodeAccessUser(nodeID, user))
        {
            throw new NoAccessException("User has no access to the node");
        }

        if (!hasNodeAccessUser(destinationNodeId, user))
        {
            throw new NoAccessException("User has no access to the destination node");
        }

        node.setParentNodeId(newParentNode.getId());

        try
        {
            node.setFullPathName(determineFullPath(node));
        }
        catch (ContentManagerException e)
        {
            throw new NodeNotFoundException("Error determining full path");
        }

        moveNodeInRepository(node, user);

        return true;
    }

    /**
     * Moves node to another parent node
     * 
     * @param node
     *            the node to update in the repository
     * @param user
     *            the current user
     * @return the updated content
     */
    protected abstract Node moveNodeInRepository(Node node, IDIFUser user);

    /**
     * Searches a node by unique identifier.
     * 
     * @param id
     *            the node ID
     * @param user
     *            the current User
     * @return a list of all existing root nodes
     * @throws ContentManagerException
     */
    protected boolean nodeExists(Long id, IDIFUser user) throws ContentManagerException
    {
        return nodeExistsInRepository(id, user);
    }

    /**
     * Searches a node by unique identifier in the repository.
     * 
     * @param id
     *            the node ID
     * @param user
     *            the currentUser
     * @return a list of all existing root nodes
     * @throws ContentManagerException
     */
    protected abstract boolean nodeExistsInRepository(Long id, IDIFUser user) throws ContentManagerException;

    /**
     * Adds a new content to repository.<br>
     * Will assign the ID internally
     * 
     * @param content
     *            the content to update in the repository
     * @param user
     * @return the updated content
     * @throws ContentItemWithDuplicateNameAndParentNodeException
     * @throws ContentManagerException
     */
    protected abstract ContentItem persistContentInRepository(ContentItem content, IDIFUser user)
            throws ContentItemWithDuplicateNameAndParentNodeException, ContentManagerException;

    /**
     * Adds a new content to repository.<br>
     * Will assign the ID internally
     * 
     * @param node
     *            node to update in the repository
     * @param user
     *            the user persisting the node
     * @return the updated content
     * @throws NodeWithDuplicatePathException
     * @throws ContentManagerException
     */
    protected abstract Node persistNodeInRepository(Node node, IDIFUser user) throws NodeWithDuplicatePathException,
            ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#revokeContentAccessToGroup(java.lang.String,
     *      java.lang.String)
     */
    public boolean revokeContentAccessToGroup(String contentId, String groupId) throws ContentManagerException
    {
        if (getContentByIdNoPermissions(contentId, null) == null)
            throw new ContentItemNotFoundException(contentId);

        return revokeContentAccessToGroupInRepository(contentId, groupId);
    }

    /**
     * Revokes group access to content item
     * 
     * @param contentId
     *            the id of the content to revoke access
     * @param groupId
     *            the group to revoke access
     * @return T if access was revoked, F otherwise.
     */
    protected abstract boolean revokeContentAccessToGroupInRepository(String contentId, String groupId);

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#revokeContentAccessToUser(java.lang.String,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean revokeContentAccessToUser(String contentId, IDIFUser user) throws ContentManagerException
    {
        if (getContentByIdNoPermissions(contentId, user) == null)
            throw new ContentItemNotFoundException(contentId);

        return revokeContentAccessToUserInRepository(contentId, user);
    }

    /**
     * Revokes user access to content item
     * 
     * @param contentId
     *            the id of the content to revoke access
     * @param user
     *            the group to revoke access
     * @return T if access was revoked, F otherwise.
     */
    protected abstract boolean revokeContentAccessToUserInRepository(String contentId, IDIFUser user);

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#revokeNodeAccessToGroup(java.lang.Long,
     *      java.lang.String)
     */
    public boolean revokeNodeAccessToGroup(Long nodeId, String groupId) throws ContentManagerException
    {
        getNodeByIdNoPermissions(nodeId, null);

        return revokeNodeAccessToGroupInRepository(nodeId, groupId);
    }

    /**
     * Revokes group access to node
     * 
     * @param nodeId
     *            the id of the node to revoke access
     * @param groupId
     *            the group to revoke access
     * @return T if access was revoked, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean revokeNodeAccessToGroupInRepository(Long nodeId, String groupId)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#revokeNodeAccessToUser(java.lang.Long,
     *      pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public boolean revokeNodeAccessToUser(Long nodeId, IDIFUser user) throws ContentManagerException
    {
        getNodeByIdNoPermissions(nodeId, user);

        return revokeNodeAccessToUserInRepository(nodeId, user);
    }

    /**
     * Revokes user access to node
     * 
     * @param nodeId
     *            the id of the node to revoke access
     * @param user
     *            the group to revoke access
     * @return T if access was revoked, F otherwise.
     * @throws ContentManagerException
     */
    protected abstract boolean revokeNodeAccessToUserInRepository(Long nodeId, IDIFUser user)
            throws ContentManagerException;

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#updateContent(pt.digitalis.dif.utils.extensions.cms.ContentItem)
     */
    public ContentItem updateContent(ContentItem content) throws ContentManagerException
    {
        if (content == null)
            throw new ContentItemNotFoundException();

        if (content.getUser() == null)
        {
            throw new NoAccessException("The Content User attribute must be filled to complete the insert");
        }

        if (!hasContentAccessUser(content, content.getUser()))
            throw new NoAccessException("User doesn't have access to the parent node");

        ContentItem currentContent = getContentById(content.getId(), content.getUser());

        if (currentContent == null)
            throw new ContentItemNotFoundException();

        currentContent.setName(content.getName());
        currentContent.setDescription(content.getDescription());
        currentContent.setContent(content.getContent());

        return mergeContentInRepository(currentContent, content.getUser());
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.cms.IContentManager#updateNode(pt.digitalis.dif.utils.extensions.cms.Node)
     */
    public Node updateNode(Node node) throws ContentManagerException
    {
        if (node == null)
            throw new NodeNotFoundException();

        if (node.getUser() == null)
        {
            throw new NoAccessException("The Node User attribute must be filled to complete the insert");
        }

        if (!hasNodeAccessUser(node, node.getUser()))
            throw new NoAccessException("User doesn't have access to the node");

        Node currentNode = getNodeByIdNoPermissions(node.getId(), node.getUser());

        currentNode.setDescription(node.getDescription());
        currentNode.setPublic(node.isPublic());

        if (!currentNode.getName().equals(node.getName()))
        {

            try
            {
                node.setFullPathName(determineFullPath(node));
            }
            catch (ContentManagerException e)
            {
                throw new NodeNotFoundException("Error determining full path");
            }
            currentNode.setName(node.getName());
            currentNode.setFullPathName(node.getFullPathName());
        }

        return mergeNodeInRepository(currentNode, node.getUser());
    }

    /**
     * Validates name and path
     * 
     * @param name
     *            full path
     * @param isPath
     *            check if it is a path
     * @return the validation result
     */
    private boolean validadeNameAndPath(String name, boolean isPath)
    {
        char ch;
        for (int i = 0; i < name.length(); i++)
        {
            ch = name.charAt(i);

            if (ch >= 'A' && ch <= 'Z')
                continue;
            if (ch >= 'a' && ch <= 'z')
                continue;
            if (ch >= '0' && ch <= '9')
                continue;
            if (ch == '-' || ch == '_')
                continue;
            if (isPath && ch == SEPARATOR)
                continue;

            return false;
        }

        return true;
    }

}
