/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.security.managers;

import java.util.List;
import java.util.Map;
import java.util.Set;

import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.utils.Pagination;

/**
 * Defines the expected behavior for an identity manager. This manager supports users {@link IDIFUser} and groups
 * {@link IDIFGroup}.
 * 
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Luis Pinto <a href="lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @author Fbio Souto <a href="mailto:fsouto@digitalis.pt">fsouto@digitalis.pt</a><br/>
 * @created 2007/12/04
 */
public interface IIdentityManager {

    // --------- User operations ---------

    /**
     * Adds a new group to the manager.
     * 
     * @param newGroup
     *            the group to add
     * @throws IdentityManagerException
     *             if the user can't be added
     */
    public void addGroup(IDIFGroup newGroup) throws IdentityManagerException;

    /**
     * Adds a managed attribute.
     * 
     * @param attribute
     *            The attribute
     */
    public void addManagedAttribute(String attribute);

    /**
     * Adds a comma-separated list of managed attributes.
     * 
     * @param attributes
     *            The comma-separated list of managed attributes.
     */
    public void addManagedAttributes(String attributes);

    /**
     * Adds a new user to the manager.
     * 
     * @param newUser
     *            the new user to add to the manager
     * @throws IdentityManagerException
     *             if the user can't be added
     */
    public void addUser(IDIFUser newUser) throws IdentityManagerException;

    /**
     * Adds an existing user to an existing group.
     * 
     * @param groupID
     *            the group ID
     * @param userID
     *            the user ID
     * @throws IdentityManagerException
     *             if the user can't be added to the group
     */
    public void addUserToGroup(String userID, String groupID) throws IdentityManagerException;

    /**
     * Changes the password for the supplied user.
     * 
     * @param userID
     *            the user ID
     * @param newPassword
     *            the new password
     * @throws IdentityManagerException
     *             if the password can't be changed
     */
    public void changePassword(String userID, String newPassword) throws IdentityManagerException;

    /**
     * Checks if a parameter exists.
     * 
     * @param userID
     *            the user ID
     * @param parameterID
     *            the parameter ID
     * @return T if the parameter exists, F otherwise
     */
    public boolean containsUserParameter(String userID, String parameterID);

    /**
     * Counts the number of groups in the system
     * 
     * @return The number of groups in the system
     * @throws IdentityManagerException
     *             * If the operation cannot be executed
     */
    public int countAllGroups() throws IdentityManagerException;

    /**
     * Counts all groups of a given user.
     * 
     * @param userId
     *            The user identifier
     * @return The number of groups of this user.
     * @throws IdentityManagerException
     */
    public int countAllGroupsOfUser(String userId) throws IdentityManagerException;

    /**
     * Counts all groups of a given user.
     * 
     * @param userId
     *            The user identifier
     * @param parseGroupHierarchy
     *            contemplates the group Hierarchy
     * @return The number of groups of this user.
     * @throws IdentityManagerException
     */
    public int countAllGroupsOfUser(String userId, boolean parseGroupHierarchy) throws IdentityManagerException;

    /**
     * Counts the number of users in the system
     * 
     * @return The number of users on the system
     * @throws IdentityManagerException
     *             If the operation cannot be executed
     */
    public int countAllUsers() throws IdentityManagerException;

    /**
     * Counts all users of a given group.
     * 
     * @param groupId
     *            The group identifier
     * @return The number of users of this group.
     * @throws IdentityManagerException
     */
    public int countAllUsers(String groupId) throws IdentityManagerException;

    /**
     * Count the number of users in the system which match all of the provided attributes
     * 
     * @param attributes
     *            The attributes
     * @return The number of users in the system which match all of the provided attributes
     * @throws IdentityManagerException
     *             If the operation cannot be executed
     */
    public int countUsers(Map<String, String> attributes) throws IdentityManagerException;

    /**
     * Gathers the managed attributes. This method must be used to initialize the managed attributes.
     * 
     * @return The list containing the managed attributes
     */
    public List<String> gatherManagedAttributes();

    /**
     * Get all the groups registered in the system
     * 
     * @return A set containing all the groups
     * @throws IdentityManagerException
     *             If the operation cannot be executed
     */
    public Set<IDIFGroup> getAllGroups() throws IdentityManagerException;

    /**
     * Get all the users in the system
     * 
     * @return A set containing all the users
     * @throws IdentityManagerException
     *             If the operation cannot be executed
     */
    public Set<IDIFUser> getAllUsers() throws IdentityManagerException;

    /**
     * Get the exclusion charaters
     * 
     * @return value
     */
    public String getExclusionCharaters();

    /**
     * Gets a group from its id, if it exists.
     * 
     * @param groupID
     *            the group id
     * @return the group with the given id if it exists, null otherwise
     * @throws IdentityManagerException
     *             if the group can't be found
     */
    public IDIFGroup getGroup(String groupID) throws IdentityManagerException;

    /**
     * Gets the group attribute name
     * 
     * @return The string containing the attribute name
     */
    public String getGroupAttributeName();

    /**
     * Returns the list of groups whose parent group is the given group
     * 
     * @param parentGroupID
     *            the parent group to search for children groups
     * @return the list of groups
     * @throws IdentityManagerException
     *             if the group's groups can't be found
     */
    public Map<String, IDIFGroup> getGroupGroups(String parentGroupID) throws IdentityManagerException;

    /**
     * Returns a subset of all groups in the system, according to the parameters
     * 
     * @param page
     *            The required page
     * @return A set of groups containing the specified subset
     * @throws IdentityManagerException
     *             If the operation can't be executed
     */
    public Set<IDIFGroup> getGroups(Pagination page) throws IdentityManagerException;

    /**
     * Returns the list of users of a given group
     * 
     * @param groupID
     *            the group
     * @return the list of users
     * @throws IdentityManagerException
     *             if the operation can't be executed
     */
    public Map<String, IDIFUser> getGroupUsers(String groupID) throws IdentityManagerException;

    /**
     * Gets the mail attribute name
     * 
     * @return The string containing the attribute name.
     */
    public String getMailAttributeName();

    /**
     * Gets the framework managed attributes.
     * 
     * @return the managedAttributes value
     */
    public List<String> getManagedAttributes();

    /**
     * Gets the name attribute name
     * 
     * @return The string containing the attribute name.
     */
    public String getNameAttributeName();

    /**
     * Gets an user from its id, if it exists.
     * 
     * @param userID
     *            the user id
     * @return the user that matches the supplied id if it exists, null otherwise
     * @throws IdentityManagerException
     *             if the user can't be fetched
     */
    public IDIFUser getUser(String userID) throws IdentityManagerException;

    /**
     * Returns the list of groups of a given user
     * 
     * @param userID
     *            the user
     * @return the list of groups
     * @throws IdentityManagerException
     *             if the group's users can't be found
     */
    public Map<String, IDIFGroup> getUserGroups(String userID) throws IdentityManagerException;

    /**
     * Returns the list of groups of a given user
     * 
     * @param userID
     *            the user
     * @param parseGroupHierarchy
     *            contemplates the group Hierarchy
     * @return the list of groups
     * @throws IdentityManagerException
     *             if the group's users can't be found
     */
    public Map<String, IDIFGroup> getUserGroups(String userID, boolean parseGroupHierarchy)
            throws IdentityManagerException;

    /**
     * Returns the list of group IDs of a given user
     * 
     * @param userID
     *            the user
     * @return the list of groups
     * @throws IdentityManagerException
     *             if the group's users can't be found
     */
    public Set<String> getUserGroupsIDs(String userID) throws IdentityManagerException;

    /**
     * Returns the list of group IDs of a given user
     * 
     * @param userID
     *            the user
     * @param parseGroupHierarchy
     *            contemplates the group Hierarchy
     * @return the list of groups
     * @throws IdentityManagerException
     *             if the group's users can't be found
     */
    public Set<String> getUserGroupsIDs(String userID, boolean parseGroupHierarchy) throws IdentityManagerException;

    /**
     * Returns the list of groups of a given user, with pagination. This method doesn't consider the profile as a group.
     * 
     * @param userID
     *            the user
     * @param page
     *            The page to be returned
     * @return the list of groups
     * @throws IdentityManagerException
     *             if the group's users can't be found
     */
    public Map<String, IDIFGroup> getUserGroupsPagination(String userID, Pagination page)
            throws IdentityManagerException;

    // --------- Group operations ---------

    /**
     * Returns the list of user IDs of a given group
     * 
     * @param groupID
     *            the group
     * @return the list of users
     * @throws IdentityManagerException
     *             if the operation can't be executed
     */
    public Set<String> getUserIDsInGroup(String groupID) throws IdentityManagerException;

    /**
     * Gets the user login attribute name
     * 
     * @return The string containing the attribute name.
     */
    public String getUserLoginAttributeName();

    /**
     * Gets a given parameter from a given user.
     * 
     * @param userID
     * @param parameterID
     * @return the parameter value if it exists, null otherwise
     */
    public Object getUserParameter(String userID, String parameterID);

    /**
     * Gets all the parameters from a given user.
     * 
     * @param userID
     *            the user ID
     * @return the parameters map if it exists, null otherwise
     */
    public Map<String, Object> getUserParameters(String userID);

    /**
     * Returns the user parent group attribute name
     * 
     * @return A string containing the user parent group attribute name
     */
    public String getUserParentGroupAttributeName();

    /**
     * Returns a subset of all users in the system, according to the parameters
     * 
     * @param page
     *            The required page
     * @return A set of users containing the specified subset
     * @throws IdentityManagerException
     *             If the operation can't be executed
     */
    public Set<IDIFUser> getUsers(Pagination page) throws IdentityManagerException;

    // --------- Relation operations ---------

    /**
     * Get users that match at least one of the specified attributes.<br/>
     * Supports partial matches by adding "%" or "*" before and/or after the attribute value
     * 
     * @param attributes
     *            A map of attributes to be searched, where the keys are the attribute names, and the values are the
     *            attribute values.
     * @return A set containing the users that obey all of the specified criteria
     * @throws IdentityManagerException
     *             If the operation can't be executed
     */
    public Set<IDIFUser> getUsersByAnyAttribute(Map<String, String> attributes) throws IdentityManagerException;

    /**
     * Returns the list of users with a given attribute value.<br/>
     * Supports partial matches by adding "%" or "*" before and/or after the attribute value
     * 
     * @param attribute
     *            the attribute to search
     * @param value
     *            the value for the attribute
     * @return the LDAPUser with the given login name
     * @throws IdentityManagerException
     *             if the operation cannot be executed
     */
    public Set<IDIFUser> getUsersByAttribute(String attribute, String value) throws IdentityManagerException;

    /**
     * Get users that match all specified attributes.<br/>
     * Supports partial matches by adding "%" or "*" before and/or after the attribute value
     * 
     * @param attributes
     *            A map of attributes to be searched, where the keys are the attribute names, and the values are the
     *            attribute values.
     * @return A set containing the users that obey all of the specified criteria
     * @throws IdentityManagerException
     *             If the operation can't be executed
     */
    public Set<IDIFUser> getUsersByAttributes(Map<String, String> attributes) throws IdentityManagerException;

    /**
     * Get users that match specified attributes.<br/>
     * Supports partial matches by adding "%" or "*" before and/or after the attribute value
     * 
     * @param attributes
     *            A map of attributes to be searched, where the keys are the attribute names, and the values are the
     *            attribute values.
     * @param page
     *            The page to be obtained
     * @return A set containing the users that obey all of the specified criteria
     * @throws IdentityManagerException
     *             If the operation can't be executed
     */
    public Set<IDIFUser> getUsersByAttributes(Map<String, String> attributes, Pagination page)
            throws IdentityManagerException;

    /**
     * Returns the list of users with a given email attribute value.
     * 
     * @param value
     *            the value for the attribute email
     * @return the LDAPUser with the given login name
     * @throws IdentityManagerException
     *             if the operation cannot be executed
     */
    public Set<IDIFUser> getUsersByEmail(String value) throws IdentityManagerException;

    /**
     * Checks if a group exists on the manager.
     * 
     * @param groupID
     *            the ID of the group
     * @return T if the group exists, F otherwise
     * @throws IdentityManagerException
     *             if the operation can't be executed
     */
    public boolean groupExists(String groupID) throws IdentityManagerException;

    /**
     * Checks if the supplied password matches the one defined for a user with a given ID.
     * 
     * @param userID
     *            the user ID
     * @param suppliedPassword
     *            the supplied user password
     * @return T if user if its identity is valid, F otherwise
     * @throws IdentityManagerException
     *             if the group can't be added as member of the parent group
     */
    public boolean isIdentityValid(String userID, String suppliedPassword) throws IdentityManagerException;

    /**
     * Get the ldap readOnly configuration value.
     * 
     * @return true or false
     */
    public boolean isReadOnly();

    /**
     * Checks if a given user belongs to a given group.
     * 
     * @param userID
     *            the user ID
     * @param groupID
     *            the group ID
     * @return T if the user belongs to the group, F otherwise
     * @throws IdentityManagerException
     *             if the operation can't be executed
     */
    public boolean isUserInGroup(String userID, String groupID) throws IdentityManagerException;

    /**
     * Checks if a given user belongs to a given group.
     * 
     * @param userID
     *            the user ID
     * @param groupID
     *            the group ID
     * @param parseGroupHierarchy
     *            contemplates the group Hierarchy
     * @return T if the user belongs to the group, F otherwise
     * @throws IdentityManagerException
     *             if the operation can't be executed
     */
    public boolean isUserInGroup(String userID, String groupID, boolean parseGroupHierarchy)
            throws IdentityManagerException;

    // --------- Identity manager related operations ---------

    /**
     * Removes an existing group from the manager, if it exists.
     * 
     * @param groupID
     *            the id of the group to remove
     * @throws IdentityManagerException
     *             if the group can't be removed
     */
    public void removeGroup(String groupID) throws IdentityManagerException;

    /**
     * Removes an existing user from the manager, if it exists.
     * 
     * @param userID
     *            the id of the user to remove
     * @throws IdentityManagerException
     *             if the user can't be removed
     */
    public void removeUser(String userID) throws IdentityManagerException;

    /**
     * Removes an existing user from an existing group.
     * 
     * @param groupID
     *            the group ID
     * @param userID
     *            the user ID
     * @throws IdentityManagerException
     *             if the user can't be removed from the group
     */
    public void removeUserFromGroup(String userID, String groupID) throws IdentityManagerException;

    /**
     * Removes a given parameter from a given user.
     * 
     * @param userID
     *            the user ID
     * @param parameterID
     *            the parameter ID
     */
    public void removeUserParameter(String userID, String parameterID);

    /** Resets the identity manager. Different implementations will offer distinct types of reset. */
    public void resetIdentityManager();

    /**
     * Sets an user parameter.
     * 
     * @param userID
     *            the user ID
     * @param parameterID
     *            the parameter ID
     * @param parameterValue
     *            the parameter value
     */
    public void setUserParameter(String userID, String parameterID, Object parameterValue);

    /**
     * Sets the user parameters as an whole.
     * 
     * @param userID
     *            the user ID
     * @param parametersMap
     *            the parameters map
     */
    public void setUserParameters(String userID, Map<String, Object> parametersMap);

    /**
     * Updates an existing group. Does nothing if the group doesn't exist.
     * 
     * @param existingGroup
     *            the group to update
     * @throws IdentityManagerException
     *             if the group can't be updated
     */
    public void updateGroup(IDIFGroup existingGroup) throws IdentityManagerException;

    /**
     * Updates an existing user. Does nothing if the user doesn't exist.
     * 
     * @param existingUser
     *            the user to update
     * @param userID
     *            the user ID (must be supplied separately to ensure data integrity if the user ID is changed)
     * @throws IdentityManagerException
     *             if the operation can't be executed
     */
    public void updateUser(IDIFUser existingUser, String userID) throws IdentityManagerException;

    /**
     * Updates an attribute value on a given user and persists it.
     * 
     * @param userID
     *            the ID of the user to update
     * @param attributeID
     *            the ID of the attribute to update
     * @param attributeValue
     *            the value of the attribute to update
     * @throws IdentityManagerException
     *             if the attribute can't be updated
     */
    public void updateUserAttribute(String userID, String attributeID, Object attributeValue)
            throws IdentityManagerException;

    /**
     * Updates all of the user's attributes.
     * 
     * @param userID
     *            the ID of the user to update
     * @param attributes
     *            the attributes to update
     * @throws IdentityManagerException
     *             if the attribute can't be updated
     */
    public void updateUserAttributes(String userID, Map<String, Object> attributes) throws IdentityManagerException;

    /**
     * Checks if a user exists on the manager.
     * 
     * @param userID
     *            the ID of the user
     * @return T if the user exists, F otherwise
     * @throws IdentityManagerException
     *             if the operation can't be executed
     */
    public boolean userExists(String userID) throws IdentityManagerException;

}
