/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.security.objects;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManagerPrivate;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.exception.security.AuthorizationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;

/**
 * Default implementation VALIDATE: Check the Cloneable usage good/bad practice!
 * 
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Dec 3, 2007
 */
public class DIFUserImpl implements IDIFClonableUser, Cloneable {

    /** The authorization manager implementation */
    static private IAuthorizationManager authorizationManager = DIFIoCRegistry.getRegistry().getImplementation(
            IAuthorizationManager.class);

    /** The identity manager implementation */
    static private IIdentityManager identityManager = DIFIoCRegistry.getRegistry().getImplementation(
            IIdentityManager.class);

    /** User attributes. */
    private CaseInsensitiveHashMap<Object> attributes = new CaseInsensitiveHashMap<Object>();

    /** User attributes to remove. */
    private List<String> attributesToRemove = new ArrayList<String>();

    /** The email of the user */
    private String email;

    /** If the user is enabled */
    private boolean enabled;

    /** Local cache for user groups */
    private Map<String, IDIFGroup> groupsCache = null;

    /** The unique id for the user */
    private String id;

    /** Is "default user"? */
    private boolean isDefault;

    /** The full name for the user */
    private String name;

    /** The nick for the user */
    private String nick;

    /** The password of the user */
    private String password;

    /** the user profile group identifier */
    private String profileID;

    /** Temporary groups to include into the final user groups. This groups must have the session life expectancy */
    private final Map<String, IDIFGroup> temporaryGroups = new HashMap<String, IDIFGroup>();

    /**
     * Default constructor
     */
    public DIFUserImpl()
    {
        this.cleanCache();
    }

    /**
     * Constructor from a IDIFUser object
     * 
     * @param user
     *            the IDIFUser to use as a base for the new user to create
     * @param password
     *            the password to set
     * @throws IdentityManagerException
     *             if the profile can't be accessed
     */
    public DIFUserImpl(IDIFUser user, String password) throws IdentityManagerException
    {
        setProps(user);

        this.password = password;
        this.cleanCache();
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#addTempGroup(java.lang.String)
     */
    public void addTempGroup(String groupId)
    {
        try
        {
            IDIFGroup group = identityManager.getGroup(groupId);
            if (group != null)
            {
                temporaryGroups.put(group.getID(), group);
            }
        }
        catch (IdentityManagerException e)
        {
            // Do nothing
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IUserAuthorization#canAccess(pt.digitalis.dif.dem.Entity,
     *      java.lang.String)
     */
    public boolean canAccess(Entity resourceType, String resourceId)
    {
        try
        {
            return authorizationManager.hasAccessUser(this, resourceType, resourceId)
                    || authorizationManager.hasAccessPublic(resourceType, resourceId);
        }
        catch (AuthorizationManagerException authorizationManagerException)
        {
            return false;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IUserAuthorization#canAccess(pt.digitalis.dif.dem.interfaces.IStage)
     */
    public boolean canAccess(IStage stage)
    {
        if (stage == null)
            return false;

        return this.canAccess(Entity.STAGE, stage.getID());
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#cleanCache()
     */
    public void cleanCache()
    {
        groupsCache = null;
    }

    /**
     * @see java.lang.Object#clone()
     */
    @Override
    protected Object clone() throws CloneNotSupportedException
    {
        try
        {
            DIFUserImpl clone = new DIFUserImpl(this, this.password);

            return clone;
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new RuntimeException(
                    "Could not clone object because the identity manager raised an excpetion an vital data could not be fetched!",
                    identityManagerException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFClonableUser#cloneUser()
     */
    public IDIFClonableUser cloneUser()
    {
        try
        {
            return (IDIFClonableUser) this.clone();

        }
        catch (CloneNotSupportedException e)
        {
            return null;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#containsAttribute(java.lang.String)
     */
    public boolean containsAttribute(String id)
    {
        return attributes.containsKey(id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#containsParameter(java.lang.String)
     */
    public boolean containsParameter(String id)
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class)
                .containsUserParameter(this.getID(), id);
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        DIFUserImpl other = (DIFUserImpl) obj;
        if (id == null)
        {
            if (other.id != null)
                return false;
        }
        else if (!id.equals(other.id))
            return false;
        return true;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getAttribute(java.lang.String)
     */
    public Object getAttribute(String id)
    {
        return attributes.get(id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getAttributes()
     */
    public CaseInsensitiveHashMap<Object> getAttributes()
    {
        return attributes;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getAttributesToRemove()
     */
    public List<String> getAttributesToRemove()
    {
        return attributesToRemove;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getEmail()
     */
    public String getEmail()
    {
        return email;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getGroupIDs()
     */
    public Set<String> getGroupIDs() throws IdentityManagerException
    {
        return getGroups().keySet();
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getGroups()
     */
    public Map<String, IDIFGroup> getGroups() throws IdentityManagerException
    {
        if (groupsCache == null)
        {
            groupsCache = identityManager.getUserGroups(id);
            groupsCache.putAll(this.temporaryGroups);
        }

        return groupsCache;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getID()
     */
    public String getID()
    {
        return id;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getName()
     */
    public String getName()
    {
        return name;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getNick()
     */
    public String getNick()
    {
        return nick;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getParameter(String)
     */
    public Object getParameter(String id)
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class)
                .getUserParameter(this.getID(), id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getParameters()
     */
    public Map<String, Object> getParameters()
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class).getUserParameters(this.getID());
    }

    /**
     * @return the user password
     */
    public String getPassword()
    {
        return password;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getProfile()
     */
    public IDIFGroup getProfile() throws IdentityManagerException
    {
        if (profileID == null)
            return null;

        return getGroups().get(profileID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#getProfileID()
     */
    public String getProfileID()
    {
        return this.profileID;
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode()
    {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((id == null) ? 0 : id.hashCode());
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#initializeAttributes(java.util.Map)
     */
    public void initializeAttributes(Map<String, Object> attrs)
    {
        attributes = new CaseInsensitiveHashMap<Object>();
        attributes.putAll(attrs);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#isDefault()
     */
    public boolean isDefault()
    {
        return this.isDefault;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#isEnabled()
     */
    public boolean isEnabled()
    {
        return enabled;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#refresh()
     */
    public void refresh()
    {
        IDIFUser updatedUser;
        try
        {
            updatedUser = identityManager.getUser(id);
            if (updatedUser != null)
            {
                setProps(updatedUser);
            }
        }
        catch (IdentityManagerException e)
        {
            // Nothing. Exception will be logged and refresh will only apply to cache vars...
        }

        cleanCache();
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#removeAttribute(java.lang.String)
     */
    public void removeAttribute(String id)
    {
        attributes.remove(id);
        attributesToRemove.add(id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#removeParameter(java.lang.String)
     */
    public void removeParameter(String id)
    {
        DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class).removeUserParameter(this.getID(), id);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#removeTempGroup(java.lang.String)
     */
    public void removeTempGroup(String groupId)
    {
        temporaryGroups.remove(groupId);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setAttribute(java.lang.String, java.lang.Object)
     */
    public void setAttribute(String id, Object attribute) throws InternalFrameworkException
    {

        // Store old value
        Object formerAttributeValue = this.attributes.get(id);

        try
        {
            // Set new attribute value on object
            this.attributes.put(id, attribute);
            // Set new attribute on Id Manager
            ((IIdentityManagerPrivate) DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class))
                    .persistUserAttribute(this.getID(), id, attribute);
        }
        catch (IdentityManagerException identityManagerException)
        {
            // Roll back attribute value
            this.attributes.put(id, formerAttributeValue);
            // Issue warning
            throw new InternalFrameworkException("Could not update the attribute value on the Identity Manager!",
                    identityManagerException, null);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setAttributes(java.util.Map)
     */
    public void setAttributes(Map<String, Object> attributes) throws InternalFrameworkException
    {
        // Store old values
        CaseInsensitiveHashMap<Object> formerAttributeValues = this.attributes;

        try
        {
            // Set new values on the object
            this.attributes = new CaseInsensitiveHashMap<Object>();

            if (attributes != null)
                this.attributes.putAll(attributes);

            // Set values on Id Manager
            ((IIdentityManagerPrivate) DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class))
                    .persistUserAttributes(this.getID(), attributes);
        }
        catch (IdentityManagerException identityManagerException)
        {
            // Roll back attribute values
            this.attributes = formerAttributeValues;
            // Issue warning
            throw new InternalFrameworkException("Could not update the attribute values on the Identity Manager!",
                    identityManagerException, null);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setDefault(boolean)
     */
    public void setDefault(boolean isDefault)
    {
        this.isDefault = isDefault;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setEmail(java.lang.String)
     */
    public void setEmail(String email)
    {
        this.email = email;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setEnabled(boolean)
     */
    public void setEnabled(boolean enabled)
    {
        this.enabled = enabled;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setID(java.lang.String)
     */
    public void setID(String id)
    {
        this.id = id;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setName(java.lang.String)
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setNick(java.lang.String)
     */
    public void setNick(String nick)
    {
        this.nick = nick;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setParameter(java.lang.String, java.lang.Object)
     */
    public void setParameter(String id, Object parameter)
    {
        DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class)
                .setUserParameter(this.getID(), id, parameter);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setParameters(java.util.Map)
     */
    public void setParameters(Map<String, Object> parameters)
    {
        DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class)
                .setUserParameters(this.getID(), parameters);
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setPassword(java.lang.String)
     */
    public void setPassword(String password)
    {
        this.password = password;
    }

    /**
     * @see pt.digitalis.dif.controller.security.objects.IDIFUser#setProfileID(java.lang.String)
     */
    public void setProfileID(String profileGroupID)
    {
        this.profileID = profileGroupID;

        cleanCache();
    }

    /**
     * Sets the internal user properties with the given user ones
     * 
     * @param user
     *            the user to get the updated properties from
     * @throws IdentityManagerException
     */
    private void setProps(IDIFUser user) throws IdentityManagerException
    {

        this.id = user.getID();
        this.nick = user.getNick();
        this.name = user.getName();
        this.email = user.getEmail();
        this.enabled = user.isEnabled();
        this.attributes = user.getAttributes();
        this.profileID = user.getProfileID();
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();

        formatter.addItem("ID", getID());
        formatter.addItemIfNotNull("Name", getName());
        formatter.addItemIfNotNull("Password", getPassword());
        formatter.addItemIfNotNull("Nick", getNick());
        formatter.addItemIfNotNull("Email", getEmail());
        formatter.addItem("Enabled", isEnabled());
        formatter.addItemIfNotNull("Attributes", attributes);
        formatter.addItemIfNotNull("Parameters", this.getParameters());

        try
        {
            IDIFGroup profile = getProfile();
            formatter.addItemIfNotNull("Profile", profile);
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new RuntimeException("Could not access the user's profile on the identity manager!",
                    identityManagerException);
        }

        try
        {
            Map<String, IDIFGroup> groups = getGroups();
            formatter.addItemIfNotNull("Groups", groups);
        }
        catch (IdentityManagerException identityManagerException)
        {
            throw new RuntimeException("Could not access the user's groups on the identity manager!",
                    identityManagerException);
        }

        return formatter.getFormatedObject();
    }
}