/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javassist.bytecode.annotation.MemberValue;

import org.apache.commons.lang.StringUtils;

import pt.digitalis.dif.codegen.CGAncillaries;
import pt.digitalis.dif.codegen.util.ClassEnhancementContext;
import pt.digitalis.dif.codegen.util.DEMLoaderEntityRegistry;
import pt.digitalis.dif.codegen.util.DEMLoaderHelper;
import pt.digitalis.dif.codegen.util.EntityUtils;
import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.dem.annotations.AnnotationMemberTags;
import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.dem.annotations.Application;
import pt.digitalis.dif.dem.annotations.Provider;
import pt.digitalis.dif.dem.annotations.ProviderList;
import pt.digitalis.dif.dem.annotations.Service;
import pt.digitalis.dif.dem.annotations.entities.ApplicationDefinition;
import pt.digitalis.dif.dem.annotations.entities.ProviderDefinition;
import pt.digitalis.dif.dem.annotations.entities.ServiceDefinition;
import pt.digitalis.dif.dem.annotations.entities.StageDefinition;
import pt.digitalis.dif.dem.annotations.entities.ValidatorDefinition;
import pt.digitalis.dif.dem.annotations.metaannotations.MetaAnnotationMemberTags;
import pt.digitalis.dif.dem.annotations.metaannotations.Primary;
import pt.digitalis.dif.dem.annotations.parameter.AddDocumentToRepository;
import pt.digitalis.dif.dem.annotations.parameter.CustomParameters;
import pt.digitalis.dif.dem.annotations.parameter.FormConfigurable;
import pt.digitalis.dif.dem.annotations.parameter.InjectParameter;
import pt.digitalis.dif.dem.annotations.parameter.InjectParameterErrors;
import pt.digitalis.dif.dem.annotations.parameter.InjectParameters;
import pt.digitalis.dif.dem.annotations.parameter.Parameter;
import pt.digitalis.dif.dem.annotations.parameter.Persist;
import pt.digitalis.dif.dem.annotations.security.AccessControl;
import pt.digitalis.dif.dem.annotations.stage.Callback;
import pt.digitalis.dif.dem.annotations.stage.Context;
import pt.digitalis.dif.dem.annotations.stage.ErrorStage;
import pt.digitalis.dif.dem.annotations.stage.ErrorView;
import pt.digitalis.dif.dem.annotations.stage.Execute;
import pt.digitalis.dif.dem.annotations.stage.Finalize;
import pt.digitalis.dif.dem.annotations.stage.Init;
import pt.digitalis.dif.dem.annotations.stage.InjectMessages;
import pt.digitalis.dif.dem.annotations.stage.Stage;
import pt.digitalis.dif.dem.annotations.stage.View;
import pt.digitalis.dif.dem.annotations.stage.controller.DispatcherMode;
import pt.digitalis.dif.dem.annotations.stage.controller.InjectAuthenticationError;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.dem.managers.IParameterManager;
import pt.digitalis.dif.dem.managers.impl.UsageIssuesManagerImpl;
import pt.digitalis.dif.dem.objects.ViewType;
import pt.digitalis.dif.dem.objects.issues.IssueScope;
import pt.digitalis.dif.dem.objects.issues.IssueType;
import pt.digitalis.dif.dem.objects.messages.MessageList;
import pt.digitalis.dif.dem.objects.parameters.IParameter;
import pt.digitalis.dif.dem.objects.parameters.IParameters;
import pt.digitalis.dif.dem.objects.parameters.ParameterScope;
import pt.digitalis.dif.exception.codegen.AnnotationMisuseException;
import pt.digitalis.dif.exception.codegen.DIFCodeGenerationException;
import pt.digitalis.dif.exception.codegen.IllegalAnnotationUsage;
import pt.digitalis.dif.exception.codegen.IncompatiblePrimaryAnnotationsException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.CodeGenUtils;
import pt.digitalis.utils.bytecode.holders.AnnotationHolder;
import pt.digitalis.utils.bytecode.holders.AnnotationMemberValueHolder;
import pt.digitalis.utils.bytecode.holders.AttributeHolder;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.bytecode.holders.MethodHolder;
import pt.digitalis.utils.inspection.exception.AuxiliaryOperationException;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import com.google.inject.Inject;

/**
 * Defines validation and enrichment logic for an annotationName or family of annotations. A set of methods validate
 * general rules for any annotationName. Specific methods that can be overrided on sub-classes allow fine-grained
 * validation rules. Methods that define the source code used on the code generation are also part of this class. TODO:
 * Some logic that read the annotation values for processing is repeated here and in the ClassEnhancerImpl. Should be
 * refactored for a single read operation. Maybe in 2.1 TODO: Making the xHolder classes inherit from a parent would
 * eliminate the need of several similar methods with just different parameter types (as in validate() and
 * validatePrimary()).
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Joo Galaio <a href="mailto:jgalaio@digitalis.pt">jgalaio@digitalis.pt</a>
 * @created Aug 29, 2007
 */
public class DEMAnnotationLogic {

    /** The 'application' attribute name. */
    final static protected String APPLICATION_ATTRIBUTE_NAME = "application";

    /** The authorization manager instance */
    static private IAuthorizationManager authorizationManager;

    /** The the default DEM Annotation Logic class. */
    final static public String DEFAULT_DEM_ANNOTATION_LOGIC_CLASS = DEMAnnotationLogic.class.getSimpleName();

    /** List of invalid characters for entity names/id's */
    static private final String ENTITY_NAME_BAD_CHARS = "\"\'?&";

    /** The 'provider' attribute name. */
    final static protected String PROVIDER_ATTRIBUTE_NAME = "provider";

    /** The 'service' attribute name. */
    final static protected String SERVICE_ATTRIBUTE_NAME = "service";

    /** The 'target' attribute name. */
    final static protected String TARGET_ATTRIBUTE_NAME = "target";

    /**
     * Factory method for DEMAnnotationLogic objects. Gets the name of a DEMAnnotationLogic subtype and generates the
     * appropriate object. Encapsulates the complex code used to create the objects by reflection. This method's client
     * is the <code>pt.digitalis.dif.codegen.util.DEMLoaderHelper#getAnnotationLogicMap()</code> which generates a map
     * whose values are DEMAnnotationLogic subclasses. These values can't be inserted on the map if they belong to a
     * runtime determined type.
     * 
     * @param annotationLogicClassName
     *            The name of the annotationName logic class
     * @param annotationClass
     *            The argument annotationName
     * @return a new DEMAnnotationLogic instance of the appropriate subclass
     * @throws ResourceNotFoundException
     *             if any needed class cannot be found
     * @throws AuxiliaryOperationException
     *             if any illegal operation or instantiation exceptions are raised
     */
    static public DEMAnnotationLogic makeObject(String annotationLogicClassName, ClassHolder annotationClass)
            throws ResourceNotFoundException, AuxiliaryOperationException
    {

        String annotationClassName = annotationClass.getName();
        boolean isPrimary = annotationClass.containsAnnotation(Primary.class.getCanonicalName());

        DEMAnnotationLogic annotationLogicClassObject = null;

        if (annotationLogicClassName.equals(DEMAnnotationLogic.DEFAULT_DEM_ANNOTATION_LOGIC_CLASS))
            annotationLogicClassObject = new DEMAnnotationLogic(annotationClass.getFQName(), annotationClassName,
                    isPrimary);

        else
        {

            // TODO: All this should be in the inspection utils utility classes.
            // Move it there.

            Class<?> annotationLogicClass = null;

            try
            {
                // Create an instance of the logic class
                annotationLogicClass = Class.forName(annotationLogicClassName);

            }
            catch (ClassNotFoundException classNotFoundException)
            {
                throw new ResourceNotFoundException("Could not create class " + annotationLogicClassName
                        + " (Does such type exists?)", classNotFoundException);
            }

            // Convert to subclass (NOTE: any class is a subclass of itself)
            Class<? extends DEMAnnotationLogic> annotationLogicClassAsSubclass = annotationLogicClass
                    .asSubclass(DEMAnnotationLogic.class);

            // DIFInitializer ctor holder
            Constructor<? extends DEMAnnotationLogic> constructor;
            try
            {
                // Get the constructor with the appropriate number of args
                constructor = annotationLogicClassAsSubclass.getConstructor(String.class, String.class, boolean.class);

            }
            catch (NoSuchMethodException noSuchMethodException)
            {
                throw new ResourceNotFoundException("Could not found an appropriate constructor for "
                        + annotationLogicClassAsSubclass.getCanonicalName() + " (Wrong number or type of args?)",
                        noSuchMethodException);
            }

            try
            {
                // Create object of the exact type
                annotationLogicClassObject = constructor.newInstance(annotationClass.getFQName(), annotationClassName,
                        isPrimary);

            }
            catch (IllegalArgumentException illegalArgumentException)
            {
                throw new AuxiliaryOperationException("Could not create "
                        + annotationLogicClassAsSubclass.getCanonicalName() + " object! (Illegal argument?)",
                        illegalArgumentException);
            }
            catch (InstantiationException instantiationException)
            {
                throw new AuxiliaryOperationException(
                        "Could not create " + annotationLogicClassAsSubclass.getCanonicalName()
                                + " object! (Interface or abstract class?)", instantiationException);
            }
            catch (IllegalAccessException illegalAccessException)
            {
                throw new AuxiliaryOperationException("Could not create "
                        + annotationLogicClassAsSubclass.getCanonicalName() + " object! (Non-public constructor?)",
                        illegalAccessException);
            }
            catch (InvocationTargetException invocationTargetException)
            {
                throw new AuxiliaryOperationException(
                        "Could not create " + annotationLogicClassAsSubclass.getCanonicalName()
                                + " object! (Previously thrown exception?)", invocationTargetException);
            }
        }

        // Return result
        return annotationLogicClassObject;
    }

    /** The canonical annotation name to validate. */
    private final String annotationFQName;

    /** The annotation name to validate. */
    private String annotationName;

    /** DEM primary annotationName flag. */
    private final boolean primary;

    /**
     * Constructor.
     * 
     * @param annotationFQName
     *            the annotation fully qualified name
     * @param annotationName
     *            the annotation simple name
     * @param isPrimary
     *            T if annotation is primary, F otherwise
     */
    public DEMAnnotationLogic(String annotationFQName, String annotationName, boolean isPrimary)
    {
        this.annotationFQName = annotationFQName;
        this.annotationName = annotationName;
        this.primary = isPrimary;
    }

    /**
     * This method will define the code to enhance the classes based on the annotations. It will create the appropriate
     * methods and their code and add them to the classEnhancement object.
     * 
     * @param classEnhancementContext
     *            the class enhancement context
     * @param annotation
     *            the annotation used to define the source code needed for enhancement
     * @throws ResourceNotFoundException
     *             if annotation members can't be accessed
     * @throws DIFCodeGenerationException
     */
    public void addSourceCodeForAnnotation(AnnotationHolder annotation, ClassEnhancementContext classEnhancementContext)
            throws ResourceNotFoundException, DIFCodeGenerationException
    {
        try
        {
            // Get the class holder
            ClassHolder clazz = classEnhancementContext.getOriginalClassObject();

            // Get member values
            Map<String, AnnotationMemberValueHolder> memberValues = annotation.getMembers();

            // Get annotation type name
            String annotationName = annotation.getName();

            // Prepare source for annotations...
            if (annotationName.equals(ProviderDefinition.class.getCanonicalName()))
            {
                // Get annotation member values
                String id = memberValues.get(AnnotationMemberTags.ENTITY_DEFINITION_ID).toString();
                boolean defaultProvider = memberValues.get(AnnotationMemberTags.PROVIDER_DEFINITION_DEFAULT_PROVIDER)
                        .toBoolean();

                // Check if id generation is needed...
                if (AnnotationTags.GENERATE_ID.equals(id))
                    id = CodeGenUtils.generateID(clazz.getName());

                // Add source to class enhancement context
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_ID_METHOD, "{return \"" + id.toString()
                        + "\";}");

                // Get annotation member values for name
                String name = memberValues.get(AnnotationMemberTags.ENTITY_DEFINITION_NAME).toString();
                // Add source to class enhancement context
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_NAME_METHOD,
                        "{return \"" + name.toString() + "\";}");
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_ORIGINALCLASSNAME_METHOD, "{return \""
                        + clazz.getFQName() + "\";}");

                if (!StringUtils.containsNone(id, ENTITY_NAME_BAD_CHARS.toCharArray()))
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                            classEnhancementContext.getOriginalClassObject().getFQName(),
                            "Cannot use any of \"" + ENTITY_NAME_BAD_CHARS + "\" for entity id's", null);
                if (!StringUtils.containsNone(name, ENTITY_NAME_BAD_CHARS.toCharArray()))
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                            classEnhancementContext.getOriginalClassObject().getFQName(),
                            "Cannot use any of \"" + ENTITY_NAME_BAD_CHARS + "\" for entity names", null);

                if (defaultProvider)
                    DEMRegistryImpl.setDefaultProvider(id);

            }
            else if (annotationName.equals(ApplicationDefinition.class.getCanonicalName()))
            {
                // Get annotation member values for id
                String id = memberValues.get(AnnotationMemberTags.ENTITY_DEFINITION_ID).toString();

                // Check if id generation is needed...
                if (AnnotationTags.GENERATE_ID.equals(id))
                    id = CodeGenUtils.generateID(clazz.getName());

                // Add source to class enhancement context
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_ID_METHOD, "{return \"" + id.toString()
                        + "\";}");

                // Get annotation member values for name
                String name = memberValues.get(AnnotationMemberTags.ENTITY_DEFINITION_NAME).toString();

                if (!StringUtils.containsNone(id, ENTITY_NAME_BAD_CHARS.toCharArray()))
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                            classEnhancementContext.getOriginalClassObject().getFQName(),
                            "Cannot use any of \"" + ENTITY_NAME_BAD_CHARS + "\" for entity id's", null);
                if (!StringUtils.containsNone(name, ENTITY_NAME_BAD_CHARS.toCharArray()))
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                            classEnhancementContext.getOriginalClassObject().getFQName(),
                            "Cannot use any of \"" + ENTITY_NAME_BAD_CHARS + "\" for entity names", null);

                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_NAME_METHOD,
                        "{return \"" + name.toString() + "\";}");
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_ORIGINALCLASSNAME_METHOD, "{return \""
                        + clazz.getFQName() + "\";}");

                // Get annotation member values for provider
                String provider = memberValues.get(AnnotationMemberTags.APPLICATION_DEFINITION_PROVIDER).toString();

                // Check if it's the default provider and proceed accordingly
                if (AnnotationTags.DEFAULT_PROVIDER.equals(provider))
                    classEnhancementContext.addEnhancement(CGAncillaries.APPLICATION_GET_PROVIDER_METHOD,
                            "{return getTemplateResources().getDEMManager().getDefaultProvider();}");
                else
                    classEnhancementContext.addEnhancement(CGAncillaries.APPLICATION_GET_PROVIDER_METHOD,
                            "{return getTemplateResources().getDEMManager().getProvider(\"" + provider + "\");}");

                // Add default access. If access control exists it will be processed
                // later on
                if (!clazz.containsAnnotation(AccessControl.class.getCanonicalName()))
                    getAuthorizationManager().grantDefaultPublicAccess(Entity.APPLICATION, id);

            }
            else if (annotationName.equals(ServiceDefinition.class.getCanonicalName()))
            {

                // Get annotation member values for id
                String id = memberValues.get(AnnotationMemberTags.ENTITY_DEFINITION_ID).toString();

                // Check if id generation is needed...
                if (AnnotationTags.GENERATE_ID.equals(id))
                    id = CodeGenUtils.generateID(clazz.getName());

                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_ID_METHOD, "{return \"" + id.toString()
                        + "\";}");

                // Get annotation member values for name
                String name = memberValues.get(AnnotationMemberTags.ENTITY_DEFINITION_NAME).toString();
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_NAME_METHOD,
                        "{return \"" + name.toString() + "\";}");
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_ORIGINALCLASSNAME_METHOD, "{return \""
                        + clazz.getFQName() + "\";}");

                // Get annotation member values for application
                String application = memberValues.get(AnnotationMemberTags.SERVICE_DEFINITION_APPLICATION).toString();

                classEnhancementContext.addEnhancement(CGAncillaries.SERVICE_GET_APPLICATION_METHOD,
                        "{return getTemplateResources().getDEMManager().getApplication(\"" + application + "\");}");

                if (!StringUtils.containsNone(id, ENTITY_NAME_BAD_CHARS.toCharArray()))
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                            classEnhancementContext.getOriginalClassObject().getFQName(),
                            "Cannot use any of \"" + ENTITY_NAME_BAD_CHARS + "\" for entity id's", null);
                if (!StringUtils.containsNone(name, ENTITY_NAME_BAD_CHARS.toCharArray()))
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                            classEnhancementContext.getOriginalClassObject().getFQName(),
                            "Cannot use any of \"" + ENTITY_NAME_BAD_CHARS + "\" for entity names", null);

                // Add default access. If access control exists it will be processed
                // later on
                if (!clazz.containsAnnotation(AccessControl.class.getCanonicalName()))
                    getAuthorizationManager().grantDefaultPublicAccess(Entity.SERVICE, id);

            }
            else if (annotationName.equals(StageDefinition.class.getCanonicalName()))
            {

                // Get annotation member values for id and override stage
                String id = memberValues.get(AnnotationMemberTags.ENTITY_DEFINITION_ID).toString();
                String override = memberValues.get(AnnotationMemberTags.STAGE_OVERRIDE_DEFAULT).toString();

                // Check if id generation is needed...
                if (AnnotationTags.GENERATE_ID.equals(id))
                    id = CodeGenUtils.generateID(clazz.getName());
                else
                    id = id.toLowerCase();

                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_ID_METHOD, "{return \"" + id + "\";}");

                if (!AnnotationTags.NONE.equals(override))
                    classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_OVERRIDE_METHOD, "{return \""
                            + override + "\";}");

                // Get annotation member values for name
                String name = memberValues.get(AnnotationMemberTags.ENTITY_DEFINITION_NAME).toString();
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_NAME_METHOD,
                        "{return \"" + name.toString() + "\";}");
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_ORIGINALCLASSNAME_METHOD, "{return \""
                        + clazz.getFQName() + "\";}");
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_STAGEINSTANCECLASSNAME_METHOD,
                        "{return \"" + clazz.getFQName() + CGAncillaries.STAGE_INSTANCE_ID + "\";}");

                // Get annotation member values for service
                String service = memberValues.get(AnnotationMemberTags.STAGE_DEFINITION_SERVICE).toString();

                if (!StringUtils.containsNone(id, ENTITY_NAME_BAD_CHARS.toCharArray()))
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                            classEnhancementContext.getOriginalClassObject().getFQName(),
                            "Cannot use any of \"" + ENTITY_NAME_BAD_CHARS + "\" for entity id's", null);
                if (!StringUtils.containsNone(name, ENTITY_NAME_BAD_CHARS.toCharArray()))
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME,
                            classEnhancementContext.getOriginalClassObject().getFQName(),
                            "Cannot use any of \"" + ENTITY_NAME_BAD_CHARS + "\" for entity names", null);

                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_GET_SERVICE_METHOD,
                        "{return getTemplateResources().getDEMManager().getService(\"" + service + "\");}");

                // Add default access. If access control exists it will be processed
                // later on
                if (!clazz.containsAnnotation(AccessControl.class.getCanonicalName()))
                    getAuthorizationManager().grantDefaultPublicAccess(Entity.STAGE, id);

            }
            else if (annotationName.equals(ValidatorDefinition.class.getCanonicalName()))
            {

                // TODO: Create Unit tests for this annotation when the ValidatorManager is implemented

                String validatorID = annotation.getMembers().get(AnnotationMemberTags.ENTITY_DEFINITION_ID).toString();

                // Implementing IValidator interface methods
                classEnhancementContext.addEnhancement(CGAncillaries.ENTITY_GET_ID_METHOD, "{return \"" + validatorID
                        + "\";}");

                // TODO: This validatorManager does not exist. Determine what this
                // implementation should be.
                classEnhancementContext.addEnhancement(CGAncillaries.VALIDATOR_GET_BOUND_PARAMETER_IDS_METHOD,
                        "{return getTemplateResources().getValidatorManager().getBoundParameters(\"" + validatorID
                                + "\");}");

                // Throw UnsupportedOpertationException to comply with Java
                // standards and warn user of unimplemented
                // feature.
                throw new UnsupportedOperationException(annotationName
                        + " not yet supported! Please remove it from class " + clazz.getFQName());
            }

            else if (annotationName.equals(Callback.class.getCanonicalName()))
            {

                CallbackType callbackType;

                // Get Callback type
                AnnotationMemberValueHolder memberValue = memberValues.get(AnnotationMemberTags.ANNOTATION_VALUE);

                if (memberValue.getManagedAnnotationValue() == null)
                    callbackType = CallbackType.SIMPLE;
                else
                    callbackType = CallbackType.getCallbackTypeByName(memberValue.enumValuetoString());

                // If callback type is OFF the code is the same copied from the
                // template
                if (!callbackType.equals(CallbackType.OFF))
                {
                    classEnhancementContext.addEnhancement(CGAncillaries.CALLBACK_GET_CALLBACK_TYPE_METHOD,
                            "{return getTemplateResources().getCallBack(\"" + callbackType + "\");}");

                    classEnhancementContext.addEnhancement(CGAncillaries.CALLBACK_HAS_CALLBACK_ENABLED_METHOD,
                            "{return true;}");
                }

            }
            else if (annotationName.equals(ErrorStage.class.getCanonicalName()))
            {
                addStage(annotation, classEnhancementContext, true, true, null);
            }
            else if (annotationName.equals(View.class.getCanonicalName()))
            {
                addView(annotation, classEnhancementContext, false, true, null);
            }
            else if (annotationName.equals(ErrorView.class.getCanonicalName()))
            {
                addView(annotation, classEnhancementContext, true, true, null);
            }
            else if (annotationName.equals(DispatcherMode.class.getCanonicalName()))
            {

                /*
                 * Implementation Note: This annotation treatment is done if the members evaluate to false!
                 */

                if (annotation.getMembers().containsKey(AnnotationMemberTags.DISPATCHER_MODE_AUTHENTICATE)
                        && !annotation.getMembers().get(AnnotationMemberTags.DISPATCHER_MODE_AUTHENTICATE).toBoolean())
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_HAS_AUTHENTICATION, "{return false;}");

                if (annotation.getMembers().containsKey(AnnotationMemberTags.DISPATCHER_MODE_AUTHORIZE)
                        && !annotation.getMembers().get(AnnotationMemberTags.DISPATCHER_MODE_AUTHORIZE).toBoolean())
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_HAS_AUTHORIZATION, "{return false;}");
            }
        }
        catch (ResourceNotFoundException e)
        {
            // Rethrow for outer handling
            throw e;
        }
        catch (Exception e)
        {
            DIFCodeGenerationException codeGenException;

            if (e instanceof DIFCodeGenerationException)
            {
                codeGenException = (DIFCodeGenerationException) e;
            }
            else
            {

                codeGenException = new DIFCodeGenerationException(e);
                codeGenException.addToExceptionContext("Original Class Name", classEnhancementContext
                        .getOriginalClassObject().getFQName());
            }

            codeGenException.addToExceptionContext("Annotation", annotation.getName());

            throw codeGenException;
        }
    }

    /**
     * This method will define the code to enhance the attributes based on the annotations. It will create the
     * appropriate methods and their code and add them to the classEnhancement object.
     * 
     * @param classEnhancementContext
     *            the class enhancement context
     * @param annotation
     *            the annotation used to define the source code needed for enhancement
     * @param attribute
     *            the attribute to be enhanced (to make the context available to the source building process)
     * @throws ResourceNotFoundException
     *             if annotation members can't be accessed
     * @throws DIFCodeGenerationException
     */
    public void addSourceCodeForAnnotation(ClassEnhancementContext classEnhancementContext,
            AnnotationHolder annotation, AttributeHolder attribute) throws ResourceNotFoundException,
            DIFCodeGenerationException
    {
        try
        {
            // Get the attribute name
            String attributeName = attribute.getName();

            // Get annotation type name
            String annotationName = annotation.getName();

            // Get member values
            Map<String, AnnotationMemberValueHolder> memberValues = annotation.getMembers();

            if (annotationName.equals(Context.class.getCanonicalName()))
            {
                // DIFContext is the 1st argument of
                // EntityCGTemplate.DIF_INIT_METHOD_NAME
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, "this." + attributeName
                                + "=this.getContext();");

            }
            else if (annotationName.equals(InjectMessages.class.getCanonicalName()))
            {
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, "this." + attributeName
                                + "=this.getMessages();");
            }
            else if (annotationName.equals(Stage.class.getCanonicalName()))
            {
                addStage(annotation, classEnhancementContext, false, false, attributeName);
            }
            else if (annotationName.equals(ErrorStage.class.getCanonicalName()))
            {
                addStage(annotation, classEnhancementContext, true, false, attributeName);
            }
            else if (annotationName.equals(View.class.getCanonicalName()))
            {
                addView(annotation, classEnhancementContext, false, annotation.getMembers().get("defaultView")
                        .toBoolean(), attributeName);
            }
            else if (annotationName.equals(ErrorView.class.getCanonicalName()))
            {
                addView(annotation, classEnhancementContext, true, false, attributeName);
            }
            else if (annotationName.equals(Provider.class.getCanonicalName()))
            {
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attributeName
                                + "=getService().getApplication().getProvider();");
            }
            else if (annotationName.equals(Application.class.getCanonicalName()))
            {
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attributeName
                                + "=getService().getApplication();");
            }
            else if (annotationName.equals(Service.class.getCanonicalName()))
            {
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attributeName + "=getService();");

                // The following section implements the following annotations:
                // Parameter, Persist, Validator
            }
            else if (annotationName.equals(Parameter.class.getCanonicalName()))
            {
                // Fetch id
                String id = memberValues.get(AnnotationMemberTags.PARAMETER_ID).toString();

                // No id given, use the attribute name to lower case
                if (id.equals(AnnotationTags.GENERATE_ID))
                    id = attribute.getName().toLowerCase();

                // Get the ID and the type of the parent...
                String parentID = EntityUtils.getEntityID(attribute.getParentClass());
                Entity parentType = EntityUtils.getEntityType(attribute.getParentClass());

                IParameterManager parameterManager = DIFIoCRegistry.getRegistry().getImplementation(
                        IParameterManager.class);

                IParameter<?> param = parameterManager.getParameter(parentType, parentID, id);

                if (param == null)
                {
                    // Parameter was erronious. Report it.
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.ERROR, IssueScope.LOADTIME, parentID,
                            "Bad parameter: \"" + id + "\"", null);
                }
                else
                {
                    // Init/finalize code...
                    String attrInitializer = attributeName
                            + " = ("
                            + annotation.getParentAttribute().getAttributeType()
                            + ")_CG_parameterErrors.refreshParameter(getParameters().getStageParameters().getParameter(\""
                            + id + "\"), this);";

                    if (attribute.containsAnnotation(AddDocumentToRepository.class.getCanonicalName()))
                    {
                        if (annotation.getParentAttribute().getAttributeType()
                                .equals(DocumentRepositoryEntry.class.getCanonicalName()))
                            attrInitializer += " if (" + attributeName + " != null && " + attributeName
                                    + ".getId() == null ) {" + attributeName
                                    + " = getTemplateResources().getDocumentRepositoryManager().addDocument("
                                    + attributeName + ");" + "}";
                    }

                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attrInitializer);

                    if (!param.isReadonly())
                    {
                        String attrFinalizer = "getParameters().getStageParameters().getParameter(\"" + id
                                + "\").setValue(" + attributeName + ", this);";
                        classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                                CGAncillaries.STAGE_POSTPROCESSING_METHOD_NAME, attrFinalizer);
                    }
                }

            }
            else if (annotationName.equals(InjectParameter.class.getCanonicalName()))
            {

                // Fetch id
                String id = memberValues.get(AnnotationMemberTags.PARAMETER_ID).toString();

                // No id given, use the attribute name to lower case
                if (id.equals(AnnotationTags.GENERATE_ID))
                    id = attribute.getName().toLowerCase();

                // Get the parent service ID...
                AnnotationHolder stageDefinitionAnnotation = attribute.getParentClass().getAnnotations()
                        .get(StageDefinition.class.getCanonicalName());
                String serviceID = stageDefinitionAnnotation.getMembers()
                        .get(AnnotationMemberTags.STAGE_DEFINITION_SERVICE).toString();

                IParameterManager parameterManager = DIFIoCRegistry.getRegistry().getImplementation(
                        IParameterManager.class);

                IParameter<?> param = parameterManager.getParameter(Entity.SERVICE, serviceID, id);
                String parameterGetter = null;

                // Try to find the parameter through the DEM hierarchy
                if (param != null)
                    parameterGetter = "getServiceParameters()";
                else
                {
                    IDEMManager demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);
                    IApplication application = demManager.getService(serviceID).getApplication();
                    param = parameterManager.getParameters(application).getParameter(id);

                    if (param != null)
                        parameterGetter = "getApplicationParameters()";
                    else
                    {
                        IProvider provider = application.getProvider();
                        param = parameterManager.getParameters(provider).getParameter(id);

                        if (param != null)
                            parameterGetter = "getProviderParameters()";
                    }
                }

                if (parameterGetter == null)
                    // Throw UnsupportedOpertationException to comply with Java
                    // standards and warn user of the misusage.
                    throw new UnsupportedOperationException("Parameter injection failed for attribute: \""
                            + attributeName + "\"\n" + "Could not find a parameter for the id \"" + id
                            + "\" in the DEM hierarchy.");

                // Init/finalize code...
                String attrInitializer = attributeName + " = (" + annotation.getParentAttribute().getAttributeType()
                        + ")_CG_parameterErrors.refreshParameter(getParameters()." + parameterGetter
                        + ".getParameter(\"" + id + "\"), this);";

                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attrInitializer);

                if (param != null && !param.isReadonly())
                {
                    String attrFinalizer = "getParameters()." + parameterGetter + ".getParameter(\"" + id
                            + "\").setValue(" + attributeName + ", this);";
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.STAGE_POSTPROCESSING_METHOD_NAME, attrFinalizer);
                }

            }
            else if (annotationName.equals(InjectParameters.class.getCanonicalName()))
            {
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attributeName + "= getParameters();");

            }
            else if (annotationName.equals(InjectParameterErrors.class.getCanonicalName()))
            {
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attributeName
                                + "= getParameterErrors();");
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_POSTPROCESSING_METHOD_NAME, "setParameterErrors(" + attributeName + ");");

                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_HAS_PARAMETER_ERROR_INJECTION,
                        "return true;");

            }
            else if (annotationName.equals(ProviderList.class.getCanonicalName()))
            {

                String source = null;
                String signature = attribute.getSignature();

                // TODO: The invocation of other classes is slow to compile
                // in Javassist. This line is taking 50 to
                // 100 times more to compile than all other ones.

                if (signature.contains(CGAncillaries.MAP_RETURN))
                {
                    source = attributeName + "=getTemplateResources().getDEMManager().getProviders();";
                }
                else if (signature.contains(CGAncillaries.LIST_RETURN))
                {
                    source = attributeName + "= getTemplateResources().getProvidersAsList();";
                }

                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, source);

            }
            else if (annotationName.equals(Inject.class.getCanonicalName()))
            {
                Entity parentType = EntityUtils.getEntityType(attribute.getParentClass());
                if (parentType == Entity.STAGE)
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_PROXY_ID,
                            CGAncillaries.STAGE_HAS_INJECTED_CONTRIBUTIONS, "return true;");
                else
                    throw new UnsupportedOperationException("Dependency injection failed for attribute: \""
                            + attributeName + "\"\n" + "@Inject can only be used in STAGE classes and not in "
                            + parentType.toString() + ".");
            }
            else if (annotationName.equals(InjectAuthenticationError.class.getCanonicalName()))
            {
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attributeName
                                + "= getAuthenticationError();");
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_PROXY_ID,
                        CGAncillaries.STAGE_HAS_AUTHENTICATION_ERROR_INJECTION, "return true;");
            }
        }
        catch (ResourceNotFoundException e)
        {
            // Rethrow for outer handling
            throw e;
        }
        catch (Exception e)
        {
            DIFCodeGenerationException codeGenException;

            if (e instanceof DIFCodeGenerationException)
            {
                codeGenException = (DIFCodeGenerationException) e;
            }
            else
            {

                codeGenException = new DIFCodeGenerationException(e);
                codeGenException.addToExceptionContext("Original Class Name", classEnhancementContext
                        .getOriginalClassObject().getFQName());
            }

            codeGenException.addToExceptionContext("Annotation", annotation.getName());
            codeGenException.addToExceptionContext("Attribute", attribute.getName());

            throw codeGenException;
        }
    }

    /**
     * This method will define the code to enhance the methods based on the annotations. It will create the appropriate
     * methods and their code and add them to the classEnhancement object.
     * 
     * @param classEnhancementContext
     *            the class enhancement context
     * @param annotation
     *            the annotation used to define the source code needed for enhancement
     * @param method
     *            the method to be enhanced (to make the context available to the source building process)
     * @throws ResourceNotFoundException
     *             if annotation members can't be accessed
     * @throws DIFCodeGenerationException
     */
    public void addSourceCodeForAnnotation(ClassEnhancementContext classEnhancementContext,
            AnnotationHolder annotation, MethodHolder method) throws ResourceNotFoundException,
            DIFCodeGenerationException
    {
        try
        {
            // Get annotation and method info
            String annotationName = annotation.getName();
            String methodName = method.getName();
            String signature = method.getSignature();

            if (annotationName.equals(CustomParameters.class.getCanonicalName()))
            {

                // Get the ID and the type of the parent...
                String parentID = EntityUtils.getEntityID(method.getParentClass());
                Entity parentType = EntityUtils.getEntityType(method.getParentClass());

                if (parentType == Entity.STAGE)
                {
                    if (signature.equals(CGAncillaries.IPARAMETERS_ARGS + CGAncillaries.VOID_RETURN))
                    {
                        classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                                CGAncillaries.CALL_INIT_CUSTOM_PARAMETERS_METHOD, "{this." + methodName
                                        + "($1);_CG_hasCustomParameters = true;}");
                    }
                    else
                    {
                        throw new UnsupportedOperationException("(" + parentType.toString() + ": " + parentID
                                + ", method: " + methodName + ") @" + CustomParameters.class.getSimpleName()
                                + " can only be used on methods with void return and a single parameter of type "
                                + IParameters.class.getSimpleName() + ".");
                    }
                }
                else
                    throw new UnsupportedOperationException("(" + parentType.toString() + ": " + parentID + ") @"
                            + CustomParameters.class.getSimpleName() + " can only be used for Stage entities.");
            }
            else if (annotationName.equals(Init.class.getCanonicalName()))
            {

                // Get the ID and the type of the parent...
                String parentID = EntityUtils.getEntityID(method.getParentClass());
                Entity parentType = EntityUtils.getEntityType(method.getParentClass());

                if (parentType == Entity.APPLICATION)
                {
                    if (signature.equals(CGAncillaries.VOID_ARGS + CGAncillaries.VOID_RETURN))
                    {
                        classEnhancementContext.addEnhancement(CGAncillaries.APP_INIT_METHOD, "{this." + methodName
                                + "();}");
                    }
                    else
                    {
                        throw new UnsupportedOperationException("(" + parentType.toString() + ": " + parentID
                                + ") @Init events for applications must comply with: no arguments and void return.");
                    }
                }
                else if (parentType == Entity.STAGE)
                {
                    if (signature.equals(CGAncillaries.VOID_ARGS + CGAncillaries.VOID_RETURN))
                    {
                        /*
                         * IMPLEMENTATION NOTE: Albeit the annotated method's return is void, IStage#init(DIFContext)
                         * returns a boolean, so the code generation must inject a boolean return. It always returns T.
                         */
                        classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                                CGAncillaries.CALL_INIT_METHOD, "{this." + methodName + "(); return true;}");

                    }
                    else if (signature.equals(CGAncillaries.IDIF_CONTEXT_ARGS + CGAncillaries.VOID_RETURN))
                    {
                        /*
                         * IMPLEMENTATION NOTE: Albeit the annotated method's return is void, IStage#init(DIFContext)
                         * returns a boolean, so the code generation must inject a boolean return. It always returns T.
                         */
                        classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                                CGAncillaries.CALL_INIT_METHOD, "{this." + methodName + "($1); return true;}");

                    }
                    else if (signature.equals(CGAncillaries.VOID_ARGS + CGAncillaries.BOOLEAN_RETURN))
                    {
                        classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                                CGAncillaries.CALL_INIT_METHOD, "{return this." + methodName + "();}");

                    }
                    else if (signature.equals(CGAncillaries.IDIF_CONTEXT_ARGS + CGAncillaries.BOOLEAN_RETURN))
                    {
                        classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                                CGAncillaries.CALL_INIT_METHOD, "{return this." + methodName + "($1);}");
                    }
                }
                else
                    throw new UnsupportedOperationException("(" + parentType.toString() + ": " + parentID
                            + ") @Init can only be user for Stage or Application entities.");
            }
            else if (annotationName.equals(Execute.class.getCanonicalName()))
            {

                String code = getCodeForExecutionMethod(signature, methodName);

                if (code != null)
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_METHOD, code);

            }
            else if (annotationName.equals(Finalize.class.getCanonicalName()))
            {

                if (signature.equals(CGAncillaries.VOID_ARGS + CGAncillaries.VOID_RETURN))
                {
                    /*
                     * IMPLEMENTATION NOTE: Although the annotated method's return is void, IStage#finalize(DIFContext)
                     * returns a boolean, so the code generation must inject a boolean return. It always returns T.
                     */
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_FINALIZE_METHOD, "{this." + methodName + "(); return true;}");
                }
                else if (signature.equals(CGAncillaries.IDIF_CONTEXT_ARGS + CGAncillaries.VOID_RETURN))
                {
                    /*
                     * IMPLEMENTATION NOTE: Albeit the annotated method's return is void, IStage#finalize(DIFContext)
                     * returns a boolean, so the code generation must inject a boolean return. It always returns T.
                     */
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_FINALIZE_METHOD, "{this." + methodName + "($1); return true;}");
                }
                else if (signature.equals(CGAncillaries.VOID_ARGS + CGAncillaries.BOOLEAN_RETURN))
                {
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_FINALIZE_METHOD, "{return this." + methodName + "();}");
                }
                else if (signature.equals(CGAncillaries.IDIF_CONTEXT_ARGS + CGAncillaries.BOOLEAN_RETURN))
                {
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_FINALIZE_METHOD, "{return this." + methodName + "($1);}");
                }
            }
            else if (annotationName.equals(Inject.class.getCanonicalName()))
            {
                Entity parentType = EntityUtils.getEntityType(method.getParentClass());
                if (parentType == Entity.STAGE)
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_PROXY_ID,
                            CGAncillaries.STAGE_HAS_INJECTED_CONTRIBUTIONS, "return true;");
                else
                    throw new UnsupportedOperationException("Dependency injection failed for method: \"" + method
                            + "\"\n" + "@Inject can only be used in STAGE classes and not in " + parentType.toString()
                            + ".");
            }
        }
        catch (ResourceNotFoundException e)
        {
            // Rethrow for outer handling
            throw e;
        }
        catch (Exception e)
        {
            DIFCodeGenerationException codeGenException;

            if (e instanceof DIFCodeGenerationException)
            {
                codeGenException = (DIFCodeGenerationException) e;
            }
            else
            {

                codeGenException = new DIFCodeGenerationException(e);
                codeGenException.addToExceptionContext("Original Class Name", classEnhancementContext
                        .getOriginalClassObject().getFQName());
            }

            codeGenException.addToExceptionContext("Annotation", annotation.getName());
            codeGenException.addToExceptionContext("Method", method.getName());

            throw codeGenException;
        }
    }

    /**
     * Adds an injected stage to the current stage to enhance. Injects it the the attribute associated (if passed) and
     * to the inner Maps.
     * 
     * @param annotation
     *            the annotation that declared the stage
     * @param classEnhancementContext
     *            the class enhancement context
     * @param errorStage
     *            if the stage to add is an error stage
     * @param defaultStage
     *            if the current stage is the default stage
     * @param attributeName
     *            the name of the attribute to assign the new stage to
     * @throws ResourceNotFoundException
     *             if the annotation cannot be read
     * @throws DIFCodeGenerationException
     */
    protected void addStage(AnnotationHolder annotation, ClassEnhancementContext classEnhancementContext,
            boolean errorStage, boolean defaultStage, String attributeName) throws ResourceNotFoundException,
            DIFCodeGenerationException
    {
        try
        {
            String stageID;

            AnnotationMemberValueHolder target = annotation.getMembers().get("target");

            if (target == null)
                target = annotation.getMembers().get("value");

            if (target != null)
                stageID = "\"" + target.toString() + "\"";
            else
                stageID = attributeName + ".getID()";

            String stageDeclaration = "getTemplateResources().getDEMManager().getStage(" + stageID + ")";

            // If no attribute is available the stage will be obtained from the
            // manager...
            if (attributeName != null)
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attributeName + " = "
                                + stageDeclaration + ";");

            classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INJECTED_STAGES_BUILDER, "injectedStages.add("
                    + stageID + ");");

            if (errorStage)
            {
                String[] exceptions = CGAncillaries.parse(
                        annotation.getMembers().get(AnnotationMemberTags.ERROR_STAGE_EXCEPTIONS).toString(),
                        CGAncillaries.COMMA);

                for (int i = 0; i < exceptions.length; i++)
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INJECTED_ERRORSTAGES_BUILDER,
                            "errorStages.put(\"" + exceptions[i] + "\"," + stageID + ");");
            }

            String key = CGAncillaries.STAGE_GET_DEFAULT_ERROR_STAGE;

            // If it's the default error stage or there is none yet defined yet,
            // this will become it!
            if (errorStage && (defaultStage || !classEnhancementContext.containsEnhancement(key)))
            {
                classEnhancementContext.addEnhancement(key, "{return " + stageDeclaration + ";}");
            }
        }
        catch (ResourceNotFoundException e)
        {
            // Rethrow for outer handling
            throw e;
        }
        catch (Exception e)
        {
            DIFCodeGenerationException codeGenException;

            if (e instanceof DIFCodeGenerationException)
            {
                codeGenException = (DIFCodeGenerationException) e;
            }
            else
            {

                codeGenException = new DIFCodeGenerationException(e);
                codeGenException.addToExceptionContext("Original Class Name", classEnhancementContext
                        .getOriginalClassObject().getFQName());
            }

            codeGenException.addToExceptionContext("Annotation", annotation.getName());
            codeGenException.addToExceptionContext("Attribute", attributeName);

            throw codeGenException;
        }
    }

    /**
     * Adds an injected view to the current stage to enhance. Injects it the the attribute associated (if passed) and to
     * the inner Maps.
     * 
     * @param annotation
     *            the annotation that declared the stage
     * @param classEnhancementContext
     *            the class enhancement context
     * @param errorView
     *            if the view to add is an error view
     * @param defaultView
     *            if the view to add is the default stage view
     * @param attributeName
     *            the name of the attribute to assign the new view to
     * @throws ResourceNotFoundException
     *             if the annotation cannot be read
     * @throws DIFCodeGenerationException
     */
    protected void addView(AnnotationHolder annotation, ClassEnhancementContext classEnhancementContext,
            boolean errorView, boolean defaultView, String attributeName) throws ResourceNotFoundException,
            DIFCodeGenerationException
    {
        try
        {
            // Get Error Stage id
            String target = annotation.getMembers().get(AnnotationMemberTags.VIEW_TARGET).toString();

            // Get view engine
            String engine = annotation.getMembers().get(AnnotationMemberTags.VIEW_ENGINE).toString();

            // Infer the view engine if not passed
            if (engine.equals(AnnotationTags.NONE))
            {
                int posExtension = target.lastIndexOf(".");

                if (posExtension != -1)
                    engine = target.substring(posExtension + 1);
            }

            // Set view type
            ViewType viewType = null;

            if (errorView)
                viewType = ViewType.ERROR;
            else
                viewType = ViewType.NORMAL;

            String viewDeclaration = "createView(\"" + engine + "\",\"" + viewType + "\",\"" + target + "\","
                    + defaultView + ")";

            // If no attribute is available the view will be generated...
            if (attributeName != null)
                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attributeName + " = "
                                + viewDeclaration + ";");

            classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INJECTED_VIEWS_BUILDER, "injectedViews.add("
                    + viewDeclaration + ");");

            if (errorView)
            {
                String[] exceptions = CGAncillaries.parse(
                        annotation.getMembers().get(AnnotationMemberTags.ERROR_STAGE_EXCEPTIONS).toString(),
                        CGAncillaries.COMMA);

                for (int i = 0; i < exceptions.length; i++)
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INJECTED_ERRORVIEWS_BUILDER,
                            "\nerrorViews.put(\"" + exceptions[i] + "\"," + viewDeclaration + ");");
            }

            String key = errorView ? CGAncillaries.STAGE_GET_DEFAULT_ERROR_VIEW : CGAncillaries.STAGE_GET_DEFAULT_VIEW;

            // If it's the default error stage or there is none yet defined yet,
            // this will become it!
            if (defaultView || !classEnhancementContext.containsEnhancement(key))
            {
                classEnhancementContext.addEnhancement(key, "{return " + viewDeclaration + ";}");
            }
        }
        catch (ResourceNotFoundException e)
        {
            // Rethrow for outer handling
            throw e;
        }
        catch (Exception e)
        {
            DIFCodeGenerationException codeGenException;

            if (e instanceof DIFCodeGenerationException)
            {
                codeGenException = (DIFCodeGenerationException) e;
            }
            else
            {

                codeGenException = new DIFCodeGenerationException(e);
                codeGenException.addToExceptionContext("Original Class Name", classEnhancementContext
                        .getOriginalClassObject().getFQName());
            }

            codeGenException.addToExceptionContext("Annotation", annotation.getName());
            codeGenException.addToExceptionContext("Attribute", attributeName);

            throw codeGenException;
        }
    }

    /**
     * Checks if a given annotation can be used with another.
     * 
     * @param annotationToCheck
     *            the annotation to check
     * @param thisAnnotation
     *            the annotation to validate against
     * @return F if the annotation to check is primary and doesn't yield to the other, T otherwise
     * @throws ResourceNotFoundException
     *             if any resource needed for validation could not be found.
     */
    protected boolean checkUseWith(AnnotationHolder annotationToCheck, AnnotationHolder thisAnnotation)
            throws ResourceNotFoundException
    {

        if (annotationToCheck.getName().equals(thisAnnotation.getName()))
            return true;

        if (isPrimary(annotationToCheck) && isPrimary(thisAnnotation))
        {
            if (yieldsTo(annotationToCheck, thisAnnotation.getName())
                    || yieldsTo(thisAnnotation, annotationToCheck.getName()))
                return true;
            else
                return false;
        }

        return true;
    }

    /**
     * Returns the annotationName.
     * 
     * @return the value of the annotationName
     */
    public String getAnnotation()
    {
        return this.annotationName;
    }

    /**
     * @return the authorizationManager
     */
    protected IAuthorizationManager getAuthorizationManager()
    {
        if (authorizationManager == null)
            authorizationManager = DIFIoCRegistry.getRegistry().getImplementation(IAuthorizationManager.class);

        return authorizationManager;
    }

    /**
     * Return the appropriated execute method call injecting a return view if necessary and the appropriated parameters
     * 
     * @param signature
     *            the method to call signature
     * @param methodName
     *            the method name to call
     * @return the line of code for the calling of the method
     */
    protected String getCodeForExecutionMethod(String signature, String methodName)
    {

        /*
         * IMPLEMENTATION NOTE: Although the annotated method's return is void, IStage#execute(DIFContext) returns a
         * ViewObject, so the code generation must inject a ViewObject return. It returns the default view.
         */

        if (signature.equals(CGAncillaries.VOID_ARGS + CGAncillaries.VOID_RETURN))
        {
            return "{this." + methodName + "(); return this.getDefaultView();}";

        }
        else if (signature.equals(CGAncillaries.IDIF_CONTEXT_ARGS + CGAncillaries.VOID_RETURN))
        {
            return "{this." + methodName + "($1); return this.getDefaultView();}";

        }
        else if (signature.equals(CGAncillaries.IDIF_CONTEXT_ARGS + CGAncillaries.VIEW_OBJECT_RETURN))
        {
            return "{return this." + methodName + "($1);}";

        }
        else if (signature.equals(CGAncillaries.VOID_ARGS + CGAncillaries.VIEW_OBJECT_RETURN))
        {
            return "{return this." + methodName + "();}";

        }
        else
            return null;
    }

    /**
     * @return T if the current annotation is a DEM annotation
     */
    protected boolean isDEMAnnotation()
    {
        try
        {
            // Google Guice's @Inject is a special case since it exists in the DEMLogicMap but is not itself a DEM
            // annotation
            return (!annotationFQName.equals(Inject.class.getCanonicalName()))
                    && (DEMLoaderHelper.getAnnotationLogicMap().keySet().contains(this.annotationFQName));

        }
        catch (ResourceNotFoundException e)
        {
            return false;
        }
        catch (AuxiliaryOperationException e)
        {
            return false;
        }
    }

    /**
     * Returns the value of the primary annotationName flag.
     * 
     * @return T if the annotationName is primary, F otherwise
     */
    final public boolean isPrimary()
    {
        return primary;
    }

    /**
     * Checks if a given annotation is primary.
     * 
     * @param annotation
     *            the annotation to check
     * @return T if annotation is meta-annotated with <code>@Primary</code>, F otherwise.
     * @throws ResourceNotFoundException
     *             if meta-annotations can't be found
     */
    protected boolean isPrimary(AnnotationHolder annotation) throws ResourceNotFoundException
    {
        return annotation.getMetaAnnotations().keySet().contains(Primary.class.getCanonicalName());
    }

    /**
     * Checks if a given set of access flags matches a static member.
     * 
     * @param accessFlags
     *            the members access flags
     * @return T if it's a private member, F otherwise
     */
    private boolean isPrivate(int accessFlags)
    {
        if ((accessFlags & 2) == 2)
            return true;
        else
            return false;
    }

    /**
     * Checks if a given set of access flags matches a static member.
     * 
     * @param accessFlags
     *            the member's access flags
     * @return T if it's a static member, F otherwise
     */
    private boolean isStatic(int accessFlags)
    {
        // 8 is the static flag on the JVM. See the JVM Specs for more info.
        if ((accessFlags & 8) == 8)
            return true;
        else
            return false;
    }

    /**
     * Prepares the exception context from the attribute.
     * 
     * @param attribute
     *            the class object
     * @return the exception context
     */
    protected Map<String, Object> prepareExceptionContext(AttributeHolder attribute)
    {
        // Set exception context
        Map<String, Object> exceptionContext = new HashMap<String, Object>();

        exceptionContext.put(AnnotationMisuseException.ContextKeys.ANNOTATION_SCOPE,
                IncompatiblePrimaryAnnotationsException.ATTRIBUTE_SCOPE);
        exceptionContext.put(AnnotationMisuseException.ContextKeys.CLASS, attribute.getParentClassName());
        exceptionContext.put(AnnotationMisuseException.ContextKeys.METHOD, attribute.getName());

        try
        {

            List<String> annotationsFound = new ArrayList<String>();

            for (AnnotationHolder annotation: attribute.getAnnotations().values())
            {
                if (annotation.getMetaAnnotations().containsKey(Primary.class.getCanonicalName()))
                {
                    annotationsFound.add(annotation.getName());
                }
            }

            exceptionContext.put(AnnotationMisuseException.ContextKeys.ANNOTATIONS_FOUND, annotationsFound);

        }
        catch (ResourceNotFoundException resourceNotFoundException)
        {
            exceptionContext.put(AnnotationMisuseException.ContextKeys.ANNOTATIONS_FOUND, resourceNotFoundException);
        }

        return exceptionContext;
    }

    /**
     * Prepares the exception context from the class.
     * 
     * @param clazz
     *            the class object
     * @return the exception context
     */
    protected Map<String, Object> prepareExceptionContext(ClassHolder clazz)
    {
        // Set exception context
        Map<String, Object> exceptionContext = new HashMap<String, Object>();

        exceptionContext.put(AnnotationMisuseException.ContextKeys.ANNOTATION_SCOPE,
                IncompatiblePrimaryAnnotationsException.CLASS_SCOPE);
        exceptionContext.put(AnnotationMisuseException.ContextKeys.CLASS, clazz.getFQName());

        try
        {

            List<String> annotationsFound = new ArrayList<String>();

            for (AnnotationHolder annotation: clazz.getAnnotations().values())
            {
                if (annotation.getMetaAnnotations().containsKey(Primary.class.getCanonicalName()))
                {
                    annotationsFound.add(annotation.getName());
                }
            }

            exceptionContext.put(AnnotationMisuseException.ContextKeys.ANNOTATIONS_FOUND, annotationsFound);

        }
        catch (ResourceNotFoundException resourceNotFoundException)
        {
            exceptionContext.put(AnnotationMisuseException.ContextKeys.ANNOTATIONS_FOUND, resourceNotFoundException);
        }

        return exceptionContext;
    }

    /**
     * Prepares the exception context from the method.
     * 
     * @param method
     *            the class object
     * @return the exception context
     */
    protected Map<String, Object> prepareExceptionContext(MethodHolder method)
    {
        // Set exception context
        Map<String, Object> exceptionContext = new HashMap<String, Object>();

        exceptionContext.put(AnnotationMisuseException.ContextKeys.ANNOTATION_SCOPE,
                IncompatiblePrimaryAnnotationsException.METHOD_SCOPE);
        exceptionContext.put(AnnotationMisuseException.ContextKeys.CLASS, method.getParentClassName());
        exceptionContext.put(AnnotationMisuseException.ContextKeys.METHOD, method.getName());

        try
        {

            List<String> annotationsFound = new ArrayList<String>();

            for (AnnotationHolder annotation: method.getAnnotations().values())
            {
                if (annotation.getMetaAnnotations().containsKey(Primary.class.getCanonicalName()))
                {
                    annotationsFound.add(annotation.getName());
                }
            }

            exceptionContext.put(AnnotationMisuseException.ContextKeys.ANNOTATIONS_FOUND, annotationsFound);

        }
        catch (ResourceNotFoundException resourceNotFoundException)
        {
            exceptionContext.put(AnnotationMisuseException.ContextKeys.ANNOTATIONS_FOUND, resourceNotFoundException);
        }

        return exceptionContext;
    }

    /**
     * Sets the annotationName value.
     * 
     * @param newAnnotation
     *            the new annotationName value
     */
    public void setAnnotation(String newAnnotation)
    {
        this.annotationName = newAnnotation;
    }

    /**
     * Validates if the entities referred on the annotation parameters are valid on the DEM context.
     * 
     * @param annotationName
     *            the name of the annotation to validate
     * @param clazz
     *            the class holder
     * @throws ResourceNotFoundException
     *             if any resource needed for the validation can't be found
     */
    protected void validateDEMConsistency(String annotationName, ClassHolder clazz) throws ResourceNotFoundException
    {

        // Fetch class annotations
        Map<String, AnnotationHolder> classAnnotations = clazz.getAnnotations();

        AnnotationHolder theAnnotation = null;

        // @ApplicationDefinition annotation
        if (ApplicationDefinition.class.getCanonicalName().contains(annotationName)
                && classAnnotations.containsKey(ApplicationDefinition.class.getCanonicalName()))
        {

            theAnnotation = classAnnotations.get(ApplicationDefinition.class.getCanonicalName());

            String provider = theAnnotation.getMembers().get(PROVIDER_ATTRIBUTE_NAME).toString();

            if (DEMLoaderEntityRegistry.getProvider(provider.toLowerCase()) == null)
                warnOfEnityMisuseOnClass(clazz.getFQName(), provider);
        }
        // @ServiceDefinition annotation
        else if (ServiceDefinition.class.getCanonicalName().contains(annotationName)
                && classAnnotations.containsKey(ServiceDefinition.class.getCanonicalName()))
        {
            theAnnotation = classAnnotations.get(ServiceDefinition.class.getCanonicalName());

            String application = theAnnotation.getMembers().get(APPLICATION_ATTRIBUTE_NAME).toString();

            if (DEMLoaderEntityRegistry.getApplication(application.toLowerCase()) == null)
                warnOfEnityMisuseOnClass(clazz.getFQName(), application);
        }
        // @StageDefinition annotation
        else if (StageDefinition.class.getCanonicalName().contains(annotationName)
                && classAnnotations.containsKey(StageDefinition.class.getCanonicalName()))
        {
            theAnnotation = classAnnotations.get(StageDefinition.class.getCanonicalName());

            String service = theAnnotation.getMembers().get(SERVICE_ATTRIBUTE_NAME).toString();

            if (DEMLoaderEntityRegistry.getService(service.toLowerCase()) == null)
                warnOfEnityMisuseOnClass(clazz.getFQName(), service);
        }
    }

    /**
     * Validates if a given class message file exists. Only works if DiF is in DEBUG mode.
     * 
     * @param clazz
     *            the class to validate the messages *
     * @throws ResourceNotFoundException
     *             if any resource needed for the validation can't be found
     */
    protected void validateMessages(ClassHolder clazz) throws ResourceNotFoundException
    {

        // If DiF's in DEBUG...
        if (LogLevel.DEBUG.equals(DIFStartupConfiguration.getLogLevel()))
        {

            Entity entityType = null;
            String parentID = null;

            Map<String, AnnotationHolder> annotations = clazz.getAnnotations();

            // @ProviderDefinition annotation (does not have a parent, parentID will be null)
            if (annotations.containsKey(ProviderDefinition.class.getCanonicalName()))
            {
                entityType = Entity.PROVIDER;
            }
            // @ApplicationDefinition annotation
            else if (annotations.containsKey(ApplicationDefinition.class.getCanonicalName()))
            {
                entityType = Entity.APPLICATION;
                parentID = annotations.get(ApplicationDefinition.class.getCanonicalName()).getMembers()
                        .get(PROVIDER_ATTRIBUTE_NAME).getManagedAnnotationValue().toString();
            }
            // @ServiceDefinition annotation
            else if (annotations.containsKey(ServiceDefinition.class.getCanonicalName()))
            {
                entityType = Entity.SERVICE;
                parentID = annotations.get(ServiceDefinition.class.getCanonicalName()).getMembers()
                        .get(APPLICATION_ATTRIBUTE_NAME).getManagedAnnotationValue().toString();
            }
            // @StageDefinition annotation
            else if (annotations.containsKey(StageDefinition.class.getCanonicalName()))
            {
                entityType = Entity.STAGE;
                parentID = annotations.get(StageDefinition.class.getCanonicalName()).getMembers()
                        .get(SERVICE_ATTRIBUTE_NAME).getManagedAnnotationValue().toString();

            }

            // Remove the leading and trailing quotation marks
            if (parentID != null)
                parentID = parentID.substring(1, parentID.length() - 1);

            // Get the message manager...
            IMessageManager messageManager = DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class);

            // ...fetch messages for this class from the message manager
            MessageList messages = messageManager.collectEntityMessagesFromRepository(entityType, clazz.generateID(),
                    clazz.getFQName(), parentID);

            // If there are no messages, warn user..
            if (messages == null || messages.getMessageIDs().size() == 0)
            {
                UsageIssuesManagerImpl.getInstance().addIssue(IssueType.WARN, IssueScope.LOADTIME, clazz.getFQName(),
                        "Missing messages file!", null);
            }
        }
    }

    /**
     * Validates that the attribute is non-private.
     * 
     * @param attribute
     *            the attribute to validate
     * @return T if the attribute is non-private, F otherwise
     */
    protected boolean validateNonPrivate(AttributeHolder attribute)
    {

        // Should only validate DEM annotations
        if (isDEMAnnotation())
        {
            int accessFlags = attribute.getManagedAttribute().getModifiers();

            if (isPrivate(accessFlags))
                return false;
            else
                return true;

        }
        else
            return true;
    }

    /**
     * Validates the use of primary annotations on a given attribute.
     * 
     * @param attribute
     *            the attribute to search for primary annotations
     * @throws AnnotationMisuseException
     *             if an annotation is misused
     * @throws ResourceNotFoundException
     *             if any resource needed for validation could not be found.
     */
    protected void validatePrimary(AttributeHolder attribute) throws AnnotationMisuseException,
            ResourceNotFoundException
    {
        // If this annotation is primary...
        if (this.isPrimary())
        {

            // Find the FQN for the current annotation
            Set<String> annotationNames = attribute.getAnnotations().keySet();

            String thisAnnotationFQN = null;

            for (String annotationName: annotationNames)
            {
                if (annotationName.contains(this.annotationName))
                {
                    thisAnnotationFQN = annotationName;
                    break;
                }
            }

            // Fetch instance of current annotation
            AnnotationHolder thisAnnotation = new AnnotationHolder(attribute, attribute.getAnnotations()
                    .get(thisAnnotationFQN).getManagedAnnotation());

            // Get all the method's annotations...
            for (AnnotationHolder annotation: attribute.getAnnotations().values())
            {
                // Check use with the the present annotation
                if (!checkUseWith(annotation, thisAnnotation))
                    // If annotation is misused throw an exception
                    throw new IncompatiblePrimaryAnnotationsException(
                            IncompatiblePrimaryAnnotationsException.INCOMPATIBLE_PRIMARY_ANNOTATIONS_MESSAGE,
                            prepareExceptionContext(attribute));
            }
        }
    }

    /**
     * Validates the use of primary annotations on a given class.
     * 
     * @param clazz
     *            the class to search for primary annotations
     * @throws AnnotationMisuseException
     *             if an annotation is misused
     * @throws ResourceNotFoundException
     *             if any resource needed for validation could not be found.
     */
    protected void validatePrimary(ClassHolder clazz) throws AnnotationMisuseException, ResourceNotFoundException
    {
        // If this annotation is primary...
        if (this.isPrimary())
        {

            // Find the FQN for the current annotation
            Set<String> annotationNames = clazz.getAnnotations().keySet();

            String thisAnnotationFQN = null;

            for (String annotationName: annotationNames)
            {
                if (annotationName.contains(this.annotationName))
                {
                    thisAnnotationFQN = annotationName;
                    break;
                }
            }

            // Fetch instance of current annotation
            AnnotationHolder thisAnnotation = new AnnotationHolder(clazz, clazz.getAnnotations().get(thisAnnotationFQN)
                    .getManagedAnnotation());

            for (AnnotationHolder annotation: clazz.getAnnotations().values())
            {
                // Check use with the the present annotation
                if (!checkUseWith(annotation, thisAnnotation))
                    // If annotation is misused throw an exception
                    throw new IncompatiblePrimaryAnnotationsException(
                            IncompatiblePrimaryAnnotationsException.INCOMPATIBLE_PRIMARY_ANNOTATIONS_MESSAGE,
                            prepareExceptionContext(clazz));
            }
        }
    }

    /**
     * Validates the use of primary annotations on a given method.
     * 
     * @param method
     *            the method to search for primary annotations
     * @throws AnnotationMisuseException
     *             if an annotation is misused
     * @throws ResourceNotFoundException
     *             if any resource needed for validation could not be found.
     */
    protected void validatePrimary(MethodHolder method) throws AnnotationMisuseException, ResourceNotFoundException
    {
        // If this annotation is primary...
        if (this.isPrimary())
        {

            // Find the FQN for the current annotation
            Set<String> annotationNames = method.getAnnotations().keySet();

            String thisAnnotationFQN = null;

            for (String annotationName: annotationNames)
            {
                if (annotationName.contains(this.annotationName))
                {
                    thisAnnotationFQN = annotationName;
                    break;
                }
            }

            // Fetch instance of current annotation
            AnnotationHolder thisAnnotation = new AnnotationHolder(method, method.getAnnotations()
                    .get(thisAnnotationFQN).getManagedAnnotation());

            // Get all the method's annotations...
            for (AnnotationHolder annotation: method.getAnnotations().values())
            {
                // Check use with the the present annotation
                if (!checkUseWith(annotation, thisAnnotation))
                    // If annotation is misused throw an exception
                    throw new IncompatiblePrimaryAnnotationsException(
                            IncompatiblePrimaryAnnotationsException.INCOMPATIBLE_PRIMARY_ANNOTATIONS_MESSAGE,
                            prepareExceptionContext(method));
            }
        }
    }

    /**
     * Validates the usage scope of attribute annotations.
     * 
     * @param annotationName
     *            the name of the annotation to validate
     * @param attribute
     *            the attribute to validate
     * @return T if the scope is correct, F otherwise
     * @throws ResourceNotFoundException
     *             if any resource needed for the validation can't be found
     */
    protected boolean validateScope(String annotationName, AttributeHolder attribute) throws ResourceNotFoundException
    {

        Map<String, AnnotationHolder> attributeAnnotations = attribute.getAnnotations();

        // @Persist annotation
        if (Persist.class.getCanonicalName().contains(annotationName)
                && attributeAnnotations.containsKey(Persist.class.getCanonicalName()))
        {
            final String SCOPE_ID = "scope";
            // Check scope
            if ((attributeAnnotations.get(Persist.class.getCanonicalName())).getMembers().containsKey(SCOPE_ID))
            {
                AnnotationMemberValueHolder scope = attributeAnnotations.get(Persist.class.getName()).getMembers()
                        .get(SCOPE_ID);
                MemberValue scopeValue = scope.getManagedAnnotationValue();

                if (scopeValue.toString().contains(ParameterScope.STATIC.toString())
                        && !isStatic(attribute.getManagedAttribute().getModifiers()))
                {
                    UsageIssuesManagerImpl.getInstance().addIssue(
                            IssueType.WARN,
                            IssueScope.LOADTIME,
                            attribute.getParentClassName(),
                            "Using @" + Persist.class.getSimpleName() + " scope " + ParameterScope.STATIC
                                    + " on non-static attribute: " + attribute.getName(), null);
                    return false;
                }
            }
        }
        else if (FormConfigurable.class.getCanonicalName().contains(annotationName)
                && attributeAnnotations.containsKey(FormConfigurable.class.getCanonicalName()))
        {
            final String LINKED_FORM_ID = "linkToForm";

            // Check linkedToForm
            AnnotationMemberValueHolder linkedToForm = null;
            AnnotationHolder parameterAnnotation = attributeAnnotations.get(Parameter.class.getName());
            if (parameterAnnotation != null)
                linkedToForm = parameterAnnotation.getMembers().get(LINKED_FORM_ID);

            if (linkedToForm == null)
            {
                UsageIssuesManagerImpl.getInstance().addIssue(
                        IssueType.WARN,
                        IssueScope.LOADTIME,
                        attribute.getParentClassName(),
                        "Using @" + FormConfigurable.class.getSimpleName() + " for attribute: " + attribute.getName()
                                + ", that does not have a @" + Parameter.class.getSimpleName() + " with the "
                                + LINKED_FORM_ID + " attribute defined", null);
                return false;
            }
        }

        return true;
    }

    /**
     * Defines specific rules for applying the annotationName on a given attribute. Returns T by default.
     * 
     * @param attribute
     *            the annotated attribute
     * @param annotationName
     *            the annotationName
     * @return T if all conditions are verified, F otherwise
     * @throws ResourceNotFoundException
     */
    protected boolean validateSpecificRules(String annotationName, AttributeHolder attribute)
            throws ResourceNotFoundException
    {
        // TODO: Uncomment after bug #9863
        // validateViewExistence(attribute.getName(), attribute.getParentClassName(), attribute.getAnnotations());
        return true;
    }

    /**
     * Defines specific rules for applying the annotationName on a given class. Returns T by default.
     * 
     * @param annotationName
     *            the annotationName
     * @param clazz
     *            the annotated class
     * @return T if all conditions are verified, F otherwise
     * @throws ResourceNotFoundException
     *             if any needed resource for validation is not found
     */
    protected boolean validateSpecificRules(String annotationName, ClassHolder clazz) throws ResourceNotFoundException
    {
        // TODO: Uncomment after bug #9863
        // validateViewExistence(null, clazz.getFQName(), clazz.getAnnotations());

        if (annotationName.equals(ProviderDefinition.class.getSimpleName())
                || annotationName.equals(ApplicationDefinition.class.getSimpleName())
                || annotationName.equals(ServiceDefinition.class.getSimpleName())
                || annotationName.equals(StageDefinition.class.getSimpleName()))
        {
            validateMessages(clazz);
        }

        return true;
    }

    /**
     * Defines specific rules for applying the annotationName on a given method. Returns T by default.
     * 
     * @param method
     *            the annotated method
     * @param annotationName
     *            the annotationName
     * @return T if all conditions are verified, F otherwise
     * @throws ResourceNotFoundException
     */
    protected boolean validateSpecificRules(String annotationName, MethodHolder method)
            throws ResourceNotFoundException
    {
        return true;
    }

    /**
     * Validates the use of the annotation on a given attribute. Validation includes checking if multiple primary
     * annotations are used, if the scope is correct, if the annotation members provide information consistent with the
     * DEM and if any other specific rules apply.
     * 
     * @param attribute
     *            the attribute on which the annotationName is used
     * @throws AnnotationMisuseException
     *             if an inappropriate annotation usage was made
     * @throws ResourceNotFoundException
     *             if any resource needed for the validation can't be found
     */
    final public void validateUsage(AttributeHolder attribute) throws AnnotationMisuseException,
            ResourceNotFoundException
    {
        if (!validateNonPrivate(attribute))
        {
            throw new IllegalAnnotationUsage("Annotated attribute " + attribute.getName() + " on class "
                    + attribute.getParentClassName()
                    + " is private! (Annotations can only be applied on non-private members)",
                    prepareExceptionContext(attribute));
        }
        validatePrimary(attribute);
        validateScope(annotationName, attribute);
        validateSpecificRules(annotationName, attribute);
    }

    /**
     * Validates the use of the annotation on a given class. Validation includes checking if multiple primary
     * annotations are used, if the scope is correct, if the annotation members provide information consistent with the
     * DEM and if any other specific rules apply.
     * 
     * @param clazz
     *            the class on which the annotation is used
     * @throws AnnotationMisuseException
     *             if an inappropriate annotation usage was made
     * @throws ResourceNotFoundException
     *             if any resource needed for the validation can't be found
     */
    final public void validateUsage(ClassHolder clazz) throws AnnotationMisuseException, ResourceNotFoundException
    {
        validatePrimary(clazz);
        // validateScope(clazz);
        validateDEMConsistency(annotationName, clazz);
        validateSpecificRules(annotationName, clazz);
    }

    /**
     * Validates the use of the annotation on a given method. Validation includes checking if multiple primary
     * annotations are used, if the scope is correct and if any other specific rules apply.
     * 
     * @param method
     *            the method on which the annotation is used
     * @throws AnnotationMisuseException
     *             if an inappropriate annotation usage was made
     * @throws ResourceNotFoundException
     *             if any resource needed for the validation can't be found
     */
    final public void validateUsage(MethodHolder method) throws AnnotationMisuseException, ResourceNotFoundException
    {
        validatePrimary(method);
        // validateScope(method);
        validateSpecificRules(annotationName, method);
    }

    /**
     * Checks if the received set of annotations contains the
     * 
     * @View annotation and if the referenced view exists. Logs a warning if the view doesn't exist.
     * @param attributeName
     *            the name of the annotated attribute if it exists
     * @param className
     *            the name of the annotated class or the attribute's class name
     * @param annotations
     *            the annotation set to validate
     * @throws ResourceNotFoundException
     *             if any resource needed for the validation can't be found
     */
    protected void validateViewExistence(String attributeName, String className,
            Map<String, AnnotationHolder> annotations) throws ResourceNotFoundException
    {
        AnnotationHolder theAnnotation = null;

        if (View.class.getCanonicalName().contains(annotationName)
                && annotations.containsKey(View.class.getCanonicalName()))
        {
            theAnnotation = annotations.get(View.class.getCanonicalName());

            String target = theAnnotation.getMembers().get(TARGET_ATTRIBUTE_NAME).toString();

            URL view = this.getClass().getClassLoader().getResource(target);

            if (view == null)
            {
                if (attributeName != null)
                {
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.WARN, IssueScope.LOADTIME, className,
                            "Attribute " + attributeName + " refers to inexistant view: " + target, null);
                }
                else
                    UsageIssuesManagerImpl.getInstance().addIssue(IssueType.WARN, IssueScope.LOADTIME, className,
                            "Class refers to inexistant view: " + target, null);
            }
        }
    }

    /**
     * Warns of an entity misuse on a given class.
     * 
     * @param className
     *            the name of the class where the reference is being made
     * @param entityName
     *            the name of the referred entity
     */
    protected void warnOfEnityMisuseOnClass(String className, String entityName)
    {
        UsageIssuesManagerImpl.getInstance().addIssue(IssueType.WARN, IssueScope.LOADTIME, className,
                "Class refers to an inexistant DEM entity: " + entityName, null);
    }

    /**
     * Checks if a given primary annotation has the 'yieldTo' member value and if it contains a given annotation name.
     * 
     * @param annotation
     *            the primary annotation to check
     * @param yieldToAnnotationName
     *            the name on the 'yieldTo' value
     * @return T if the primary annotation yields to the passed name, F otherwise
     */
    protected boolean yieldsTo(AnnotationHolder annotation, String yieldToAnnotationName)
    {

        try
        {
            AnnotationHolder primaryMetaAnnotation = annotation.getMetaAnnotations().get(
                    Primary.class.getCanonicalName());

            if (primaryMetaAnnotation.getMembers().get(MetaAnnotationMemberTags.PRIMARY_YIELD_TO).toString()
                    .contains(yieldToAnnotationName))
            {
                return true;
            }

        }
        catch (ResourceNotFoundException resourceNotFoundException)
        {

        }

        return false;
    }
}
