/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects;

import java.text.ParseException;
import java.util.Date;

import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.DateUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.crypto.IEncryptor;
import pt.digitalis.utils.crypto.exeption.CryptoException;
import pt.digitalis.utils.crypto.impl.EncryptorBase64Impl;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Nov 1, 2007
 */
/**
 * This class will hold the information of a given registration. It allows the management of registrations including
 * creation and deletion. TODO: Candidate for another project.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Sep 26, 2007
 */
public class LicenseImpl implements ILicense {

    /** Active Encryptor implementation. */
    private static IEncryptor encryptor = null;

    /**
     * Full version date.
     */
    private static final String FULL_VERSION_DATE = "00000000";

    /** just a variable */
    private static final char[] hideInPlainSight = {(char) 97, (char) 115, (char) 100, (char) 97, (char) 97,
                        (char) 115, (char) 100, (char) 102, (char) 56, (char) 115, (char) 97, (char) 100, (char) 57,
                        (char) 102, (char) 56, (char) 55, (char) 97, (char) 115};

    /**
     * Registration key spliter.
     */
    private static final String SPLITER_REGISTRATION_KEY = "-&-";

    /** The variable to return */
    private static final String variableToReturn = new String(hideInPlainSight);

    /** The License Edition */
    private LicenseEditionType edition;

    /** The entity id */
    private String entityId;

    /** The licence expiration date */
    private Date expirationDate = null;

    /** The registration key */
    private String key;

    /** Use to see if the key as ben check once */
    private Boolean keyChecked = false;

    /** The name of the registration */
    private String name;

    /** True if it is possible to register this element */
    private boolean registrable = true;

    /** The result of key check */
    private boolean validKey = false;

    /**
     * Constructor for an unregistered element
     */
    public LicenseImpl()
    {}

    /**
     * Constructor for an unregistered element
     * 
     * @param name
     *            the name of the element
     */
    public LicenseImpl(String name)
    {
        this.name = name;
    }

    /**
     * Constructor for a registered element
     * 
     * @param name
     *            the name of the element
     * @param key
     *            the key of the registered element
     */
    public LicenseImpl(String name, String key)
    {
        this.name = name;
        this.key = key;
    }

    /**
     * @return the key validation
     */
    private boolean checkKey()
    {
        String client = getClientName();
        boolean keyOk = false;

        try
        {

            if (client == null || "".equals(client))
            {
                DIFLogger.getLogger().debug("The client name is not configured.");
                return false;
            }
            if (this.entityId == null)
            {
                DIFLogger.getLogger().debug("The entityID cannot be null.");
                return false;
            }

            String theClient = StringUtils.removeAccents(client);
            String theEntity = StringUtils.removeAccents(this.entityId);

            // Verifies if the key is null or doesn't has the two phases separator
            if (!simpleKeyCheck(this.key))
            {
                return false;
            }

            // The two Phases separator
            String[] str = this.key.split(SPLITER_REGISTRATION_KEY);

            // Phase One

            // Inverts the second step of the first phase
            String aux = getEncryptator().decrypt(str[0]);

            // Obtains the client which has register the application (first step of first phase)
            String clientEnc = getEncryptator().decrypt(aux, theEntity);

            // Verifies the client's authenticity
            keyOk = clientEnc.equals(theClient);

            // Second phase - If the first phase was wrong it will not be necessary process the second
            if (keyOk)
            {
                // Inverts the third step of second phase
                aux = getEncryptator().decrypt(str[1]);

                // Inverts the second step of second phase
                aux = getEncryptator().decrypt(aux, theEntity);

                // Obtains the registry key expiration date (first step of second phase)
                String dateEnc = getEncryptator().decrypt(aux, theClient);

                // Verify if is a trial version
                if (!dateEnc.equals(FULL_VERSION_DATE))
                {
                    try
                    {
                        this.expirationDate = DateUtils.stringToSimpleDate(dateEnc);
                        keyOk = !this.hasExpired();
                    }
                    catch (ParseException pe)
                    {
                        DIFLogger.getLogger().debug(pe.getMessage());
                        keyOk = false;
                    }
                }
            }
            // Third phase - Determine the edition
            this.edition = LicenseEditionType.STANDARD;
            if (keyOk && str.length > 2)
            {
                try
                {
                    // Inverts the third step of second phase
                    aux = getEncryptator().decrypt(str[2]);

                    // Inverts the second step of second phase
                    aux = getEncryptator().decrypt(aux, theEntity);

                    // Obtains the registry key exparicy date (first stpe of second phase)
                    String decryptEdition = getEncryptator().decrypt(aux, theClient);

                    if (LicenseEditionType.PREMIUM.name().equals(decryptEdition))
                    {
                        this.edition = LicenseEditionType.PREMIUM;
                    }
                }
                catch (CryptoException ce)
                {
                    keyOk = false;
                }
            }
            this.name = clientEnc;
        }
        catch (CryptoException ce)
        {
            DIFLogger.getLogger().debug(ce.getMessage());
            keyOk = false;
        }
        return keyOk;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#generateKey(java.lang.String, java.lang.String, java.lang.String,
     *      pt.digitalis.dif.dem.objects.LicenseEditionType)
     */
    public String generateKey(String client, String entity, String date, LicenseEditionType edition)
    {
        if (edition == null)
        {
            edition = LicenseEditionType.STANDARD;
        }

        if (date == null || "".equals(date.trim()))
        {
            date = FULL_VERSION_DATE;
        }
        String theClient;
        if (client == null || "".equals(client.trim()))
        {
            theClient = StringUtils.removeAccents(this.getClientName());
        }
        else
        {
            theClient = StringUtils.removeAccents(client);
        }

        String theApp = StringUtils.removeAccents(entity);

        String result = null;
        String phaseOne = null;
        String phaseTwo = null;
        String phaseThree = null;
        try
        {
            // Phase One

            // encrypts the institution(client) with the given application as key(seed)
            String aux = getEncryptator().encrypt(theClient, theApp);

            // encrypts the result with the internal seed
            phaseOne = getEncryptator().encrypt(aux);

            // Phase Two

            // Encrypts the date using the institution(Client) as base
            aux = getEncryptator().encrypt(date, theClient);
            // Encrypt the last result with the application as key(seed)
            aux = getEncryptator().encrypt(aux, theApp);
            // encriptar o resultado anterior com a key interna.
            phaseTwo = getEncryptator().encrypt(aux);

            // Encrypts the Edition using the institution(Client) as base
            aux = getEncryptator().encrypt(edition.name(), theClient);
            // Encrypt the last result with the application as key(seed)
            aux = getEncryptator().encrypt(aux, theApp);
            // encriptar o resultado anterior com a key interna.
            phaseThree = getEncryptator().encrypt(aux);

            // Concatenates the thow phases result with the separator
            result = phaseOne + SPLITER_REGISTRATION_KEY + phaseTwo + SPLITER_REGISTRATION_KEY + phaseThree;

            return result;
        }
        catch (CryptoException ce)
        {
            DIFLogger.getLogger().info(ce.getMessage());

            return null;
        }
    }

    /**
     * Get the Client Name
     * 
     * @return the Client Name
     */
    protected String getClientName()
    {
        return DIFGeneralConfigurationParameters.getInstance().getClient();
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#getEdition()
     */
    public LicenseEditionType getEdition()
    {

        LicenseEditionType result = null;
        if (isRegistered())
        {
            result = this.edition;
        }
        return result;
    }

    /**
     * Get the active encryptor instance.
     * 
     * @return the encryptor instance
     */
    private IEncryptor getEncryptator()
    {
        if (encryptor == null)
        {
            encryptor = new EncryptorBase64Impl();
            encryptor.setSeed(getSeed());
        }
        return encryptor;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#getExpirationDate()
     */
    public Date getExpirationDate()
    {
        return this.expirationDate;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#getKey()
     */
    public String getKey()
    {
        return key;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#getName()
     */
    public String getName()
    {
        return name;
    }

    /**
     * The defined default Seed for DIF.
     * 
     * @return the seed
     */
    protected String getSeed()
    {
        return variableToReturn;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#hasExpirationDate()
     */
    public boolean hasExpirationDate()
    {
        return this.getExpirationDate() != null;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#hasExpired()
     */
    public boolean hasExpired()
    {
        if (this.expirationDate == null)
            return false;

        Date dateCur = DateUtils.getTrimmedDate();
        // If the trial version expired will set an invalid key
        return (dateCur.after(this.expirationDate));
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#isRegistered()
     */
    public boolean isRegistered()
    {

        if (this.keyChecked)
        {
            if (this.hasExpirationDate())
            {
                return !this.hasExpired();
            }
            return this.isValidKey();
        }
        else
        {
            this.setValidKey(this.checkKey());
            this.keyChecked = isValidKey();
        }
        return isValidKey();
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#isRegistrable()
     */
    public boolean isRegistrable()
    {
        return registrable;
    }

    /**
     * Inspector for the 'validKey' attribute.
     * 
     * @return the validKey value
     */
    private boolean isValidKey()
    {
        return validKey;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#register(java.lang.String, java.lang.String)
     */
    public boolean register(String key, String entityId)
    {
        if (isRegistrable())
        {
            this.key = key;
            this.entityId = entityId;
            this.keyChecked = false;
            return isRegistered();
        }
        else
            return true;

    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#setName(java.lang.String)
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#setRegistrable(boolean)
     */
    public void setRegistrable(boolean registrable)
    {
        this.registrable = registrable;
    }

    /**
     * Modifier for the 'validKey' attribute.
     * 
     * @param validKey
     *            the new validKey value to set
     */
    private void setValidKey(boolean validKey)
    {
        this.validKey = validKey;
    }

    /**
     * Checks if the key has the correct format.
     * 
     * @param _regKeyToCheck
     *            the key to check
     * @return <code>true</code> if the key is not <code>null</code> and has the correct format
     */
    private boolean simpleKeyCheck(String _regKeyToCheck)
    {
        // Verifies if the key is null or doesn't has the two phases separator
        if (_regKeyToCheck == null || _regKeyToCheck.indexOf(SPLITER_REGISTRATION_KEY) == -1)
        {
            return false;
        }
        return true;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.ILicense#unregister()
     */
    public void unregister()
    {
        key = null;
        validKey = false;
        this.keyChecked = false;
        this.expirationDate = null;
    }

}
