/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dem.objects.messages;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;

/**
 * Represents a view message with language translations and persistence information
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @author Rodrigo Goncalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created Nov 2, 2007
 */
public class MessageList {

    /** The messages */
    private Map<String, MessageTranslations> messages = new CaseInsensitiveHashMap<MessageTranslations>();

    /**
     * Adds a message to the list. Uses the default language
     * 
     * @param id
     *            the message id
     * @param message
     *            the message
     */
    public void addMessage(String id, Message message)
    {
        addMessageTranslation(id, DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage(), message);
    }

    /**
     * Adds a message to the list. Note: It overrides the current translations if they exist.
     * 
     * @param id
     *            the message id
     * @param translations
     *            the translations list
     */
    public void addMessage(String id, MessageTranslations translations)
    {
        messages.put(id, translations);
    }

    /**
     * Adds a message to the list. Uses the default language and no custom message
     * 
     * @param id
     *            the message id
     * @param message
     *            the message
     */
    public void addMessage(String id, String message)
    {
        addMessage(id, new Message(message));
    }

    /**
     * Adds a message list to the list. All identical ids will override the current ones.
     * 
     * @param list
     *            the list of message translations to add
     */
    public void addMessageList(MessageList list)
    {
        for (String id: list.getMessageIDs())
        {
            messages.put(id, list.getMessageTranslations(id));
        }
    }

    /**
     * Adds a message translation to the list
     * 
     * @param id
     * @param language
     * @param message
     */
    public void addMessageTranslation(String id, String language, Message message)
    {
        MessageTranslations translations = messages.get(id);
        language = language.toLowerCase();

        if (translations == null)
        {
            translations = new MessageTranslations();
            translations.addTranslation(language, message);

            messages.put(id, translations);

        }
        else
            translations.addTranslation(language, message);
    }

    /**
     * Checks if a given message is contained on the list.
     * 
     * @param messageID
     *            the message ID
     * @return T if the message with the given ID is contained within the list, F otherwise
     */
    public boolean containsMessageWithID(String messageID)
    {
        return this.messages.containsKey(messageID);
    }

    /**
     * Gets the list of messages IDs
     * 
     * @return the message translations
     */
    public List<String> getMessageIDs()
    {
        return new ArrayList<String>(messages.keySet());
    }

    /**
     * Gets the list of messages and their default translation message
     * 
     * @return the message map
     */
    public Map<String, String> getMessages()
    {
        return getMessages(DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage());
    }

    /**
     * Gets the list of messages and their default message translation to the given language
     * 
     * @param language
     *            the language desired
     * @return the message map
     */
    public Map<String, String> getMessages(String language)
    {

        Map<String, String> messageValues = new CaseInsensitiveHashMap<String>();

        for (String id: getMessageIDs())
        {
            if ((messages.get(id) != null) && (messages.get(id).getMessage(language) != null))
                messageValues.put(id, messages.get(id).getMessage(language).getMessage());
            else
                messageValues.put(id, null);

        }

        return messageValues;
    }

    /**
     * Gets the translations for a given message
     * 
     * @param id
     *            the id of the message to search
     * @return the message translations for the given message id
     */
    public MessageTranslations getMessageTranslations(String id)
    {
        return messages.get(id);
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();

        formatter.addItem("Messages", messages);

        return formatter.getFormatedObject();
    }
}
