/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dem.objects.parameters.rules;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.dem.interfaces.ICustomFormDefinition;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.ICustomFormManager;
import pt.digitalis.dif.dem.objects.FeatureState;
import pt.digitalis.dif.dem.objects.FormFieldCustomization;
import pt.digitalis.dif.dem.objects.parameters.IEditableParameter;
import pt.digitalis.dif.dem.objects.parameters.IParameter;
import pt.digitalis.dif.dem.objects.parameters.types.AbstractParameter;
import pt.digitalis.dif.dem.objects.parameters.types.BooleanParameter;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;

/**
 * Dependent parameter rule. Validates that the parameter list can only be selected hen this parameter is filled.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @param <T>
 *            the parameter base type
 * @created Mar 31, 2009
 */
public class ParameterRuleDependent<T> extends AbstractParameterRule<T> {

    /**
     * @param stage
     * @param valid
     * @param initializationInProgress
     * @return The ParameterRuleResult
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    private ParameterRuleResult getErrorMessage(IStageInstance stage, boolean valid, boolean initializationInProgress)
    {
        List<IParameter> invalidParameters = new ArrayList<IParameter>();

        if (stage != null)
        {
            // Get custom form configuration if present
            ICustomFormDefinition customFormDef = null;
            FormFieldCustomization customFormFieldDef = null;
            String formSubmitName = (String) stage.getContext().getRequest()
                    .getParameter(HTTPConstants.FORM_SUBMIT_NAME);
            String formConfigID = (String) stage.getContext().getRequest()
                    .getParameter(HTTPConstants.FORM_SUBMIT__CONFIG_BUSINESS_ID);
            boolean hasCheckedForCustomForm = false;

            for (String parameterID: getParameters())
            {
                IParameter param = null;
                boolean skipValidation = false;

                try
                {
                    param = stage.getParameters().getAllAvailableParameters().getParameter(parameterID);

                    if (param == null)
                        throw new ParameterException("Parameter does not exist: " + parameterID, null, null);

                    boolean parameterIsFromForm = false;
                    String paramFormName = param.getFormLinked();

                    if (paramFormName != null && formSubmitName != null)
                        parameterIsFromForm = paramFormName.equals(formSubmitName);

                    // Form parameter that has a configuration active (as submitted by the form)
                    if (parameterIsFromForm && param.isFormConfigurable())
                    {
                        if (!hasCheckedForCustomForm)
                        {
                            customFormDef = DIFIoCRegistry.getRegistry().getImplementation(ICustomFormManager.class)
                                    .getConfiguration(stage, formConfigID);
                            hasCheckedForCustomForm = true;
                        }

                        if (customFormDef != null)
                        {
                            customFormFieldDef = customFormDef.getCustomizedParameters().get(
                                    param.getId().toLowerCase());

                            if (customFormFieldDef != null)
                            {
                                if (customFormFieldDef.getMandatory() != FeatureState.DEFAULT)
                                    ((IEditableParameter) param)
                                            .setRequired(customFormFieldDef.getMandatory() == FeatureState.ON);

                                if (customFormDef.getExcludedParameters().contains(param.getId()))
                                    skipValidation = true;
                            }
                        }
                    }

                    if (!skipValidation)
                    {
                        boolean isBoolean = (param instanceof BooleanParameter);

                        if (initializationInProgress)
                            param.refreshParameterValue(stage);

                        if (valid)
                        {
                            if (param.isRequired())
                            {
                                // Has required constraint, thus must be filled...
                                if ((!isBoolean && param.getValue(stage.getContext()) == null)
                                        || (isBoolean && !param.getValueAsBoolean(stage.getContext())))
                                    invalidParameters.add(param);
                            }
                        }
                        else
                        {
                            if ((!isBoolean && param.getValue(stage.getContext()) != null)
                                    || (isBoolean && param.getValueAsBoolean(stage.getContext())))
                                invalidParameters.add(param);
                        }
                    }

                }
                catch (ParameterException e)
                {
                    if (param != null)
                        invalidParameters.add(param);
                }
            }
        }

        if (stage == null || invalidParameters.isEmpty())
            return new ParameterRuleResult(true);
        else
        {
            StringBuffer buffer = new StringBuffer();

            if (invalidParameters.size() == 1)
                buffer.append(this.getMessages(stage).get("field") + " ");
            else
                buffer.append(this.getMessages(stage).get("fieldPlural") + " ");

            int count = 0;
            for (IParameter invalidParam: invalidParameters)
            {
                stage.getContext().getRequest().addParameter(invalidParam.getId(), null);
                invalidParam.setValue(null, stage, true);

                if (count > 0)
                    buffer.append(", ");

                buffer.append("\"" + getParameterName(stage, invalidParam.getId()) + "\"");
                count++;
            }

            if (valid)
            {
                // No need to validate fields since the dependent parameter passed the rule
                if (invalidParameters.size() == 1)
                    buffer.append(" " + this.getMessages(stage).get("mustBeFilledWhen") + " ");
                else
                    buffer.append(" " + this.getMessages(stage).get("mustBeFilledWhenPlural") + " ");

                buffer.append("\"" + getParameterName(stage, parameterID) + "\"");

                if (getValue() != null)
                    buffer.append(" " + this.getMessages(stage).get("is") + " "
                            + (super.getDescriptionValue() != null ? super.getDescriptionValue() : super.getValue()));
                else if (getStartValue() != null && getEndValue() != null)
                    buffer.append(" "
                            + this.getMessages(stage).get("isBetween")
                            + " "
                            + (super.getDescriptionStartValue() != null ? super.getDescriptionStartValue() : super
                                    .getStartValue().toString())
                            + " "
                            + this.getMessages(stage).get("and")
                            + " "
                            + (super.getDescriptionEndValue() != null ? super.getDescriptionEndValue() : super
                                    .getEndValue().toString()));
                return new ParameterRuleResult(buffer.toString());
            }
            else
            {
                if (invalidParameters.size() == 1)
                    buffer.append(" " + this.getMessages(stage).get("canBeFilledIf") + " ");
                else
                    buffer.append(" " + this.getMessages(stage).get("canBeFilledIfPlural") + " ");

                buffer.append("\"" + getParameterName(stage, parameterID) + "\"");

                if (getValue() != null)
                    buffer.append(" " + this.getMessages(stage).get("is") + " "
                            + (super.getDescriptionValue() != null ? super.getDescriptionValue() : super.getValue()));
                else if (getStartValue() != null && getEndValue() != null)
                    buffer.append(" "
                            + this.getMessages(stage).get("isBetween")
                            + " "
                            + (super.getDescriptionStartValue() != null ? super.getDescriptionStartValue() : super
                                    .getStartValue().toString())
                            + " "
                            + this.getMessages(stage).get("and")
                            + " "
                            + (super.getDescriptionEndValue() != null ? super.getDescriptionEndValue() : super
                                    .getEndValue().toString()));
                else
                    buffer.append(" " + this.getMessages(stage).get("isFilled"));

                return new ParameterRuleResult(buffer.toString());
            }
        }
    }

    /**
     * Gets the validation result for the present parameter
     * 
     * @param stage
     *            the current stage
     * @param value
     *            the value to validate
     * @param initializationInProgress
     *            T if called within the dif parameter initialization
     * @param parameter
     *            the parameter to validate
     * @return the validation rule validation result
     * @throws ParameterException
     *             if the parameter does not exist
     */
    public ParameterRuleResult getValidationResult(IStageInstance stage, T value, boolean initializationInProgress,
            AbstractParameter<T> parameter) throws ParameterException
    {
        // Calculate if the current value validates the rule parameters
        boolean valid = false;

        if (getValue() != null)
        {
            if (parameter instanceof BooleanParameter && value == null)
                valid = getValue().equals("false");
            else
            {
                if (getValue().contains(","))
                {
                    List<String> list = Arrays.asList(getValue().split(","));
                    valid = (value != null && list.contains(value.toString()));
                }
                else
                    valid = (value != null && getValue().equals(value.toString()));
            }
        }
        else if (getStartValue() != null && getEndValue() != null)
        {
            // TODO: Implement this: Comparison for any object?!?!?
            valid = value != null
                    && (getStartValue().compareTo(value.toString()) <= 0 && getEndValue().compareTo(value.toString()) >= 0);
        }
        else
            valid = (value != null);

        return getErrorMessage(stage, valid, initializationInProgress);

    }
}
