/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.mail;

import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import javax.activation.DataHandler;
import javax.activation.FileDataSource;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.util.ByteArrayDataSource;

import org.apache.commons.lang.StringEscapeUtils;

import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.dif.utils.logging.IErrorLogManager;
import pt.digitalis.utils.common.FileUtils;
import pt.digitalis.utils.common.StringUtils;

/**
 * The Class MailSender.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Feb 28, 2009
 */
public class MailSender {

    /**
     * The MailSender singleton instance
     */
    protected static MailSender instance;

    /**
     * Inspector for the 'instance' attribute.
     * 
     * @return the instance value
     */
    public static MailSender getInstance()
    {

        if (instance == null)
        {
            IMailConfiguration mailConfiguration = MailConfiguration.getInstance();

            instance = new MailSender(mailConfiguration);
        }

        return instance;
    }

    /**
     * Modifier for the 'instance' attribute.
     * 
     * @param instance
     *            the new instance value to set
     */
    public static void setInstance(MailSender instance)
    {
        MailSender.instance = instance;
    }

    /** Mail Configuration */
    private IMailConfiguration configuration = null;

    /** {@link IErrorLogManager} the error Manager */
    IErrorLogManager errorLogManager = DIFIoCRegistry.getRegistry().getImplementation(IErrorLogManager.class);

    /**
     * MailSender Constructor
     * 
     * @param configuration
     *            the {@link IMailConfiguration}
     */
    public MailSender(IMailConfiguration configuration)
    {
        this.configuration = configuration;
    }

    /**
     * Inspector for the 'configuration' attribute.
     * 
     * @return the configuration value
     */
    public IMailConfiguration getConfiguration()
    {
        return configuration;
    }

    /**
     * Creates a new mail server session.
     * 
     * @return a new Mail session
     */
    private Session getMailSession()
    {
        Properties props = new Properties();
        props.setProperty("mail.transport.protocol", "smtp");
        props.setProperty("mail.store.protocol", "pop3");
        props.setProperty("mail.host", configuration.getMailServer());

        if (configuration.getMailServerUsername() != null && !"".equals(configuration.getMailServerUsername()))
            props.setProperty("mail.user", configuration.getMailServerUsername());

        if (configuration.getMailServerPassword() != null && !"".equals(configuration.getMailServerPassword()))
            props.setProperty("mail.smtp.auth", "true");
        else
            props.setProperty("mail.smtp.auth", "false");

        props.setProperty("mail.from", configuration.getDefaultFromAddress());

        if (configuration.getMailServerPort() != 0)
            props.setProperty("mail.smtp.port", configuration.getMailServerPort() + "");

        props.put("mail.debug", configuration.getDebugEnabled());

        props.put("mail.smtp.ssl.enable", configuration.getUseSSL());

        if (configuration.getOriginalConfs() != null)
        {
            props.putAll(configuration.getOriginalConfs());
        }

        DIFLogger.getLogger().debug(props);

        Session mailSession = Session.getDefaultInstance(props);

        return mailSession;
    }

    /**
     * Send an email.
     * 
     * @param type
     *            the mail type
     * @param from
     *            the sender address
     * @param to
     *            the mail destination address field.
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @param attachments
     *            the mail attachments. Names will be server paths.
     * @param images
     *            the html embedded images. Names will be server paths.
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendEmail(MailType type, String from, String to, String cc, String bcc, String subject, String body,
            List<String> attachments, Map<String, String> images) throws AddressException, MessagingException
    {
        return sendEmail(type, from, to, cc, bcc, subject, body, attachments, images, null, false);
    }

    /**
     * Send an email.
     * 
     * @param type
     *            the mail type
     * @param from
     *            the sender address
     * @param to
     *            the mail destination address field.
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @param attachments
     *            the mail attachments. Names will be server paths.
     * @param images
     *            the html embedded images. Names will be server paths.
     * @param documentRepositoryEntries
     *            the document Repository entries
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendEmail(MailType type, String from, String to, String cc, String bcc, String subject, String body,
            List<String> attachments, Map<String, String> images,
            List<DocumentRepositoryEntry> documentRepositoryEntries) throws AddressException, MessagingException
    {
        return sendEmail(type, from, to, cc, bcc, subject, body, attachments, images, documentRepositoryEntries, false);
    }

    /**
     * Send an email.
     * 
     * @param type
     *            the mail type
     * @param from
     *            the sender address
     * @param to
     *            the mail destination address field.
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @param attachments
     *            the mail attachments. Names will be server paths.
     * @param images
     *            the html embedded images. Names will be server paths.
     * @param documentRepositoryEntries
     *            the document entries
     * @param disableErrorLog
     * @return T if the mail was sent successfully. F otherwise
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendEmail(MailType type, String from, String to, String cc, String bcc, String subject, String body,
            List<String> attachments, Map<String, String> images,
            List<DocumentRepositoryEntry> documentRepositoryEntries, boolean disableErrorLog) throws MessagingException
    {
        try
        {
            subject = StringEscapeUtils.unescapeJava(subject);
            body = StringEscapeUtils.unescapeJava(body);

            // Mail session
            Session session = getMailSession();

            // Create message object
            Message message = new MimeMessage(session);

            message.setRecipients(Message.RecipientType.TO, InternetAddress.parse(to, false));

            message.setSentDate(new Date());
            message.setSubject(subject);

            if (from != null)
                message.setFrom(InternetAddress.parse(from)[0]);
            else
                message.setFrom(InternetAddress.parse(configuration.getDefaultFromAddress())[0]);

            if (cc != null)
                message.setRecipients(Message.RecipientType.CC, InternetAddress.parse(cc, false));
            if (bcc != null)
                message.setRecipients(Message.RecipientType.BCC, InternetAddress.parse(bcc, false));

            if (attachments != null || images != null
                    || (documentRepositoryEntries != null && !documentRepositoryEntries.isEmpty()))
            {
                Multipart multipart = new MimeMultipart();

                // Set the email message body...
                MimeBodyPart messagePart = new MimeBodyPart();

                if (attachments != null)
                {
                    // Set the email attachment files..
                    for (String attach: attachments)
                    {
                        MimeBodyPart attachmentPart = new MimeBodyPart();
                        attachmentPart.setDataHandler(new DataHandler(new FileDataSource(attach)));
                        // attachmentPart.setFileName(attach);

                        multipart.addBodyPart(attachmentPart);
                    }
                }

                if (images != null)
                {
                    // Set the email attachment files..
                    for (Entry<String, String> image: images.entrySet())
                    {
                        MimeBodyPart embededImagePart = new MimeBodyPart();
                        embededImagePart.setDataHandler(new DataHandler(new FileDataSource(image.getValue())));
                        embededImagePart.setHeader("Content-ID", "<" + image.getKey() + ">");

                        multipart.addBodyPart(embededImagePart);
                    }
                }

                if (documentRepositoryEntries != null && !documentRepositoryEntries.isEmpty())
                {
                    for (DocumentRepositoryEntry documentRepositoryEntry: documentRepositoryEntries)
                    {
                        MimeBodyPart attachment = new MimeBodyPart();
                        attachment.setDataHandler(new DataHandler(new ByteArrayDataSource(documentRepositoryEntry
                                .getBytes(), FileUtils.getMimeType(documentRepositoryEntry.getMimeType()))));
                        attachment.setFileName(documentRepositoryEntry.getFileName());
                        multipart.addBodyPart(attachment);
                    }
                }

                if (MailType.HTML.equals(type))
                    messagePart.setContent(body,
                            "text/html"
                                    + (StringUtils.isBlank(configuration.getEncoding()) ? "" : "; charset="
                                            + configuration.getEncoding()));
                else
                    messagePart.setContent(body,
                            "text/plain"
                                    + (StringUtils.isBlank(configuration.getEncoding()) ? "" : "; charset="
                                            + configuration.getEncoding()));

                message.setContent(multipart);
                multipart.addBodyPart(messagePart);
            }
            else
            {
                if (MailType.HTML.equals(type))
                {
                    MimeMultipart multipart = new MimeMultipart();
                    MimeBodyPart messagePart;

                    // Set the email message HTML body...
                    messagePart = new MimeBodyPart();
                    messagePart.setContent(body,
                            "text/html"
                                    + (StringUtils.isBlank(configuration.getEncoding()) ? "" : "; charset="
                                            + configuration.getEncoding()));
                    multipart.addBodyPart(messagePart);

                    // TODO: Add the code to set a plan text alternative to the HTML content...
                    // // Set the email message plain text alternative body...
                    // messagePart = new MimeBodyPart();
                    // messagePart.setText(body);
                    // multipart.addBodyPart(messagePart);
                    // multipart.setSubType("alternative");

                    // if (!StringUtils.isBlank(configuration.getEncoding()))
                    // message.setContent(multipart, "charset=ISO-8859-1");
                    // else

                    message.setContent(multipart);
                }
                else
                {
                    message.setContent(body,
                            "text/plain"
                                    + (StringUtils.isBlank(configuration.getEncoding()) ? "" : "; charset="
                                            + configuration.getEncoding()));
                    message.setText(body);
                }
            }

            // Send message with or without authentication!
            Transport tr = session.getTransport("smtp");
            tr.connect(configuration.getMailServer(), configuration.getMailServerUsername(),
                    configuration.getMailServerPassword());
            message.saveChanges();
            tr.sendMessage(message, message.getAllRecipients());
            tr.close();

        }
        catch (AddressException e)
        {
            errorLogManager.logError(MailSender.class.getSimpleName(), "sendMail", e.getMessage());
            throw e;
        }
        catch (MessagingException e)
        {
            e.printStackTrace();
            errorLogManager.logError(MailSender.class.getSimpleName(), "sendMail", e.getMessage());
            throw e;
        }

        return false;
    }

    /**
     * Send an HTML text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendHTMLEmail(String to, String cc, String bcc, String subject, String body)
            throws AddressException, MessagingException
    {
        return sendHTMLEmail(to, cc, bcc, subject, body, (List<String>) null, (Map<String, String>) null);
    }

    /**
     * Send an HTML text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @param attachments
     *            the mail attachments. Names will be server paths.
     * @param images
     *            the html embedded images. Names will be server paths.
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendHTMLEmail(String to, String cc, String bcc, String subject, String body,
            List<String> attachments, Map<String, String> images) throws AddressException, MessagingException
    {
        return sendEmail(MailType.HTML, configuration.getDefaultFromAddress(), to, cc, bcc, subject, body, attachments,
                images);
    }

    /**
     * Send an HTML text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @param attachments
     *            the mail attachments. Names will be server paths and list should be comma separated
     * @param images
     *            the html embedded images. Names will be server paths. Format is: "id=imagePath,id2=imagePath"
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendHTMLEmail(String to, String cc, String bcc, String subject, String body, String attachments,
            String images) throws AddressException, MessagingException
    {
        List<String> attachList = null;
        Map<String, String> imageMap = null;

        if (attachments != null)
            attachList = Arrays.asList(attachments.split(","));

        if (images != null)
        {
            String[] imagePairs = images.split(";");

            for (String imagePair: imagePairs)
            {
                String[] info = imagePair.split("=");
                if (info.length == 2)
                {
                    if (imageMap == null)
                    {
                        imageMap = new HashMap<String, String>();
                    }

                    imageMap.put(info[0], info[1]);
                }
            }
        }

        return sendHTMLEmail(to, cc, bcc, subject, body, attachList, imageMap);
    }

    /**
     * Send an HTML text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @param attachments
     *            the mail attachments. Names will be server paths.
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendHTMLEmailWithAttachements(String to, String cc, String bcc, String subject, String body,
            List<String> attachments) throws AddressException, MessagingException
    {
        return sendHTMLEmail(to, cc, bcc, subject, body, attachments, null);
    }

    /**
     * Send an HTML text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @param attachments
     *            the mail attachments. Names will be server paths and list should be comma separated
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendHTMLEmailWithAttachements(String to, String cc, String bcc, String subject, String body,
            String attachments) throws AddressException, MessagingException
    {
        List<String> attachList = null;

        if (attachments != null)
            attachList = Arrays.asList(attachments.split(","));

        return sendHTMLEmailWithAttachements(to, cc, bcc, subject, body, attachList);
    }

    /**
     * Send an HTML text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @param images
     *            the html embedded images. Names will be server paths.
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendHTMLEmailWithImages(String to, String cc, String bcc, String subject, String body,
            Map<String, String> images) throws AddressException, MessagingException
    {
        return sendHTMLEmail(to, cc, bcc, subject, body, null, images);
    }

    /**
     * Send an HTML text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in HTML
     * @param images
     *            the html embedded images. Names will be server paths. Format is: "id=imagePath,id2=imagePath"
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendHTMLEmailWithImages(String to, String cc, String bcc, String subject, String body, String images)
            throws AddressException, MessagingException
    {
        List<String> imageList = null;

        if (images != null)
            imageList = Arrays.asList(images.split(","));

        return sendHTMLEmailWithAttachements(to, cc, bcc, subject, body, imageList);
    }

    /**
     * Send a plain text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in plain text
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendTextEmail(String to, String cc, String bcc, String subject, String body)
            throws AddressException, MessagingException
    {
        return sendTextEmail(to, cc, bcc, subject, body, (List<String>) null);
    }

    /**
     * Send a plain text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in plain text
     * @param attachments
     *            the mail attachments. Names will be server paths.
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendTextEmail(String to, String cc, String bcc, String subject, String body, List<String> attachments)
            throws AddressException, MessagingException
    {
        return sendEmail(MailType.PLAIN_TEXT, configuration.getDefaultFromAddress(), to, cc, bcc, subject, body,
                attachments, null);
    }

    /**
     * Send a plain text email.
     * 
     * @param to
     *            the mail destination address field
     * @param cc
     *            the mail CC address field.
     * @param bcc
     *            the mail BCC address field.
     * @param subject
     *            the mail subject
     * @param body
     *            the mail body, in plain text
     * @param attachments
     *            the mail attachments. Names will be server paths and list should be comma separated
     * @return T if the mail was sent successfully. F otherwise
     * @throws AddressException
     *             when an address is invalid
     * @throws MessagingException
     *             when an error occurs when sending the email
     */
    public boolean sendTextEmail(String to, String cc, String bcc, String subject, String body, String attachments)
            throws AddressException, MessagingException
    {
        List<String> attachList = null;

        if (attachments != null)
            attachList = Arrays.asList(attachments.split(","));

        return sendTextEmail(to, cc, bcc, subject, body, attachList);
    }

}
