/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.codegen.util;

import java.util.HashMap;
import java.util.Map;

import javassist.CannotCompileException;
import pt.digitalis.dif.codegen.CGAncillaries;
import pt.digitalis.dif.codegen.templates.ApplicationCGTemplate;
import pt.digitalis.dif.codegen.templates.ProviderCGTemplate;
import pt.digitalis.dif.codegen.templates.ServiceCGTemplate;
import pt.digitalis.dif.codegen.templates.StageCGTemplate;
import pt.digitalis.dif.codegen.templates.StageInstanceCGTemplate;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.exception.codegen.DIFCodeGenerationException;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.utils.CodeGenUtil4Javassist;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

/**
 * Holds the information needed to enhance a set of classes.
 * 
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2007/12/10
 */
public class ClassEnhancementContext {

    /** The initial map capacity. */
    final static private int INITIAL_CAPACITY = 2;

    /** The initial load factor. */
    final static private float LOAD_FACTOR = 1.0f;

    /** Holds the enhancements to do on the target classes. */
    /*
     * Implementation Note: this hash-map is optimised for the situation we're having here. At most, we will have two
     * objects to enhance (Stage). By forcing a load factor of 1.0 we're guaranteeing that no rehash (hence no memory
     * footprint or processing time elapsing) will occur. An initial capacity of 2 and a load factor of 0.75 (default
     * value) would yield a rehashing to size 4 after the first insertion (HashMap source code defines the rehashing
     * threshold as: threshold = (int) (initialCapacity * loadFactor);). With the above values we will have a threshold
     * of exactly 2, and rehashing will occur only when (++size > threshold), hence when the third element is added.
     */
    private Map<String, ClassEnhancements> classEnhancements = new HashMap<String, ClassEnhancements>(INITIAL_CAPACITY,
            LOAD_FACTOR);

    /** The class ID for the entity to be returned has the entity class */
    private String entityClassID;

    /** */
    private Entity entityType;

    /** The original annotated POJO. */
    private ClassHolder theOriginalClass;

    /**
     * Builds a class enhancement context from a class holder.
     * 
     * @param classHolder
     *            the class holder for the class to enhance
     * @throws ResourceNotFoundException
     *             if the class annotations can't be read
     * @throws CodeGenerationException
     *             if the passed class holder isn't a valid DEM entity or the class holder's target semantics is
     *             incorrect
     */
    public ClassEnhancementContext(ClassHolder classHolder) throws ResourceNotFoundException, CodeGenerationException
    {

        // Store original annotated POJO
        this.theOriginalClass = classHolder;

        // Infer entity type
        this.entityType = Entity.getEntityTypeFromClass(this.theOriginalClass);

        // If the entity is not a valid DEM entity throw an exception
        if (this.entityType == null)
            throw new CodeGenerationException(this.theOriginalClass.getFQName()
                    + " is not a valid DEM entity and thus cannot be enhanced! Aborting...");

        // Set enhancements accordingly to entity type

        // For stages there will be two enriched classes, the proxy and the instances
        if (entityType.equals(Entity.STAGE))
        {
            this.prepareStageEnhancement();
        }
        // For all other entities there will be just one enriched class
        else
        {
            this.prepareNonStageEnhancement();
        }
    }

    /**
     * Builds a class enhancement context from a class ID.
     * 
     * @param originalClassID
     *            the original annotated class
     * @throws ResourceNotFoundException
     *             if the class with the given id can't be found
     * @throws CodeGenerationException
     *             if the code can't be compiled
     */
    public ClassEnhancementContext(String originalClassID) throws ResourceNotFoundException, CodeGenerationException
    {
        this(new ClassHolder(originalClassID));
    }

    /**
     * Adds an enhancement to a given class method.
     * 
     * @param className
     *            the name of the class to add the enhancement to
     * @param enhancement
     *            the enhancement to add
     * @throws DIFCodeGenerationException
     */
    public void addEnhancement(String className, ClassMethodEnhancement enhancement) throws DIFCodeGenerationException
    {
        try
        {
            this.classEnhancements.get(className).addEnhancement(enhancement);
        }
        catch (Exception e)
        {
            DIFCodeGenerationException codeGenException = new DIFCodeGenerationException(
                    "Error adding enhancement to class", e);
            codeGenException.addToExceptionContext("Original Class Name", theOriginalClass.getFQName());
            codeGenException.addToExceptionContext("Entity ID", entityClassID);
            codeGenException.addToExceptionContext("Entity Type", entityType);
            codeGenException.addToExceptionContext("Class Name", className);
            codeGenException.addToExceptionContext("Enhancement", enhancement);

            throw codeGenException;
        }
    }

    /**
     * Adds an enhancement to a given class method. This method should be used for non-stage classes only since it does
     * provide means for specifying what class is to be enhanced
     * 
     * @param methodName
     *            the name of the method to add the enhancement to
     * @param enhancement
     *            the enhancement to add
     * @throws DIFCodeGenerationException
     */
    public void addEnhancement(String methodName, String enhancement) throws DIFCodeGenerationException
    {
        if (entityType == Entity.STAGE)
            addEnhancement(CGAncillaries.STAGE_PROXY_ID, methodName, enhancement);
        else
            addEnhancement(CGAncillaries.NON_STAGE_ENRICHED_CLASS_ID, methodName, enhancement);
    }

    /**
     * Adds an enhancement to a given class method.
     * 
     * @param className
     *            the name of the class to add the enhancement to
     * @param methodName
     *            the name of the method to add the enhancement to
     * @param enhancement
     *            the enhancement to add
     * @throws DIFCodeGenerationException
     */
    public void addEnhancement(String className, String methodName, String enhancement)
            throws DIFCodeGenerationException
    {
        try
        {
            this.classEnhancements.get(className).addSource(methodName, enhancement);
        }
        catch (Exception e)
        {
            DIFCodeGenerationException codeGenException = new DIFCodeGenerationException(
                    "Error adding enhancement to class", e);
            codeGenException.addToExceptionContext("Original Class Name", theOriginalClass.getFQName());
            codeGenException.addToExceptionContext("Entity ID", entityClassID);
            codeGenException.addToExceptionContext("Entity Type", entityType);
            codeGenException.addToExceptionContext("Class Name", className);
            codeGenException.addToExceptionContext("Method Name", methodName);
            codeGenException.addToExceptionContext("Enhancement", enhancement);

            throw codeGenException;
        }
    }

    /**
     * Adds an enhancement to a given class method.
     * 
     * @param className
     *            the name of the class to add the enhancement to
     * @param methodName
     *            the name of the method to add the enhancement to
     * @param terminator
     *            the finalize code for the method
     * @throws DIFCodeGenerationException
     */
    public void addEnhancementTerminatorCode(String className, String methodName, String terminator)
            throws DIFCodeGenerationException
    {
        try
        {
            this.classEnhancements.get(className).setTerminator(methodName, terminator);
        }
        catch (Exception e)
        {
            DIFCodeGenerationException codeGenException = new DIFCodeGenerationException(
                    "Error adding enhancement to class", e);
            codeGenException.addToExceptionContext("Original Class Name", theOriginalClass.getFQName());
            codeGenException.addToExceptionContext("Entity ID", entityClassID);
            codeGenException.addToExceptionContext("Entity Type", entityType);
            codeGenException.addToExceptionContext("Class Name", className);
            codeGenException.addToExceptionContext("Method Name", methodName);
            codeGenException.addToExceptionContext("Terminator", terminator);

            throw codeGenException;
        }
    }

    /**
     * Commits all the enhancements to a class file and returns the class holder object.
     * 
     * @throws ResourceNotFoundException
     *             if the class can't be found
     * @throws CodeGenerationException
     *             if the code can't be compiled
     */
    public void commitEnhancements() throws ResourceNotFoundException, CodeGenerationException
    {
        for (String className: this.classEnhancements.keySet())
        {
            try
            {
                this.classEnhancements.get(className).commitEnhancements();
            }
            catch (Exception e)
            {
                DIFCodeGenerationException codeGenException = new DIFCodeGenerationException(
                        "Error adding enhancement to class", e);
                codeGenException.addToExceptionContext("Original Class Name", theOriginalClass.getFQName());
                codeGenException.addToExceptionContext("Entity ID", entityClassID);
                codeGenException.addToExceptionContext("Entity Type", entityType);
                codeGenException.addToExceptionContext("Class Name", className);
            }
        }
    }

    /**
     * Inspects the enhancement map to see of the given method has been inserted for enhancement
     * 
     * @param methodName
     *            the name of the method to add the enhancement to
     * @return T if the method is present
     */
    public boolean containsEnhancement(String methodName)
    {
        if (entityType == Entity.STAGE)
            return containsEnhancement(CGAncillaries.STAGE_PROXY_ID, methodName);
        else
            return containsEnhancement(CGAncillaries.NON_STAGE_ENRICHED_CLASS_ID, methodName);
    }

    /**
     * Inspects the enhancement map to see of the given method has been inserted for enhancement
     * 
     * @param className
     *            the name of the class to add the enhancement to
     * @param methodName
     *            the name of the method to add the enhancement to
     * @return T if the method is present
     */
    public boolean containsEnhancement(String className, String methodName)
    {
        return this.classEnhancements.get(className).getMethodEnhancements().containsKey(methodName);
    }

    /**
     * Returns the class enhancements map.
     * 
     * @return the class enhancements map.
     */
    public Map<String, ClassEnhancements> getClassEnhancements()
    {
        return this.classEnhancements;
    }

    /**
     * Return the enriched entity class holder.
     * 
     * @return the class holder
     */
    public ClassHolder getEntityClass()
    {
        return classEnhancements.get(entityClassID).getClassObject();
    }

    /**
     * Return the original POJO class holder.
     * 
     * @return the class holder
     */
    public ClassHolder getOriginalClassObject()
    {
        return this.theOriginalClass;
    }

    /**
     * Returns the original class name.
     * 
     * @return the original class name
     */
    public String getOriginalClassName()
    {
        return this.theOriginalClass.getFQName();
    }

    /**
     * Prepare the class enhancement context for a non-stage entity.
     * 
     * @throws CodeGenerationException
     *             if the Entity is invalid
     * @throws ResourceNotFoundException
     *             if the template class can't be found
     */
    final private void prepareNonStageEnhancement() throws CodeGenerationException, ResourceNotFoundException
    {
        // Create entity class holder
        this.entityClassID = CGAncillaries.NON_STAGE_ENRICHED_CLASS_ID;
        ClassHolder enrichedClassHolder = CodeGenUtil4Javassist.createNewClassHolder(this.theOriginalClass.getFQName()
                + CGAncillaries.NON_STAGE_ENRICHED_CLASS_ID);
        try
        {
            // Extend stage instance from original stage
            enrichedClassHolder.setSuperClass(this.theOriginalClass.getFQName());
        }
        catch (CannotCompileException cannotCompileException)
        {
            throw new CodeGenerationException("Class " + this.theOriginalClass.getFQName()
                    + " already extends from another class! Unable to compile...", cannotCompileException);
        }

        // Copy methods from the appropriate template
        switch (entityType)
        {
            case VALIDATOR:
                // No ValidatorCGTemplate class defined in this release
                // enrichedClassHolder.copyAllFromClass(new ClassHolder(ValidatorCGTemplate.class.getCanonicalName()));
                break;
            case PROVIDER:
                enrichedClassHolder.copyAllFromClass(new ClassHolder(ProviderCGTemplate.class.getCanonicalName()));
                break;
            case APPLICATION:
                enrichedClassHolder.copyAllFromClass(new ClassHolder(ApplicationCGTemplate.class.getCanonicalName()));
                break;
            case SERVICE:
                enrichedClassHolder.copyAllFromClass(new ClassHolder(ServiceCGTemplate.class.getCanonicalName()));
                break;
            default:
                throw new CodeGenerationException("Invalid entity type: " + this.theOriginalClass.getFQName()
                        + "!! Aborting...");
        }

        // Create class enhancements object
        ClassEnhancements enrichedClassEnhancements = new ClassEnhancements(enrichedClassHolder);

        // Add class to class enhancement maps
        classEnhancements.put(CGAncillaries.NON_STAGE_ENRICHED_CLASS_ID, enrichedClassEnhancements);
    }

    /**
     * Prepare the class enhancement context for a stage entity.
     * 
     * @throws ResourceNotFoundException
     *             if any class to copy or to extend from can't be found
     * @throws CodeGenerationException
     *             if the original stage POJO already inherits from another class
     */
    final private void prepareStageEnhancement() throws CodeGenerationException, ResourceNotFoundException
    {
        // Create proxy class holder
        this.entityClassID = CGAncillaries.STAGE_PROXY_ID;
        ClassHolder stageProxyClassHolder = CodeGenUtil4Javassist.createNewClassHolder(this.theOriginalClass
                .getFQName() + CGAncillaries.STAGE_PROXY_ID);
        // Copy StageCGTemplate
        stageProxyClassHolder.copyAllFromClass(new ClassHolder(StageCGTemplate.class.getCanonicalName()));
        // Create instance class holder
        ClassEnhancements stageProxyClassEnhancements = new ClassEnhancements(stageProxyClassHolder);

        // Create stage instance class holder
        ClassHolder stageInstanceClassHolder = CodeGenUtil4Javassist.createNewClassHolder(this.theOriginalClass
                .getFQName() + CGAncillaries.STAGE_INSTANCE_ID);

        try
        {
            // Extend stage instance from original stage
            stageInstanceClassHolder.setSuperClass(this.theOriginalClass.getFQName());
        }
        catch (CannotCompileException cannotCompileException)
        {
            throw new CodeGenerationException("Class " + this.theOriginalClass.getFQName()
                    + " already extends from another class! Unable to compile...", cannotCompileException);
        }
        // Copy StageInstanceCGTemplate
        stageInstanceClassHolder.copyAllFromClass(new ClassHolder(StageInstanceCGTemplate.class.getCanonicalName()));

        // Create class enhancements object
        ClassEnhancements stageInstanceClassEnhancements = new ClassEnhancements(stageInstanceClassHolder);

        // Stage proxys have list builder methods. They will be incremental.
        stageProxyClassEnhancements.registerMethodAsIncremental(CGAncillaries.STAGE_INJECTED_VIEWS_BUILDER);
        stageProxyClassEnhancements.registerMethodAsIncremental(CGAncillaries.STAGE_INJECTED_ERRORVIEWS_BUILDER);
        stageProxyClassEnhancements.registerMethodAsIncremental(CGAncillaries.STAGE_INJECTED_STAGES_BUILDER);
        stageProxyClassEnhancements.registerMethodAsIncremental(CGAncillaries.STAGE_INJECTED_ERRORSTAGES_BUILDER);
        stageProxyClassEnhancements.registerMethodAsIncremental(CGAncillaries.STAGE_EVENT_HANDLERS_BUILDER);

        // Stage instances will have initialization and finalization methods. All will be treated as incremental
        // methods, as such they must be registered.
        stageInstanceClassEnhancements
                .registerMethodAsIncremental(CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME);
        stageInstanceClassEnhancements.registerMethodAsIncremental(CGAncillaries.STAGE_POSTPROCESSING_METHOD_NAME);
        stageInstanceClassEnhancements.registerMethodAsIncremental(CGAncillaries.CALL_EXEC_ONEVENT_METHOD);
        stageInstanceClassEnhancements.registerMethodAsIncremental(CGAncillaries.CALL_EVENT_METHOD);

        // Add classes to class enhancement maps
        classEnhancements.put(CGAncillaries.STAGE_PROXY_ID, stageProxyClassEnhancements);
        classEnhancements.put(CGAncillaries.STAGE_INSTANCE_ID, stageInstanceClassEnhancements);
    }

    /**
     * Adds an enhancement to a given class method.
     * 
     * @param className
     *            the name of the class to add the enhancement to
     * @param methodName
     *            the name of the method to add the enhancement to
     * @param enhancement
     *            the enhancement to add
     */
    public void setEnhancements(String className, String methodName, String enhancement)
    {
        this.classEnhancements.get(className).addSource(methodName, enhancement);
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();

        formatter.addItem("Original Class Name", this.getOriginalClassName());
        formatter.addItem("Entity Class Type", this.entityType);
        formatter.addItem("Entity Class Name", this.entityClassID);
        formatter.addItem("Classes", this.classEnhancements);

        return formatter.getFormatedObject();
    }
}
