/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.codegen.util;

import java.util.List;
import java.util.Map;

import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.dem.config.IDEMRegistrator;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.bytecode.holders.AnnotationHolder;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

/**
 * A temporary registry of DEM entities that helps the codegen utility to track down all entities on a first pass
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Jul 23, 2007
 */
final public class DEMLoaderEntityRegistry {

    /** Map of application IDs and their class objects */
    static private Map<String, ClassHolder> applications = new CaseInsensitiveHashMap<ClassHolder>();

    /** Map of provider IDs and their class objects */
    static private Map<String, ClassHolder> providers = new CaseInsensitiveHashMap<ClassHolder>();

    /** Map of service IDs and their class objects */
    static private Map<String, ClassHolder> services = new CaseInsensitiveHashMap<ClassHolder>();

    /** Map of stage IDs and their class objects */
    static private Map<String, ClassHolder> stages = new CaseInsensitiveHashMap<ClassHolder>();

    /** Map of validator IDs and their class objects */
    static private Map<String, ClassHolder> validators = new CaseInsensitiveHashMap<ClassHolder>();

    /**
     * Adds a application class to the temp DEM registry
     * 
     * @param id
     *            the ID of the DEM Entity
     * @param clazz
     *            the class object that represents the Entity
     */
    static public void addApplication(String id, ClassHolder clazz)
    {
        applications.put(id.toLowerCase(), clazz);
    }

    /**
     * Adds a provider class to the temp DEM registry
     * 
     * @param id
     *            the ID of the DEM Entity
     * @param clazz
     *            the class object that represents the Entity
     */
    static public void addProvider(String id, ClassHolder clazz)
    {
        providers.put(id.toLowerCase(), clazz);
    }

    /**
     * Adds a service class to the temp DEM registry
     * 
     * @param id
     *            the ID of the DEM Entity
     * @param clazz
     *            the class object that represents the Entity
     */
    static public void addService(String id, ClassHolder clazz)
    {
        services.put(id.toLowerCase(), clazz);
    }

    /**
     * Adds a stage class to the temp DEM registry
     * 
     * @param id
     *            the ID of the DEM Entity
     * @param clazz
     *            the class object that represents the Entity
     */
    static public void addStage(String id, ClassHolder clazz)
    {
        stages.put(id.toLowerCase(), clazz);
    }

    /**
     * Adds a validator class to the temp DEM registry
     * 
     * @param id
     *            the ID of the DEM Entity
     * @param clazz
     *            the class object that represents the Entity
     */
    static public void addValidator(String id, ClassHolder clazz)
    {
        validators.put(id.toLowerCase(), clazz);
    }

    /**
     * Performs memory clean up by releasing the internal Maps of entities when they are no longer needed. Must be
     * called since this is a static class with static attributes for temporary data.
     */
    static public void cleanUp()
    {

        // Detach all classes...
        if (providers != null)
            for (ClassHolder clazz: providers.values())
                clazz.cleanUp();

        if (applications != null)
            for (ClassHolder clazz: applications.values())
                clazz.cleanUp();

        if (services != null)
            for (ClassHolder clazz: services.values())
                clazz.cleanUp();

        if (stages != null)
            for (ClassHolder clazz: stages.values())
                clazz.cleanUp();

        if (validators != null)
            for (ClassHolder clazz: validators.values())
                clazz.cleanUp();

        providers = new CaseInsensitiveHashMap<ClassHolder>();
        applications = new CaseInsensitiveHashMap<ClassHolder>();
        services = new CaseInsensitiveHashMap<ClassHolder>();
        stages = new CaseInsensitiveHashMap<ClassHolder>();
        validators = new CaseInsensitiveHashMap<ClassHolder>();
    }

    /**
     * Searches the applications for the entry with the given ID
     * 
     * @param id
     *            the ID of the Entity to search
     * @return the class object of the Entity
     */
    static public ClassHolder getApplication(String id)
    {
        if (applications == null)
            return null;
        else
            return applications.get(id.toLowerCase());
    }

    /**
     * Returns the registered applications count
     * 
     * @return the number of Applications
     */
    static public int getApplicationCount()
    {
        return (applications == null) ? 0 : applications.size();
    }

    /**
     * Returns the Map of registered applications with the ID and their class objects
     * 
     * @return the Applications Map
     */
    static public Map<String, ClassHolder> getApplications()
    {
        return applications;
    }

    /**
     * Returns the total number of entities registered
     * 
     * @return the entity count
     */
    static public int getEntityCount()
    {

        int entityTotal = 0;

        entityTotal += DEMLoaderEntityRegistry.getProviderCount();
        entityTotal += DEMLoaderEntityRegistry.getApplicationCount();
        entityTotal += DEMLoaderEntityRegistry.getServiceCount();
        entityTotal += DEMLoaderEntityRegistry.getStageCount();
        entityTotal += DEMLoaderEntityRegistry.getValidatorCount();

        return entityTotal;
    }

    /**
     * Searches the providers for the entry with the given ID
     * 
     * @param id
     *            the ID of the Entity to search
     * @return the class object of the Entity
     */
    static public ClassHolder getProvider(String id)
    {
        if (providers == null)
            return null;
        else
            return providers.get(id.toLowerCase());
    }

    /**
     * Returns the registered providers count
     * 
     * @return the number of Providers
     */
    static public int getProviderCount()
    {
        return (providers == null) ? 0 : providers.size();
    }

    /**
     * Returns the Map of registered providers with the ID and their class objects
     * 
     * @return the Providers Map
     */
    static public Map<String, ClassHolder> getProviders()
    {
        return providers;
    }

    /**
     * Searches the services for the entry with the given ID
     * 
     * @param id
     *            the ID of the Entity to search
     * @return the class object of the Entity
     */
    static public ClassHolder getService(String id)
    {
        if (services == null)
            return null;
        else
            return services.get(id.toLowerCase());
    }

    /**
     * Returns the registered services count
     * 
     * @return the number of Services
     */
    static public int getServiceCount()
    {
        return (services == null) ? 0 : services.size();
    }

    /**
     * Returns the Map of registered services with the ID and their class objects
     * 
     * @return the Services Map
     */
    static public Map<String, ClassHolder> getServices()
    {
        return services;
    }

    /**
     * Searches the stages for the entry with the given ID
     * 
     * @param id
     *            the ID of the Entity to search
     * @return the class object of the Entity
     */
    static public ClassHolder getStage(String id)
    {
        if (stages == null)
            return null;
        else
            return stages.get(id.toLowerCase());
    }

    /**
     * Returns the registered stages count
     * 
     * @return the number of Stages
     */
    static public int getStageCount()
    {
        return (stages == null) ? 0 : stages.size();
    }

    /**
     * Returns the Map of registered stages with the ID and their class objects
     * 
     * @return the Stages Map
     */
    static public Map<String, ClassHolder> getStages()
    {
        return stages;
    }

    /**
     * Searches the validators for the entry with the given ID
     * 
     * @param id
     *            the ID of the Entity to search
     * @return the class object of the Entity
     */
    static public ClassHolder getValidator(String id)
    {
        if (validators == null)
            return null;
        else
            return validators.get(id.toLowerCase());
    }

    /**
     * Returns the registered validators count
     * 
     * @return the number if Validators
     */
    static public int getValidatorCount()
    {
        return (validators == null) ? 0 : validators.size();
    }

    /**
     * Returns the Map of registered validators with the ID and their class objects
     * 
     * @return the Validators Map
     */
    static public Map<String, ClassHolder> getValidators()
    {
        return validators;
    }

    /**
     * Analyzes the given classes to determine witch of them are DEM Entity. It adds them to the corresponding DEM list.
     * All read operations are executed by Javassist so classes are not loaded by the JVM.
     * 
     * @param classes
     *            the classes to analyze
     * @param demRegistrator
     *            the DEM registrator
     * @throws ResourceNotFoundException
     *             if any class could not be found or an annotation can't be read
     * @throws IllegalAccessException
     * @throws InstantiationException
     * @throws ClassNotFoundException
     */
    static public void loadEntityClasses(List<ClassHolder> classes, IDEMRegistrator demRegistrator)
            throws ResourceNotFoundException, ClassNotFoundException, InstantiationException, IllegalAccessException
    {

        // For all defined classes...
        for (ClassHolder clazz: classes)
        {
            /*
             * Treat class according the annotation. Implementation Note: The algorithm just considers the first primary
             * DEM-Entity Annotation Entity found.
             */

            // If it's a @ValidationDefinition
            if (clazz.containsAnnotation(Entity.VALIDATOR.getFullyQualifiedName()))
            {
                if (!demRegistrator.getEntitiesToExclude(Entity.VALIDATOR).contains(
                        clazz.getFQName()))
                    processValidator(clazz);
                else
                {
                    DIFLogger.getLogger().info(
                            "The Validator '" + clazz.getClassInstance().getClass().getCanonicalName()
                                    + "' was excluded from load process");
                }
            }
            // If it's a @ProviderDefinition
            else if (clazz.containsAnnotation(Entity.PROVIDER.getFullyQualifiedName()))
            {
                if (!demRegistrator.getEntitiesToExclude(Entity.PROVIDER).contains(
                        clazz.getFQName()))
                    processProvider(clazz);
                else
                {
                    DIFLogger.getLogger().info(
                            "The Provider '" + clazz.getClassInstance().getClass().getCanonicalName()
                                    + "' was excluded from load process");
                }
            }

            // If it's a @ApplicationDefinition
            else if (clazz.containsAnnotation(Entity.APPLICATION.getFullyQualifiedName()))
            {
                if (!demRegistrator.getEntitiesToExclude(Entity.APPLICATION).contains(
                        clazz.getFQName()))
                    processApplication(clazz);
                else
                {
                    DIFLogger.getLogger().info(
                            "The Application '" + clazz.getClassInstance().getClass().getCanonicalName()
                                    + "' was excluded from load process");
                }
            }

            // If it's a @ServiceDefinition
            else if (clazz.containsAnnotation(Entity.SERVICE.getFullyQualifiedName()))
            {
                if (!demRegistrator.getEntitiesToExclude(Entity.SERVICE).contains(
                        clazz.getFQName()))
                    processService(clazz);
                else
                {
                    DIFLogger.getLogger().info(
                            "The Service '" + clazz.getClassInstance().getClass().getCanonicalName()
                                    + "' was excluded from load process");
                }
            }

            // If it's a @StageDefinition
            else if (clazz.containsAnnotation(Entity.STAGE.getFullyQualifiedName()))
            {
                if (!demRegistrator.getEntitiesToExclude(Entity.STAGE).contains(
                        clazz.getFQName()))
                    processStage(clazz);
                else
                {
                    DIFLogger.getLogger().info(
                            "The Stage '" + clazz.getClassInstance().getClass().getCanonicalName()
                                    + "' was excluded from load process");
                }
            }
        }
    }

    /**
     * Process the <code>@ProviderDefinition</code> annotation, Reads it's member values and
     * 
     * @param clazz
     *            the Provider class
     * @throws ResourceNotFoundException
     *             if annotation member values can't be read
     */
    static private void processApplication(ClassHolder clazz) throws ResourceNotFoundException
    {

        // Add application to DEMLoaderEntityRegistry
        addApplication(processEntity(clazz, Entity.APPLICATION), clazz);
    }

    /**
     * Process the entity annotation.
     * 
     * @param clazz
     *            the Entity class
     * @param entity
     *            the entity to process
     * @return the processed Entity ID
     * @throws ResourceNotFoundException
     *             if annotation member values can't be read
     */
    static private String processEntity(ClassHolder clazz, Entity entity) throws ResourceNotFoundException
    {
        // Get the Annotation
        AnnotationHolder entityAnnotation = clazz.getAnnotations().get(entity.getFullyQualifiedName());

        if (entityAnnotation == null)
            throw new ResourceNotFoundException("Could not get @" + entity.getName()
                    + " annotation instance! Unable to proceed...");

        String entityID = entityAnnotation.getMembers().get("id").toString();

        if (AnnotationTags.GENERATE_ID.equals(entityID))
            entityID = clazz.generateID();

        return entityID;
    }

    /**
     * Process the <code>@ProviderDefinition</code> annotation.
     * 
     * @param clazz
     *            the Provider class
     * @throws ResourceNotFoundException
     *             if annotation member values can't be read
     */
    static private void processProvider(ClassHolder clazz) throws ResourceNotFoundException
    {

        // Add provider to DEMLoaderEntityRegistry
        addProvider(processEntity(clazz, Entity.PROVIDER), clazz);
    }

    /**
     * Process the <code>@ServiceDefinition</code> annotation.
     * 
     * @param clazz
     *            the Service class
     * @throws ResourceNotFoundException
     *             if annotation member values can't be read
     */
    static private void processService(ClassHolder clazz) throws ResourceNotFoundException
    {

        // Add service to DEMLoaderEntityRegistry
        addService(processEntity(clazz, Entity.SERVICE), clazz);
    }

    /**
     * Process the <code>@StageDefinition</code> annotation.
     * 
     * @param clazz
     *            the Stage class
     * @throws ResourceNotFoundException
     *             if annotation member values can't be read
     */
    static private void processStage(ClassHolder clazz) throws ResourceNotFoundException
    {

        // Add stage to DEMLoaderEntityRegistry
        addStage(processEntity(clazz, Entity.STAGE), clazz);
    }

    /**
     * Process the <code>@ValidatorDefinition</code> annotation.
     * 
     * @param clazz
     *            the Validator class
     * @throws ResourceNotFoundException
     *             if annotation member values can't be read
     */
    static private void processValidator(ClassHolder clazz) throws ResourceNotFoundException
    {

        // Add validator to DEMLoaderEntityRegistry
        addValidator(processEntity(clazz, Entity.VALIDATOR), clazz);
    }
}