/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.objects;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import pt.digitalis.dif.controller.interfaces.INavigationHistory;
import pt.digitalis.dif.controller.interfaces.IPrivateDIFSession;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.ObjectFormatter;

/**
 * Base implementation of a DIF Session. Should be extended by each specific channel session. (i.e. DIFSessionHTTPImpl)
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Nov 29, 2007
 */
public class DIFSession implements IPrivateDIFSession {

    /** The session key indicating that authentication was performed by a remote provider */
    public static final String REMOTE_AUTHENTICATION_PROVIDER_LOGIN = "remote_authentication_provider_login";

    /** The session key indicating that authentication was performed by a remote provider */
    public static final String REMOTE_AUTHENTICATION_PROVIDER_LOGOUT = "remote_authentication_provider_logout";

    /** Session attributes. */
    private Map<String, Object> attributes = new HashMap<String, Object>();

    /** Keeps record of session creation time. */
    private long firstAccessTime = 0;

    /** The current session navigation history */
    private INavigationHistory history = null;

    /** The current session user language */
    private String language = null;

    /** Keeps record of last session access. */
    private long lastAccessTime = 0;

    /** If the session been marked for removal */
    private Boolean markedForRemoval = false;

    /** The session unique ID */
    private final String sessionID;

    /** The defined time out value for the session. */
    private long sessionTimeOut = 0;

    /** The user associated to the session. */
    private DIFUserInSession user;

    /**
     * Default constructor. Sets the first access time.
     * 
     * @param sessionID
     *            the session unique identifier
     */
    public DIFSession(String sessionID)
    {
        this.firstAccessTime = System.currentTimeMillis();
        this.lastAccessTime = System.currentTimeMillis();
        this.sessionID = sessionID;
    }

    /**
     * One-argument constructor that receives an user.
     * 
     * @param sessionID
     *            the session unique identifier
     * @param user
     *            a DIF user
     */
    public DIFSession(String sessionID, DIFUserInSession user)
    {
        this(sessionID);
        this.user = user;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#addAttribute(java.lang.String, java.lang.Object)
     */
    public void addAttribute(String key, Object value)
    {
        attributes.put(key, value);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#containsAttribute(java.lang.String)
     */
    public boolean containsAttribute(String key)
    {
        return attributes.containsKey(key);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IPrivateDIFSession#forceKeepAlive()
     */
    public void forceKeepAlive()
    {
        this.setLastAccessTime(System.currentTimeMillis());

    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getAttribute(java.lang.String)
     */
    public Object getAttribute(String key)
    {
        return attributes.get(key);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getAttributes()
     */
    public Map<String, Object> getAttributes()
    {
        return attributes;
    }

    /**
     * Inspector for the 'firstAccessTime' property.
     * 
     * @return the 'firstAccessTime' value
     */
    public long getFirstAccessTime()
    {
        return this.firstAccessTime;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getLanguage()
     */
    public String getLanguage()
    {
        return language;
    }

    /**
     * Inspector for the 'lastAccessTime' property.
     * 
     * @return the 'lastAccessTime' value
     */
    public long getLastAccessTime()
    {
        return this.lastAccessTime;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getNavigationHistory()
     */
    public INavigationHistory getNavigationHistory()
    {
        if (history == null)
            history = DIFIoCRegistry.getRegistry().getImplementation(INavigationHistory.class);

        return history;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getSessionID()
     */
    public String getSessionID()
    {
        return sessionID;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getSessionTimeOut()
     */
    public long getSessionTimeOut()
    {
        return this.sessionTimeOut;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#getUser()
     */
    public DIFUserInSession getUser()
    {
        return user;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IPrivateDIFSession#hasExpiredAfterTimeOut()
     */
    public boolean hasExpiredAfterTimeOut()
    {
        long currentTime = System.currentTimeMillis();

        return (this.getLastAccessTime() + this.getSessionTimeOut()
                + DIFGeneralConfigurationParameters.getInstance().getSessionExpirationTimeAfterTimeout() < currentTime);
    }

    /**
     * The criteria for a timed out session is that the last access time plus the session time out must be higher than
     * the current time.
     * 
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#hasTimedOut()
     */
    public boolean hasTimedOut()
    {

        long currentTime = System.currentTimeMillis();

        return (this.getLastAccessTime() + this.getSessionTimeOut() < currentTime);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#isLogged()
     */
    public boolean isLogged()
    {
        return (user != null);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#isMarkedForRemoval()
     */
    public boolean isMarkedForRemoval()
    {
        return markedForRemoval;
    }

    /**
     * Sets the 'lastAccesTime' property to "now".
     * 
     * @see pt.digitalis.dif.controller.interfaces.IPrivateDIFSession#keepAlive()
     */
    public void keepAlive()
    {
        if (!isMarkedForRemoval() && !hasTimedOut())
            forceKeepAlive();
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#setAttributes(Map)
     */
    public void setAttributes(Map<String, Object> attributes)
    {
        this.attributes = attributes;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#setLanguage(java.lang.String)
     */
    public void setLanguage(String language)
    {
        this.language = language;

    }

    /**
     * Modifier for the 'lastAccessTime' property.
     * 
     * @param lastAccessTime
     *            the 'lastAccessTime' new value to set
     */
    public void setLastAccessTime(long lastAccessTime)
    {
        this.lastAccessTime = lastAccessTime;
    }

    /**
     * @param markedForRemoval
     */
    public void setMarkedForRemoval(Boolean markedForRemoval)
    {
        this.markedForRemoval = markedForRemoval;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#setSessionTimeOut(long)
     */
    public void setSessionTimeOut(long sessionTimeOut)
    {
        this.sessionTimeOut = sessionTimeOut;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDIFSession#setUser(pt.digitalis.dif.controller.objects.DIFUserInSession)
     */
    public void setUser(DIFUserInSession user)
    {
        this.user = user;
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();

        formatter.addItem("Session ID", sessionID);
        formatter.addItem("First Access Time", firstAccessTime);
        formatter.addItem("Last Access Time", lastAccessTime);
        formatter.addItem("language", language);
        formatter.addItem("User in session", getUser());

        Map<String, Object> parsedAttributes = new HashMap<String, Object>();

        for (Entry<String, Object> entry: attributes.entrySet())
            if (!entry.getKey().startsWith("JSONCache:"))
                parsedAttributes.put(entry.getKey(), entry.getValue());

        formatter.addItemIfNotNull("Attributes", parsedAttributes);

        return formatter.getFormatedObject();
    }
}
