/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.objects;

import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.controller.interfaces.INavigationHistory;
import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.dem.CallbackType;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.utils.common.StringUtils;

/**
 * The default navigation history implementation
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created 2008/07/20
 */
public class NavigationHistoryImpl implements INavigationHistory {

    /** The authorization manager instance */
    static private IAuthorizationManager authorizationManager = DIFIoCRegistry.getRegistry().getImplementation(
            IAuthorizationManager.class);

    /** The DEM manager instance */
    static private IDEMManager demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);

    /** The crumbs history list */
    private List<Breadcrumb> historyFirstAccess = new ArrayList<Breadcrumb>();

    /** The crumbs history list */
    private List<Breadcrumb> historyLastAccess = new ArrayList<Breadcrumb>();

    /**
     * @see pt.digitalis.dif.controller.interfaces.INavigationHistory#addBreadcrumb(pt.digitalis.dif.controller.objects.Breadcrumb)
     */
    synchronized public void addBreadcrumb(Breadcrumb crumb)
    {

        int total;
        int current;
        boolean added;

        /*
         * Add to first accessed ----------------------------
         */
        total = historyFirstAccess.size();
        current = 0;
        added = false;

        while (!added && current < total)
        {
            if (historyFirstAccess.get(current).getStageID().equals(crumb.getStageID()))
            {
                // Already present, so replace it...
                historyFirstAccess.set(current, crumb);
                added = true;
            }
            else
                current++;
        }

        if (!added)
            // Does not exist, so add it...
            historyFirstAccess.add(crumb);

        /*
         * Add to last accessed ----------------------------
         */
        total = historyLastAccess.size();
        current = 0;
        added = false;

        while (current < total)
        {
            if (historyLastAccess.get(current).getStageID().equals(crumb.getStageID()))
            {
                historyLastAccess.remove(current);
                total--;
            }
            else
                current++;
        }

        historyLastAccess.add(crumb);
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.INavigationHistory#addStage(pt.digitalis.dif.dem.interfaces.IStageInstance)
     */
    public void addStage(IStageInstance stage)
    {
        // Only add if the stage has callback enabled
        if (stage.hasCallbackEnabled() && !stage.getContext().getRequest().isPopupMode()
                && !stage.getContext().getRequest().isAjaxMode() && !stage.getContext().getRequest().isComponentMode())
        {
            Breadcrumb crumb = new Breadcrumb(stage.getID());

            if (stage.getCallbackType().equals(CallbackType.SAVE_SESSION))
                // If the stage is configured to save the session do it
                // VALIDATE: Should we save the parameters also? If not the result may differ from the previous response
                crumb.setDifSession(stage.getContext().getSession());

            if (stage.getCallbackType().equals(CallbackType.SAVE_PARAMETERS)
                    || stage.getCallbackType().equals(CallbackType.SAVE_SESSION))
            {
                String parameters = null;

                for (Entry<String, ? extends Object> entry: stage.getContext().getRequest().getParameters().entrySet())
                {
                    if (HTTPConstants.getPrivateParameters().indexOf(entry.getKey()) == -1)
                    {
                        Object value = entry.getValue();

                        if (value != null)
                        {
                            if (parameters == null)
                                parameters = entry.getKey() + "=" + entry.getValue().toString();
                            else
                                parameters += "&" + entry.getKey() + "=" + entry.getValue().toString();
                        }
                    }
                }

                if (StringUtils.isNotBlank(parameters))
                    crumb.setParameterPassed(parameters);
            }

            this.addBreadcrumb(crumb);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.INavigationHistory#cleanUpAfterLogout(pt.digitalis.dif.controller.interfaces.IDIFSession)
     */
    public void cleanUpAfterLogout(IDIFSession session)
    {
        IDIFUser currentUser = session.getUser();
        boolean hasAccess = false;

        List<Breadcrumb> crumbsList = new ArrayList<Breadcrumb>();
        crumbsList.addAll(historyFirstAccess);

        for (Breadcrumb crumb: crumbsList)
        {
            IStage stage = demManager.getStage(crumb.getStageID());

            if (currentUser == null)
                hasAccess = authorizationManager.hasAccessPublic(stage);
            else
                hasAccess = currentUser.canAccess(stage);

            if (!hasAccess)
            {
                removeStage(crumb.getStageID());
            }
        }
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.INavigationHistory#getHistoryFirstAccess()
     */
    public List<Breadcrumb> getHistoryFirstAccess()
    {
        return historyFirstAccess;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.INavigationHistory#getHistoryLastAccessed()
     */
    public List<Breadcrumb> getHistoryLastAccessed()
    {
        return historyLastAccess;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.INavigationHistory#getPreviousForByFirstAccess(java.lang.String)
     */
    public Breadcrumb getPreviousForByFirstAccess(String stageToFindPrevious)
    {
        Breadcrumb previous = null;
        stageToFindPrevious = stageToFindPrevious.toLowerCase();

        for (Breadcrumb current: historyFirstAccess)
        {
            if (current.getStageID().equals(stageToFindPrevious))
                break;

            previous = current;
        }

        return previous;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.INavigationHistory#getPreviousForByLastAccess(java.lang.String)
     */
    public Breadcrumb getPreviousForByLastAccess(String stageToFindPrevious)
    {
        Breadcrumb previous = null;
        stageToFindPrevious = stageToFindPrevious.toLowerCase();

        for (Breadcrumb current: historyLastAccess)
        {
            if (current.getStageID().equals(stageToFindPrevious))
                break;

            previous = current;
        }

        return previous;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.INavigationHistory#isEmpty()
     */
    public boolean isEmpty()
    {
        return historyFirstAccess.isEmpty() || historyLastAccess.isEmpty();
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.INavigationHistory#removeStage(java.lang.String)
     */
    public void removeStage(String stage)
    {
        int index = 0;

        while (index < historyFirstAccess.size())
        {
            if (historyFirstAccess.get(index).getStageID().equals(stage))
                historyFirstAccess.remove(index);
            else
                index++;
        }

        index = 0;

        while (index < historyLastAccess.size())
        {
            if (historyLastAccess.get(index).getStageID().equals(stage))
                historyLastAccess.remove(index);
            else
                index++;
        }
    }
}
