/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.security.managers.impl;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.controller.security.objects.DIFGroupImpl;
import pt.digitalis.dif.controller.security.objects.DIFUserImpl;
import pt.digitalis.dif.controller.security.objects.IDIFClonableUser;
import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.Pagination;
import pt.digitalis.dif.utils.logging.DIFLogger;

/**
 * Default implementation for the identity manager.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Fbio Souto <a href="mailto:fsouto@digitalis.pt">fsouto@digitalis.pt</a><br/>
 * @created 2007/12/04
 */
public class IdentityManagerStaticImpl extends AbstractIdentityManager {

    /** The authorization manager. */
    static private IAuthorizationManager authorizationManager = DIFIoCRegistry.getRegistry().getImplementation(
            IAuthorizationManager.class);

    /** The '' char as a String literal. */
    final static private String EMPTY = "";

    /** The ' ' char as a String literal. */
    final static private String SPACE = " ";

    /** Users file name to load */
    private static final String USERS_FILE_NAME = "users.xml";

    /** Stores the sub-groups that belong to a given group (index for searching). <K = groupID, V = {subGroupIDs}> */
    Map<String, Set<String>> groupGroups = new HashMap<String, Set<String>>();

    /** The group list. */
    Map<String, IDIFGroup> groupList = new HashMap<String, IDIFGroup>();

    /** Stores the users that belong to a given group (index for searching). <K = groupID, V = {userIDs}> */
    Map<String, Set<String>> groupUsers = new HashMap<String, Set<String>>();

    /** Stores the groups that belong to a given user (index for searching). <K = userID, V = {groupIDs}> */
    Map<String, Set<String>> userGroups = new HashMap<String, Set<String>>();

    /** The user list. */
    Map<String, IDIFClonableUser> userList = new HashMap<String, IDIFClonableUser>();

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addGroup(IDIFGroup) */
    synchronized public void addGroup(IDIFGroup newGroup) throws IdentityManagerException
    {
        if (newGroup != null && newGroup.getID() != null && !EMPTY.equals(newGroup.getID())
                && !SPACE.equals(newGroup.getID()) && !N_A.equals(newGroup.getID()))
        {
            this.groupList.put(newGroup.getID(), newGroup);
            this.groupUsers.put(newGroup.getID(), new HashSet<String>());
        }
        else
            throw new IdentityManagerException("Mandatory field ID doesn't exist on group!");
    }

    /**
     * Adds a group to the IdentityManager if it does not exist
     * 
     * @param groupID
     * @param reportAdditions
     * @throws IdentityManagerException
     */
    private void addGroupIfNotExists(String groupID, boolean reportAdditions) throws IdentityManagerException
    {
        if (!groupExists(groupID))
        {
            IDIFGroup group = new DIFGroupImpl();
            group.setID(groupID);
            group.setName(groupID);
            group.setDescription(groupID);

            addGroup(group);

            if (reportAdditions)
                DIFLogger.getLogger().warn(
                        "  => Group \"" + groupID + "\" not found. Was added to the IdentityManager.");
        }
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addUser(IDIFUser) */
    synchronized public void addUser(IDIFUser newUser) throws IdentityManagerException
    {
        if (newUser != null && newUser.getID() != null && !EMPTY.equals(newUser.getID())
                && !SPACE.equals(newUser.getID()) && !N_A.equals(newUser.getID()))
        {
            this.userList.put(newUser.getID(), (IDIFClonableUser) newUser);
            this.userGroups.put(newUser.getID(), new HashSet<String>());
        }
        else
            throw new IdentityManagerException("Mandatory field ID doesn't exist on user!");
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addUserToGroup(String, String) */
    synchronized public void addUserToGroup(String userID, String groupID)
    {
        if (this.groupUsers.containsKey(groupID))
        {
            this.groupUsers.get(groupID).add(userID);
            this.userGroups.get(userID).add(groupID);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#changePassword(java.lang.String,
     *      java.lang.String)
     */
    synchronized public void changePassword(String userID, String newPassword) throws IdentityManagerException
    {
        if (this.userList.containsKey(userID))
        {
            this.userList.get(userID).setPassword(newPassword);
        }
    }

    /**
     * Will compare a given value to the current one.<br/>
     * Supports partial matches by adding "%" or "*" before and/or after the attribute value
     * 
     * @param value
     *            the original value to test
     * @param compareValue
     *            the value to compare
     * @return T if the compare value matches
     */
    private boolean compareAttribute(String value, String compareValue)
    {
        if (value == null || compareValue == null)
            return false;
        else
        {
            boolean wildCardAtstart = compareValue.startsWith("%") || compareValue.startsWith("*");
            boolean wildCardAtEnd = compareValue.endsWith("%") || compareValue.endsWith("*");

            if (wildCardAtstart && wildCardAtEnd)
                return value.contains(compareValue);
            else if (wildCardAtstart && !wildCardAtEnd)
                return value.endsWith(compareValue);
            else if (!wildCardAtstart && wildCardAtEnd)
                return value.startsWith(compareValue);
            else
                return value.equals(compareValue);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllGroups()
     */
    public int countAllGroups() throws IdentityManagerException
    {
        return this.groupList.values().size();
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllGroupsOfUser(String) */
    public int countAllGroupsOfUser(String userId) throws IdentityManagerException
    {
        return this.getUserGroups(userId).size();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllUsers()
     */
    public int countAllUsers() throws IdentityManagerException
    {
        return this.userList.values().size();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllUsers(java.lang.String)
     */
    public int countAllUsers(String groupID) throws IdentityManagerException
    {
        return getUserIDsInGroup(groupID).size();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countUsers(java.util.Map)
     */
    public int countUsers(Map<String, String> attributes) throws IdentityManagerException
    {
        return this.getUsersByAttributes(attributes).size();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.impl.AbstractIdentityManager#gatherManagedAttributes()
     */
    @Override
    public List<String> gatherManagedAttributes()
    {
        return new ArrayList<String>();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getAllGroups()
     */
    public Set<IDIFGroup> getAllGroups()
    {
        Set<IDIFGroup> result = new HashSet<IDIFGroup>();
        for (IDIFGroup g: this.groupList.values())
            result.add(g.cloneGroup());
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getAllUsers()
     */
    public Set<IDIFUser> getAllUsers()
    {
        Set<IDIFUser> result = new HashSet<IDIFUser>();
        for (IDIFClonableUser u: this.userList.values())
            result.add(u.cloneUser());
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getExclusionCharaters()
     */
    public String getExclusionCharaters()
    {

        return null;
    }

    /**
     * Returns null if the group doesn't exist.
     * 
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroup(String)
     */
    public IDIFGroup getGroup(String groupID)
    {
        IDIFGroup group = this.groupList.get(groupID);

        if (group != null && !group.equals(N_A))
            group = group.cloneGroup();

        return group;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupAttributeName()
     */
    public String getGroupAttributeName()
    {
        return "group";
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupGroups(java.lang.String)
     */
    public Map<String, IDIFGroup> getGroupGroups(String parentGroupID) throws IdentityManagerException
    {

        Map<String, IDIFGroup> groupsMap = new HashMap<String, IDIFGroup>();

        for (IDIFGroup group: groupList.values())
        {
            if (group.getParentGroupID() != null && group.getParentGroupID().equals(parentGroupID))
            {
                groupsMap.put(group.getID(), group);
                groupsMap.putAll(getGroupGroups(group.getID()));
            }
        }

        return groupsMap;

    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroups(Pagination)
     */
    public Set<IDIFGroup> getGroups(Pagination page) throws IdentityManagerException
    {
        ArrayList<IDIFGroup> groups = new ArrayList<IDIFGroup>(this.groupList.values());
        HashSet<IDIFGroup> result = new HashSet<IDIFGroup>();

        for (int i = page.getStartRow(); i < page.getStartRow() + page.getRowsPerPage(); i++)
        {
            IDIFGroup grp = groups.get(i);
            result.add(grp);
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupUsers(java.lang.String)
     */
    public Map<String, IDIFUser> getGroupUsers(String groupID)
    {
        Map<String, IDIFUser> users = new HashMap<String, IDIFUser>();

        for (String userID: getUserIDsInGroup(groupID))
            users.put(userID, getUser(userID));

        return users;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getMailAttributeName()
     */
    public String getMailAttributeName()
    {
        return "mail";
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getNameAttributeName()
     */
    public String getNameAttributeName()
    {
        return "name";
    }

    /**
     * Returns null if the user doesn't exist.
     * 
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUser(String)
     */
    public IDIFUser getUser(String userID)
    {

        IDIFClonableUser user = userList.get(userID);

        if (user != null)
            return user.cloneUser();
        else
            return null;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroups(java.lang.String)
     */
    public Map<String, IDIFGroup> getUserGroups(String userID) throws IdentityManagerException
    {
        return getUserGroupsPagination(userID, null);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroupsIDs(java.lang.String)
     */
    public Set<String> getUserGroupsIDs(String userID)
    {
        Set<String> result = new HashSet<String>();
        result.addAll(getUserGroupsIDsWithOutProfile(userID));
        try
        {
            String profile = this.userList.get(userID).getProfileID();

            if (profile != null)
                result.add(profile);
        }
        catch (IdentityManagerException e)
        {
            // Do nothing
        }

        return result;
    }

    /**
     * Returns the list of group IDs of a given user
     * 
     * @param userID
     *            the user
     * @return the list of groups
     */
    private Set<String> getUserGroupsIDsWithOutProfile(String userID)
    {
        Set<String> result = new HashSet<String>();
        if (this.userGroups.containsKey(userID))
        {
            result.addAll(this.userGroups.get(userID));
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroupsPagination(java.lang.String,
     *      pt.digitalis.dif.utils.Pagination)
     */
    public Map<String, IDIFGroup> getUserGroupsPagination(String userID, Pagination page)
            throws IdentityManagerException
    {
        Map<String, IDIFGroup> result = new HashMap<String, IDIFGroup>();

        Set<String> groups;
        if (page != null)
        {
            groups = getUserGroupsIDsWithOutProfile(userID);
        }
        else
        {
            groups = getUserGroupsIDs(userID);
        }

        if (page != null)
        {
            String[] groupIDs = groups.toArray(new String[0]);

            for (int i = page.getStartRow(); i < page.getStartRow() + page.getRowsPerPage(); i++)
            {
                if (i < groupIDs.length)
                {
                    result.put(groupIDs[i], getGroup(groupIDs[i]));
                }
                else
                {
                    break;
                }
            }
        }
        else
        {
            for (String groupID: groups)
                result.put(groupID, getGroup(groupID));
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserIDsInGroup(java.lang.String)
     */
    public Set<String> getUserIDsInGroup(String groupID)
    {
        if (this.groupUsers.containsKey(groupID))
            return this.groupUsers.get(groupID);
        else
            return new HashSet<String>();
    }

    // /**
    // * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addGroupToGroup(java.lang.String,
    // * java.lang.String)
    // */
    // synchronized public void addGroupToGroup(String subGroupID, String groupID) {
    // if (this.groupList.containsKey(groupID)) {
    // if (!this.groupGroups.containsKey(groupID))
    // this.groupGroups.put(groupID, new HashSet<String>());
    //
    // this.groupGroups.get(groupID).add(subGroupID);
    // }
    // }
    //
    // /**
    // * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupIDsOfGroup(java.lang.String)
    // */
    // public Set<String> getGroupIDsOfGroup(String groupID) {
    // if (this.groupGroups.containsKey(groupID))
    // return this.groupGroups.get(groupID);
    // else
    // return new HashSet<String>();
    // }

    // /**
    // * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupsOfGroup(java.lang.String)
    // */
    // public Map<String, IDIFGroup> getGroupsOfGroup(String groupID) {
    // Map<String, IDIFGroup> groups = new HashMap<String, IDIFGroup>();
    //
    // for (String subGroupID : getGroupIDsOfGroup(groupID))
    // groups.put(subGroupID, getGroup(subGroupID));
    //
    // return groups;
    // }

    // /**
    // * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isGroupInGroup(java.lang.String,
    // * java.lang.String)
    // */
    // public boolean isGroupInGroup(String subGroupID, String groupID) {
    // if (this.groupGroups.containsKey(groupID))
    // return this.groupGroups.get(groupID).contains(subGroupID);
    // else
    // return false;
    // }

    // /**
    // * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeGroupFromGroup(java.lang.String,
    // * java.lang.String)
    // */
    // synchronized public void removeGroupFromGroup(String subGroupID, String groupID) {
    // if (this.groupGroups.containsKey(groupID)) {
    // this.groupGroups.get(groupID).remove(subGroupID);
    // }
    // }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserLoginAttributeName()
     */
    public String getUserLoginAttributeName()
    {
        return "id";
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserParentGroupAttributeName()
     */
    public String getUserParentGroupAttributeName()
    {
        return "profile";
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsers(Pagination)
     */
    public Set<IDIFUser> getUsers(Pagination page) throws IdentityManagerException
    {
        ArrayList<IDIFUser> users = new ArrayList<IDIFUser>(this.userList.values());
        HashSet<IDIFUser> result = new HashSet<IDIFUser>();

        for (int i = page.getStartRow(); i < page.getStartRow() + page.getRowsPerPage(); i++)
        {
            IDIFUser usr = users.get(i);
            result.add(usr);
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAnyAttribute(java.util.Map)
     */
    public Set<IDIFUser> getUsersByAnyAttribute(Map<String, String> attributes) throws IdentityManagerException
    {
        Set<IDIFUser> result = new HashSet<IDIFUser>();
        for (IDIFClonableUser u: this.userList.values())
        {
            boolean allConditions = false;
            for (Entry<String, String> attr: attributes.entrySet())
            {
                if (attr.getValue() != null && attr.getKey() != null)
                {
                    if (attr.getKey().equals(this.getUserLoginAttributeName()))
                        allConditions |= this.compareAttribute(u.getID(), attr.getValue());
                    if (attr.getKey().equals(this.getMailAttributeName()))
                        allConditions |= this.compareAttribute(u.getEmail(), attr.getValue());
                    if (attr.getKey().equals(this.getNameAttributeName()))
                        allConditions |= this.compareAttribute(u.getName(), attr.getValue());
                    if (attr.getKey().equals(this.getUserParentGroupAttributeName()))
                        allConditions |= u.getProfile() != null
                                && this.compareAttribute(u.getProfile().getID(), attr.getValue());

                    if (u.getAttribute(attr.getKey()) != null)
                        allConditions |= this.compareAttribute(attr.getValue(), u.getAttribute(attr.getKey())
                                .toString());
                }
            }
            if (allConditions)
                result.add(u.cloneUser());
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAttribute(java.lang.String,
     *      java.lang.String)
     */
    public Set<IDIFUser> getUsersByAttribute(String attribute, String value) throws IdentityManagerException
    {
        Map<String, String> attributes = new HashMap<String, String>();
        attributes.put(attribute, value);

        return this.getUsersByAttributes(attributes);
    }

    /**
     * @throws IdentityManagerException
     *             If the operation cannot be completed.
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAttributes(java.util.Map)
     */
    public Set<IDIFUser> getUsersByAttributes(Map<String, String> attributes) throws IdentityManagerException
    {
        Set<IDIFUser> result = new HashSet<IDIFUser>();
        for (IDIFClonableUser u: this.userList.values())
        {
            boolean allConditions = true;
            for (Entry<String, String> attr: attributes.entrySet())
            {
                boolean isAttributeInUser = false;
                if (attr.getValue() != null && attr.getKey() != null)
                {
                    if (attr.getKey().equals(this.getUserLoginAttributeName()))
                        isAttributeInUser = this.compareAttribute(u.getID(), attr.getValue());
                    else if (attr.getKey().equals(this.getMailAttributeName()))
                        isAttributeInUser = this.compareAttribute(u.getEmail(), attr.getValue());
                    else if (attr.getKey().equals(this.getNameAttributeName()))
                        isAttributeInUser = this.compareAttribute(u.getName(), attr.getValue());
                    else if (attr.getKey().equals(this.getUserParentGroupAttributeName()))
                        isAttributeInUser = u.getProfile() != null
                                && this.compareAttribute(u.getProfile().getID(), attr.getValue());

                    else if (u.getAttribute(attr.getKey()) != null)
                        isAttributeInUser = this.compareAttribute(attr.getValue(), u.getAttribute(attr.getKey())
                                .toString());
                }

                allConditions &= isAttributeInUser;

                if (!allConditions)
                    break;
            }
            if (allConditions)
                result.add(u.cloneUser());
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAttributes(java.util.Map,
     *      pt.digitalis.dif.utils.Pagination)
     */
    public Set<IDIFUser> getUsersByAttributes(Map<String, String> attributes, Pagination page)
            throws IdentityManagerException
    {
        ArrayList<IDIFUser> totalUsers = new ArrayList<IDIFUser>(this.getUsersByAttributes(attributes));
        Set<IDIFUser> result = new HashSet<IDIFUser>();

        for (int i = page.getStartRow(); i < page.getStartRow() + page.getRowsPerPage(); i++)
        {
            IDIFUser usr = totalUsers.get(i);
            result.add(usr);
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByEmail(java.lang.String)
     */
    public Set<IDIFUser> getUsersByEmail(String value) throws IdentityManagerException
    {
        Set<IDIFUser> difUsers = new HashSet<IDIFUser>();
        for (IDIFClonableUser user: userList.values())
        {
            if (value.equals(user.getEmail()))
                difUsers.add(user.cloneUser());
        }
        return difUsers;
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#groupExists(String) */
    public boolean groupExists(String groupID)
    {
        return this.groupList.containsKey(groupID);
    }

    /**
     * Initializes users from custom static file "users.xml"
     */
    public void initializeStaticCustomUsers()
    {
        if (DIFStartupConfiguration.getDeveloperMode() || DIFStartupConfiguration.getTestingMode()
                || DIFStartupConfiguration.getDemoMode())
        {
            int userCount = 0;
            int groupCount = 0;

            try
            {
                DIFLogger.getLogger().info("Loading \"" + USERS_FILE_NAME + "\"...");
                InputStream userStream = Thread.currentThread().getContextClassLoader()
                        .getResourceAsStream(USERS_FILE_NAME);

                if (userStream == null)
                    DIFLogger.getLogger().info("  => No \"" + USERS_FILE_NAME + "\" found.");
                else
                {
                    DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
                    DocumentBuilder db = dbf.newDocumentBuilder();
                    Document doc = db.parse(userStream);
                    doc.getDocumentElement().normalize();

                    // Parse groups
                    NodeList groups = doc.getElementsByTagName("group");
                    for (int i = 0; i < groups.getLength(); i++)
                    {
                        String groupID = ((Element) groups.item(i)).getChildNodes().item(0).getNodeValue();
                        addGroupIfNotExists(groupID.trim(), false);

                        groupCount++;
                    }

                    if (groupCount > 0)
                        DIFLogger.getLogger().info("  => " + groupCount + " group(s) parsed");

                    // Parse users
                    NodeList users = doc.getElementsByTagName("user");

                    for (int i = 0; i < users.getLength(); i++)
                    {
                        Node userNode = users.item(i);

                        if (userNode.getNodeType() == Node.ELEMENT_NODE)
                        {
                            Element attribute = (Element) userNode;

                            // User definition
                            IDIFUser user = new DIFUserImpl();
                            user.setID(attribute.getElementsByTagName("username").item(0).getChildNodes().item(0)
                                    .getNodeValue());
                            user.setName(user.getID());
                            user.setPassword(attribute.getElementsByTagName("password").item(0).getChildNodes().item(0)
                                    .getNodeValue());

                            // Add user
                            addUser(user);

                            // User attributes
                            String attributes = attribute.getElementsByTagName("attributes").item(0).getChildNodes()
                                    .item(0).getNodeValue();

                            if (attributes != null && !"".equals(attributes))
                            {
                                String[] attributeArray = attributes.split(",");

                                for (int f = 0; f < attributeArray.length; f++)
                                {
                                    String[] attributeDef = attributeArray[f].split("=");
                                    user.setAttribute(attributeDef[0].trim(), attributeDef[1].trim());
                                }
                            }

                            String temp = attribute.getElementsByTagName("profile").item(0).getChildNodes().item(0)
                                    .getNodeValue();

                            // Set profile
                            addGroupIfNotExists(temp, true);
                            user.setProfileID(temp);

                            // User groups
                            String groupString = attribute.getElementsByTagName("groups").item(0).getChildNodes()
                                    .item(0).getNodeValue();

                            if (groupString != null && !"".equals(groupString))
                            {
                                String[] groupArray = groupString.split(",");

                                for (int f = 0; f < groupArray.length; f++)
                                {
                                    addGroupIfNotExists(groupArray[f].trim(), true);
                                    addUserToGroup(user.getID(), groupArray[f].trim());
                                }
                            }

                            updateUser(user, user.getID());
                        }

                        userCount++;
                    }

                    if (userCount > 0)
                        DIFLogger.getLogger().info("  => " + userCount + " user(s) parsed");
                }
            }
            catch (Exception e)
            {
                DIFLogger.getLogger().error("  => Load of static users failed!\n\n");
                e.printStackTrace();
            }
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isIdentityValid(java.lang.String,
     *      java.lang.String)
     */
    public boolean isIdentityValid(String userID, String suppliedPassword)
    {

        IDIFClonableUser user = userList.get(userID);

        if ((user != null) && suppliedPassword.equals(user.getPassword()))
            return true;
        else
            return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isReadOnly()
     */
    public boolean isReadOnly()
    {
        return false;
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isUserInGroup(String, String) */
    public boolean isUserInGroup(String userID, String groupID)
    {
        if (this.groupUsers.containsKey(groupID))
            return this.groupUsers.get(groupID).contains(userID);
        else
            return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManagerPrivate#persistUserAttribute(java.lang.String,
     *      java.lang.String, java.lang.Object)
     */
    synchronized public void persistUserAttribute(String userID, String attributeID, Object attributeValue)
            throws IdentityManagerException
    {
        // No need since the static implementation has no underlying repository
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManagerPrivate#persistUserAttributes(java.lang.String,
     *      java.util.Map)
     */
    public void persistUserAttributes(String userID, Map<String, Object> attributes) throws IdentityManagerException
    {
        // No need since the static implementation has no underlying repository
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeGroup(String) */
    synchronized public void removeGroup(String groupID)
    {
        this.groupList.remove(groupID);
        this.groupUsers.remove(groupID);
        authorizationManager.revokeAllAccessFromGroup(groupID);
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeUser(String) */
    synchronized public void removeUser(String userID)
    {
        this.userList.remove(userID);
        this.userGroups.remove(userID);
        authorizationManager.revokeAllAccessFromUser(userID);
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeUserFromGroup(String, String) */
    synchronized public void removeUserFromGroup(String userID, String groupID)
    {
        if (this.groupUsers.containsKey(groupID))
        {
            this.groupUsers.get(groupID).remove(userID);
            this.userGroups.get(userID).remove(groupID);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#resetIdentityManager()
     */
    public void resetIdentityManager()
    {
        this.groupGroups.clear();
        this.groupList.clear();
        this.groupUsers.clear();
        this.userGroups.clear();
        this.userList.clear();
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();
        formatter.addItem("User List", userList);
        formatter.addItem("Group List", groupList);

        return formatter.getFormatedObject();
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateGroup(IDIFGroup) */
    synchronized public void updateGroup(IDIFGroup existingGroup)
    {
        if (this.groupList.containsKey(existingGroup.getID()))
            // Implementation note: java.util.Map#put(K,V) replaces the V of an existing key, so it can be used for
            // updates.
            this.groupList.put(existingGroup.getID(), existingGroup);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateUser(pt.digitalis.dif.controller.security.objects.IDIFUser,
     *      java.lang.String)
     */
    synchronized public void updateUser(IDIFUser existingUser, String userID)
    {
        if (this.userList.containsKey(userID))
        {

            existingUser.cleanCache();

            // Implementation note: java.util.Map#put(K,V) replaces the V of an existing key, so it can be used for
            // updates.
            if (existingUser.getID() != null && !existingUser.getID().equals(userID))
            {
                this.userList.remove(userID);
                this.userList.put(existingUser.getID(), (IDIFClonableUser) existingUser);

                this.userGroups.put(existingUser.getID(), this.userGroups.get(userID));
                this.userGroups.remove(userID);

            }
            else
            {
                this.userList.put(userID, (IDIFClonableUser) existingUser);
            }
        }

        existingUser.refresh();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateUserAttribute(java.lang.String,
     *      java.lang.String, java.lang.Object)
     */
    synchronized public void updateUserAttribute(String userID, String attributeID, Object attributeValue)
            throws IdentityManagerException
    {
        if (!this.userList.containsKey(userID))
        {
            throw new IdentityManagerException("User " + userID + " does not exists on the IdentityManager!! ");
        }
        else if (!this.userList.get(userID).containsAttribute(attributeID))
        {
            throw new IdentityManagerException("User " + userID + " does not contains the attribute: " + attributeID
                    + "!");
        }

        try
        {
            this.userList.get(userID).setAttribute(attributeID, attributeValue);
        }
        catch (InternalFrameworkException e)
        {
            throw new IdentityManagerException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateUserAttributes(java.lang.String,
     *      java.util.Map)
     */
    synchronized public void updateUserAttributes(String userID, Map<String, Object> attributes)
            throws IdentityManagerException
    {
        if (!this.userList.containsKey(userID))
        {
            throw new IdentityManagerException("User " + userID + " does not exists on the IdentityManager!! ");
        }
        try
        {
            this.userList.get(userID).setAttributes(attributes);
        }
        catch (InternalFrameworkException e)
        {
            throw new IdentityManagerException(e);
        }
    }

    /** @see pt.digitalis.dif.controller.security.managers.IIdentityManager#userExists(String) */
    public boolean userExists(String userID)
    {
        if (userID == null)
            return false;
        else
            return this.userList.containsKey(userID);
    }
}
