/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.controller.security.managers.impl;

import java.util.HashMap;
import java.util.Map;

import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.controller.interfaces.IPrivateDIFSession;
import pt.digitalis.dif.controller.objects.Constants;
import pt.digitalis.dif.controller.objects.DIFSession;
import pt.digitalis.dif.controller.objects.DIFUserInSession;
import pt.digitalis.dif.controller.security.managers.IAuthenticationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.ISessionManager;
import pt.digitalis.dif.controller.security.managers.ISessionManagerInternal;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.security.AuthenticationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.ObjectFormatter;

import com.google.inject.Inject;
import com.newrelic.api.agent.Trace;

/**
 * DIF's default implementation of a session manager. REFACTOR: Should extract a superclass from this as a base
 * implementation for this and other sessionManager impl's
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created Dec 11, 2007
 */
public class SessionManagerImpl implements ISessionManager, ISessionManagerInternal {

    /** List of logged sessions. */
    private Map<String, IPrivateDIFSession> loggedSessions = new HashMap<String, IPrivateDIFSession>();

    /** The authentication manager. */
    @Inject
    protected IAuthenticationManager theAuthenticationManager;

    /** The identity manager. */
    @Inject
    protected IIdentityManager theIDManager;

    /**
     * Default constructor
     */
    public SessionManagerImpl()
    {
        // Launches the inactive session monitoring thread
        SessionGarbageCollector collector = new SessionGarbageCollector(this);
        collector.start();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#createSession(java.lang.String)
     */
    public IDIFSession createSession(String sessionID)
    {
        IPrivateDIFSession session = loggedSessions.get(sessionID);

        if (session == null || session.isMarkedForRemoval())
        {
            session = new DIFSession(sessionID);
            session.setSessionTimeOut(DIFGeneralConfigurationParameters.getInstance().getSessionTimeout());

            loggedSessions.put(sessionID, session);
        }
        else
            update(session);

        return session;
    }

    /**
     * @return the loggedSessions
     */
    public Map<String, IPrivateDIFSession> getLoggedSessions()
    {
        return loggedSessions;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#getSession(java.lang.String)
     */
    public IDIFSession getSession(String sessionID)
    {
        return loggedSessions.get(sessionID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#isSessionPresent(java.lang.String)
     */
    public boolean isSessionPresent(String sessionID)
    {
        return loggedSessions.containsKey(sessionID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#logIn(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    @Trace(metricName = "DIF:SessionManager:Login", dispatcher = true)
    public IDIFSession logIn(String sessionID, String userID, String password) throws AuthenticationManagerException
    {
        IDIFUser loggedUser = theAuthenticationManager.logIn(sessionID, userID, password);

        if (loggedUser != null)
            return this.logInNoPasswordValidation(sessionID, userID, password);
        else
            return this.createSession(sessionID);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.ISessionManagerInternal#logInNoPasswordValidation(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public IDIFSession logInNoPasswordValidation(String sessionID, String userID, String password)
            throws AuthenticationManagerException
    {
        IDIFSession session = createSession(sessionID);

        try
        {
            IDIFUser loggedUser = theIDManager.getUser(userID);

            if (loggedUser != null)
            {
                session.setUser(new DIFUserInSession(loggedUser, password));
                update(session);
            }

        }
        catch (IdentityManagerException e)
        {
            throw new AuthenticationManagerException("Problem getting the user from indentity manager", e);
        }

        return session;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#logOut(java.lang.String)
     */
    @Trace(metricName = "DIF:SessionManager:Logout", dispatcher = true)
    public IDIFSession logOut(String sessionID)
    {
        IDIFSession session = loggedSessions.get(sessionID);

        if (session != null)
        {
            session.setUser(null);
            theAuthenticationManager.logOut(sessionID);
            session.getNavigationHistory().cleanUpAfterLogout(session);
            update(session);
            reinitializeSession(sessionID);
        }

        return session;
    }

    /**
     * Cleans up an existant session, after logout or clean session request
     * 
     * @param sessionID
     *            the session ID to clean up
     */
    private void reinitializeSession(String sessionID)
    {
        IPrivateDIFSession session = loggedSessions.get(sessionID);

        if (session != null)
        {
            Object sessionInvalidBrowser = session.getAttribute(Constants.INVALID_BROWSER_ACCEPTED);
            session.setAttributes(new HashMap<String, Object>());
            session.addAttribute(Constants.INVALID_BROWSER_ACCEPTED, sessionInvalidBrowser);
            loggedSessions.put(sessionID, session);
        }
    }

    /**
     * Removes a session
     * 
     * @param sessionID
     */
    synchronized public void removeSession(String sessionID)
    {
        loggedSessions.remove(sessionID);

        // Instructs the authenticated manager that this client has disconnected from DIF. It will decide if there is an
        // authentication record present what to do with it. Keep or remove it.
        theAuthenticationManager.disconnectClient(sessionID);
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();
        formatter.addItem("Sessions", loggedSessions);

        return formatter.getFormatedObject();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.ISessionManager#update(pt.digitalis.dif.controller.interfaces.IDIFSession)
     */
    public boolean update(IDIFSession session)
    {
        boolean exists = isSessionPresent(session.getSessionID());

        if (exists)
        {
            IPrivateDIFSession privSession = (IPrivateDIFSession) session;

            privSession.keepAlive();
            loggedSessions.put(privSession.getSessionID(), privSession);
        }

        return exists;
    }
}
