/**
 * - Digitalis Internal Framework v2.0 -
 *
 * (C) 2007, Digitalis Informatica.
 *
 * Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1
 * 2780-666 Paco de Arcos
 * Telefone: (351) 21 4408990
 * Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem;

import java.util.Map;

import pt.digitalis.dif.dem.annotations.entities.ApplicationDefinition;
import pt.digitalis.dif.dem.annotations.entities.ProviderDefinition;
import pt.digitalis.dif.dem.annotations.entities.ServiceDefinition;
import pt.digitalis.dif.dem.annotations.entities.StageDefinition;
import pt.digitalis.dif.dem.annotations.entities.ValidatorDefinition;
import pt.digitalis.utils.bytecode.holders.AnnotationHolder;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

/**
 * Defines the known entity types to the framework.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created 2007/09/13
 *
 */
public enum Entity {

    /**
     * A <code>@ValidatorDefinition</code> annotated Entity.
     */
    VALIDATOR(ValidatorDefinition.class),

    /**
     * A <code>@ProviderDefinition</code> annotated Entity.
     */
    PROVIDER(ProviderDefinition.class),

    /**
     * A <code>@ApplicationDefinition</code> annotated Entity.
     */
    APPLICATION(ApplicationDefinition.class),

    /**
     * A <code>@ServiceDefinition</code> annotated Entity.
     */
    SERVICE(ServiceDefinition.class),

    /**
     * A <code>@StageDefinition</code> annotated Entity.
     */
    STAGE(StageDefinition.class);

    /** The Entity's FQN. */
    private String fullyQualifiedName;

    /** The Entity's name. */
    private String name;

    /**
     * Constructor.
     *
     * @param clazz
     *            the Entity's class
     */
    Entity(Class<?> clazz) {
        this.name = clazz.getName();
        this.fullyQualifiedName = clazz.getCanonicalName();
    }

    /**
     * Returns the Entity's FQN.
     *
     * @return the Entity's FQN.
     */
    public String getFullyQualifiedName() {
        return this.fullyQualifiedName;
    }

    /**
     * Returns the Entity's name.
     *
     * @return the Entity's name.
     */
    public String getName() {
        return this.name;
    }

    /**
     * The ID builder utility method.
     *
     * @param type
     *            the type of the entity
     * @param id
     *            the id of the entity
     * @return the compiled unique id for the given entity
     */
    static public String getID(Entity type, String id) {
        return (type + ":" + id.toLowerCase());
    }

    /**
     * Returns the Entity type of annotated class, null if no valid entity annotation is found.
     *
     * @param clazz
     *            the class to inspect
     * @return the entity type, null if no valid entity annotation is found.
     * @throws ResourceNotFoundException
     *             if the class annotations can't be read
     */
    static public Entity getEntityTypeFromClass(ClassHolder clazz) throws ResourceNotFoundException {

        // Read class annotation
        Map<String, AnnotationHolder> orignalClassAnnotations = clazz.getAnnotations();

        // Return the appropriate entity type
        if (orignalClassAnnotations.containsKey(ValidatorDefinition.class.getCanonicalName()))
            return Entity.VALIDATOR;
        else if (orignalClassAnnotations.containsKey(ProviderDefinition.class.getCanonicalName()))
            return Entity.PROVIDER;
        else if (orignalClassAnnotations.containsKey(ApplicationDefinition.class.getCanonicalName()))
            return Entity.APPLICATION;
        else if (orignalClassAnnotations.containsKey(ServiceDefinition.class.getCanonicalName()))
            return Entity.SERVICE;
        else if (orignalClassAnnotations.containsKey(StageDefinition.class.getCanonicalName()))
            return Entity.STAGE;
        else
            return null;
    }
}
