/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.managers.impl;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IRegistrationManager;
import pt.digitalis.dif.dem.objects.ILicense;
import pt.digitalis.dif.dem.objects.LicenseEditionType;
import pt.digitalis.dif.exception.manager.RegistrationManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.utils.config.IConfigurations;

import com.google.inject.Inject;

/**
 * Manages the DEM Entities registration process.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/06/01
 */
public class RegistrationManagerImpl implements IRegistrationManager {

    /** Missing registrable information tag */
    static private String MISSING_REGISTRABLE_INFO = "DIF auto-generated: Missing registrable information!";

    /** Map of registrations */
    static private Map<String, ILicense> registrations = new HashMap<String, ILicense>();

    /** ID of the config area where the Repository will manage the registrations */
    final static String REPOSITORY_CONFIG_ID = "dif2";

    /** ID that identifies the key */
    final static String REPOSITORY_KEY_ID = "key";

    /** ID that identifies the name */
    final static String REPOSITORY_NAME_ID = "name";

    /** ID of the section where the Repository will manage the registrations */
    final static String REPOSITORY_SECTION_ID = "Registrations";

    /** The registration persistence repository. */
    @Inject
    IConfigurations configRepository;

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#addToRegistry(pt.digitalis.dif.dem.Entity,
     *      java.lang.String, java.lang.String, boolean)
     */
    public void addToRegistry(Entity type, String entityID, String name, boolean registrable)
    {

        String id = Entity.getID(type, entityID.toLowerCase());

        ILicense license = DIFIoCRegistry.getRegistry().getImplementation(ILicense.class);
        license.setName(name);
        license.setRegistrable(registrable);

        // Checks if there are persistent configurations in the repository
        license = checkRepositoryForPersistentRegistration(id, license);

        registrations.put(id, license);
    }

    /**
     * Checks the persistent repository for saved registrations.
     * 
     * @param id
     *            the entity id
     * @param license
     *            the default registration record
     * @return the updated registration record
     */
    private ILicense checkRepositoryForPersistentRegistration(String id, ILicense license)
    {

        Properties props = configRepository.readConfiguration(REPOSITORY_CONFIG_ID, REPOSITORY_SECTION_ID);

        // Tries to get the registrations from the persistence configuration
        Object savedKey = props.get(id + "." + REPOSITORY_KEY_ID);

        if (savedKey != null)
            license.register(savedKey.toString(), id);

        return license;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getApplicationEdition(java.lang.String)
     */
    public LicenseEditionType getApplicationEdition(String applicationID)
    {
        return getRegistrationRecord(Entity.APPLICATION, applicationID).getEdition();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getProviderEdition(java.lang.String)
     */
    public LicenseEditionType getProviderEdition(String providerID)
    {
        return getRegistrationRecord(Entity.PROVIDER, providerID).getEdition();
    }

    /**
     * Utility accessor for the registration object for any entity. It provides the search and initialization for first
     * time access. Private use only since this is a contained helper class that encapsulates all specifics about entity
     * registration.
     * 
     * @param type
     *            the entity type to get
     * @param id
     *            the id of the Entity
     * @return the registration record for the entity
     */
    private ILicense getRegistrationRecord(Entity type, String id)
    {
        String entityID = Entity.getID(type, id);
        ILicense license = registrations.get(entityID);

        if (license == null)
        {
            // The DIF DEM rules state that missing registration information translates to a public registration.
            // If the CodeGen and DEM parsing utility has not provided any registration information then this is a
            // unregistrable entity. Create a new license accordingly.

            license = DIFIoCRegistry.getRegistry().getImplementation(ILicense.class);
            license.setName(MISSING_REGISTRABLE_INFO);
            license.setRegistrable(false);

            // Updates the Map with the registered record. This will only persist in memory until a register/unregister
            // command is issued. Only then it will be considered a true registration license. This one is temporary.
            registrations.put(entityID, license);
        }

        return license;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getServiceEdition(java.lang.String)
     */
    public LicenseEditionType getServiceEdition(String serviceID)
    {
        return getRegistrationRecord(Entity.SERVICE, serviceID).getEdition();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getStageClientName(pt.digitalis.dif.dem.interfaces.IStage)
     */
    public String getStageClientName(IStage stage)
    {
        String result = getRegistrationRecord(Entity.STAGE, stage.getID()).getName();
        if (MISSING_REGISTRABLE_INFO.equals(result))
        {
            result = getRegistrationRecord(Entity.SERVICE, stage.getService().getID()).getName();
            if (MISSING_REGISTRABLE_INFO.equals(result))
            {
                result = getRegistrationRecord(Entity.APPLICATION, stage.getService().getApplication().getID())
                        .getName();
                if (MISSING_REGISTRABLE_INFO.equals(result))
                {
                    result = getRegistrationRecord(Entity.PROVIDER,
                            stage.getService().getApplication().getProvider().getID()).getName();
                    if (MISSING_REGISTRABLE_INFO.equals(result))
                    {
                        result = DIFGeneralConfigurationParameters.getInstance().getClient();
                    }
                }
            }
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#getStageEdition(java.lang.String)
     */
    public LicenseEditionType getStageEdition(String stageID)
    {
        return getRegistrationRecord(Entity.STAGE, stageID).getEdition();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isApplicationRegistered(java.lang.String)
     */
    public boolean isApplicationRegistered(String applicationID)
    {
        return getRegistrationRecord(Entity.APPLICATION, applicationID).isRegistered();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isApplicationRegistrable(java.lang.String)
     */
    public boolean isApplicationRegistrable(String applicationID)
    {
        return getRegistrationRecord(Entity.APPLICATION, applicationID).isRegistrable();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isProviderRegistered(java.lang.String)
     */
    public boolean isProviderRegistered(String providerID)
    {
        return getRegistrationRecord(Entity.PROVIDER, providerID).isRegistered();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isProviderRegistrable(java.lang.String)
     */
    public boolean isProviderRegistrable(String providerID)
    {
        return getRegistrationRecord(Entity.PROVIDER, providerID).isRegistrable();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isServiceRegistered(java.lang.String)
     */
    public boolean isServiceRegistered(String serviceID)
    {
        return getRegistrationRecord(Entity.SERVICE, serviceID).isRegistered();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isServiceRegistrable(java.lang.String)
     */
    public boolean isServiceRegistrable(String serviceID)
    {
        return getRegistrationRecord(Entity.SERVICE, serviceID).isRegistrable();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isStageRegistered(java.lang.String)
     */
    public boolean isStageRegistered(String stageId)
    {
        return getRegistrationRecord(Entity.STAGE, stageId).isRegistered();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#isStageRegistrable(java.lang.String)
     */
    public boolean isStageRegistrable(String stageId)
    {
        return getRegistrationRecord(Entity.STAGE, stageId).isRegistrable();
    }

    /**
     * Persists the given registration to the repository
     * 
     * @param id
     *            the id of the Entity
     * @param license
     *            the registration license
     * @return T if the operation was successful
     */
    private boolean persistRegistration(String id, ILicense license)
    {

        Properties props = new Properties();

        props.put(id + "." + REPOSITORY_NAME_ID, license.getName() != null ? license.getName() : "");

        if (license.getKey() == null)
            props.put(id + "." + REPOSITORY_KEY_ID, "");
        else
            props.put(id + "." + REPOSITORY_KEY_ID, license.getKey());

        return configRepository.writeConfiguration(REPOSITORY_CONFIG_ID, REPOSITORY_SECTION_ID, props);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#registerApplication(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public boolean registerApplication(String applicationID, String name, String key)
            throws RegistrationManagerException
    {
        return registerEntity(Entity.APPLICATION, applicationID, name, key);
    }

    /**
     * Registers an entity
     * 
     * @param type
     *            the type of the entity
     * @param entityID
     *            the id of the entity
     * @param name
     *            the name to register to
     * @param key
     *            the key to register
     * @return T if the registration was successful
     * @throws RegistrationManagerException
     *             if a error occurrs in registration manager
     */
    private boolean registerEntity(Entity type, String entityID, String name, String key)
            throws RegistrationManagerException
    {
        String id = Entity.getID(type, entityID);
        ILicense license = getRegistrationRecord(type, entityID);

        if (!license.isRegistrable())
            throw new RegistrationManagerException("You are trying to register a unregistrable entity");

        license.setName(name);
        license.register(key, id);

        // Updates the Map with the registered record
        registrations.put(id, license);

        // Persists to the Repository
        persistRegistration(id, license);

        return license.isRegistered();
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#registerProvider(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public boolean registerProvider(String providerID, String name, String key) throws RegistrationManagerException
    {
        return registerEntity(Entity.PROVIDER, providerID, name, key);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#registerService(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public boolean registerService(String serviceID, String name, String key) throws RegistrationManagerException
    {
        return registerEntity(Entity.SERVICE, serviceID, name, key);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#registerStage(java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public boolean registerStage(String stageId, String name, String key) throws RegistrationManagerException
    {
        return registerEntity(Entity.STAGE, stageId, name, key);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#unregisterApplication(java.lang.String)
     */
    public void unregisterApplication(String applicationID)
    {
        unregisterEntity(Entity.APPLICATION, applicationID);
    }

    /**
     * Unregisters an entity
     * 
     * @param type
     *            the type of the entity
     * @param entityID
     *            the id of the entity
     */
    private void unregisterEntity(Entity type, String entityID)
    {
        String id = Entity.getID(type, entityID);
        ILicense license = getRegistrationRecord(type, entityID);

        license.unregister();

        // Updates the Map with the registered record
        registrations.put(id, license);

        // Persists to the Repository
        persistRegistration(id, license);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#unregisterProvider(java.lang.String)
     */
    public void unregisterProvider(String providerID)
    {
        unregisterEntity(Entity.PROVIDER, providerID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#unregisterService(java.lang.String)
     */
    public void unregisterService(String serviceID)
    {
        unregisterEntity(Entity.SERVICE, serviceID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IRegistrationManager#unregisterStage(java.lang.String)
     */
    public void unregisterStage(String stageId)
    {
        unregisterEntity(Entity.STAGE, stageId);
    }
}
