package pt.digitalis.dif.dem.objects;

import pt.digitalis.dif.utils.ObjectFormatter;

/**
 * This class represents a ViewObject entity on the DEM.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/04/24
 */
public class ViewObject {

    /** The ViewObject's rendering engine. */
    private String engine;

    /** The ViewObject's id. */
    private String id = new String();

    /** Keeps record if this is the default ViewObject. */
    private boolean isDefault = true;

    /** The ViewObject's name. */
    private String name = new String();

    /** The ViewObject's path. */
    private String target = new String();

    /** The ViewObject's type. */
    private ViewType type = ViewType.NORMAL;

    /** Default constructor. */
    public ViewObject()
    {

    }

    /**
     * ViewObject constructor.
     * 
     * @param engine
     *            the view engine
     * @param type
     *            the view type
     * @param target
     *            the view target
     * @param isDefault
     *            T if it's the default view, F otherwise
     */
    public ViewObject(String engine, ViewType type, String target, boolean isDefault)
    {
        this.engine = engine;
        this.type = type;
        this.target = target;
        this.isDefault = isDefault;
    }

    /**
     * Returns the rendering engine defined for the ViewObject.
     * 
     * @return the rendering engine defined for the ViewObject
     */
    public String getEngine()
    {
        return engine;
    }

    /**
     * Returns the ViewObject's id.
     * 
     * @return the ViewObject's id
     */
    public String getId()
    {
        return id;
    }

    /**
     * Returns the ViewObject's name.
     * 
     * @return the ViewObject's name
     */
    public String getName()
    {
        return name;
    }

    /**
     * Returns the ViewObject's path.
     * 
     * @return the ViewObject's path
     */
    public String getTarget()
    {
        return target;
    }

    /**
     * Returns the ViewObject's type.
     * 
     * @return the ViewObject's type
     */
    public ViewType getType()
    {
        return type;
    }

    /**
     * Returns information on whether the ViewObject is the default ViewObject
     * 
     * @return T if this is the default ViewObject, F otherwise
     */
    public boolean isDefault()
    {
        return isDefault;
    }

    /**
     * Sets the ViewObject as the default ViewObject.
     * 
     * @param isDefault
     *            new ViewObject default state
     */
    public void setDefault(boolean isDefault)
    {
        this.isDefault = isDefault;
    }

    /**
     * Sets the ViewObject's rendering engine.
     * 
     * @param newEngine
     *            the new rendering engine
     */
    public void setEngine(String newEngine)
    {
        this.engine = newEngine;
    }

    /**
     * Sets the ViewObject's ID.
     * 
     * @param newID
     *            the ViewObject's new ID
     */
    public void setId(String newID)
    {
        this.id = newID;
    }

    /**
     * Sets the ViewObject's name.
     * 
     * @param newName
     *            the ViewObject's name
     */
    public void setName(String newName)
    {
        this.name = newName;
    }

    /**
     * Sets the ViewObject' path.
     * 
     * @param newTarget
     *            the ViewObject's new path
     */
    public void setTarget(String newTarget)
    {
        this.target = newTarget;
    }

    /**
     * Sets the ViewObject's type.
     * 
     * @param newType
     *            The ViewObject's new type
     */
    public void setType(ViewType newType)
    {
        this.type = newType;
    }

    /**
     * Prints the ViewObject object in a human-readable form. Overrides java.lang.Object#toString().
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();
        formatter.addItem("ID", id);
        formatter.addItem("Name", name);
        formatter.addItem("Type", type);
        formatter.addItem("Default", isDefault);
        formatter.addItem("Engine", engine);
        formatter.addItem("Target", target);

        return formatter.getFormatedObject();
    }
}
