/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.dem.objects.parameters.errors;

import java.util.ArrayList;
import java.util.List;

import pt.digitalis.dif.dem.objects.parameters.constraints.IParameterConstraint;
import pt.digitalis.dif.dem.objects.parameters.constraints.ParameterConstraintResult;
import pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule;
import pt.digitalis.dif.dem.objects.parameters.validators.IParameterValidator;
import pt.digitalis.dif.utils.ObjectFormatter;

/**
 * Defines a type that represents an error on a parameter.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 */
public class ParameterError {

    /** The constraint that caused the error */
    private IParameterConstraint constraint;

    /** The parameter error IDs */
    private List<String> errorIDs = new ArrayList<String>();

    /** The type of the error */
    private ParameterErrorType errorType;

    /** The generated exception if it is the case */
    private Exception exception;

    /** Always report these errors */
    private boolean forceShowError = false;

    /** The id of the error. ID of the constrain/validator or key that identifies the other kind of error */
    private String id;

    /**
     * The message that describes the error
     */
    private String message;

    /** The rule that caused the error */
    private IParameterRule<?> rule;

    /** The validator that caused the error */
    private IParameterValidator validator;

    /**
     * Constructor
     */
    public ParameterError()
    {
        super();
    }

    /**
     * Constructor
     * 
     * @param constraintId
     *            The constraint Id
     * @param constraintResult
     *            The constraint result
     */
    public ParameterError(String constraintId, ParameterConstraintResult constraintResult)
    {
        if (constraintResult.getLanguage() != null)
        {
            this.message = constraintResult.getConstraint().validationErrorMessage(constraintResult.getLanguage());
        }
        else
        {
            this.message = constraintResult.getConstraint().validationErrorMessage();
        }

        this.errorType = ParameterErrorType.CONSTRAINT;
        this.id = constraintId;
        this.constraint = constraintResult.getConstraint();
        this.validator = null;
        if (constraintResult.getErrorIDs() != null)
        {
            this.errorIDs.addAll(constraintResult.getErrorIDs());
        }
    }

    /**
     * Constructor.
     * 
     * @param message
     *            the error message.
     * @param errorType
     *            the error type
     */
    public ParameterError(String message, ParameterErrorType errorType)
    {
        this.message = message;
        this.errorType = errorType;
        this.id = null;
        this.constraint = null;
        this.validator = null;
    }

    /**
     * Inspector for the error constraint.
     * 
     * @return the constraint the constraint
     */
    public IParameterConstraint getConstraint()
    {
        return constraint;
    }

    /**
     * Inspector for the 'errorIDs' attribute.
     * 
     * @return the errorIDs value
     */
    public List<String> getErrorIDs()
    {
        return errorIDs;
    }

    /**
     * Inspector for the error type.
     * 
     * @return the errorType the error type
     */
    public ParameterErrorType getErrorType()
    {
        return errorType;
    }

    /**
     * Inspector for the error exception.
     * 
     * @return the exception the error exception
     */
    public Exception getException()
    {
        return exception;
    }

    /**
     * Inspector for the error ID.
     * 
     * @return the id the error ID.
     */
    public String getId()
    {
        return id;
    }

    /**
     * Inspector for the error message.
     * 
     * @return the message the error message
     */
    public String getMessage()
    {
        return message;
    }

    /**
     * Inspector for the 'rule' attribute.
     * 
     * @return the rule value
     */
    public IParameterRule<?> getRule()
    {
        return rule;
    }

    /**
     * Inspector for the validator.
     * 
     * @return the validator
     */
    public IParameterValidator getValidator()
    {
        return validator;
    }

    /**
     * Inspector for the 'forceShowError' attribute.
     * 
     * @return the forceShowError value
     */
    public boolean isForceShowError()
    {
        return forceShowError;
    }

    /**
     * Modifier for the constraint.
     * 
     * @param id
     *            the id of the constraint
     * @param constraint
     *            the constraint to set
     */
    public void setConstraint(String id, IParameterConstraint constraint)
    {
        this.id = id;
        this.constraint = constraint;
    }

    /**
     * Modifier for the 'errorIDs' attribute.
     * 
     * @param errorIDs
     *            the new errorIDs value to set
     */
    public void setErrorIDs(List<String> errorIDs)
    {
        this.errorIDs = errorIDs;
    }

    /**
     * Modifier for the exception.
     * 
     * @param exception
     *            the exception to set
     */
    public void setException(Exception exception)
    {
        this.exception = exception;
    }

    /**
     * Modifier for the 'forceShowError' attribute.
     * 
     * @param forceShowError
     *            the new forceShowError value to set
     */
    public void setForceShowError(boolean forceShowError)
    {
        this.forceShowError = forceShowError;
    }

    /**
     * Inspector for the error ID.
     * 
     * @param newID
     *            the new ID.
     */
    public void setId(String newID)
    {
        this.id = newID;
    }

    /**
     * Modifier for the 'message' attribute.
     * 
     * @param message
     *            the new message value to set
     */
    public void setMessage(String message)
    {
        this.message = message;
    }

    /**
     * Modifier for the 'rule' attribute.
     * 
     * @param rule
     *            the new rule value to set
     */
    public void setRule(IParameterRule<?> rule)
    {
        this.rule = rule;
    }

    /**
     * Modifier for the validator.
     * 
     * @param id
     *            the id of the validator
     * @param validator
     *            the validator to set
     */
    public void setValidator(String id, IParameterValidator validator)
    {
        this.id = id;
        this.validator = validator;
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();
        formatter.addItem("Message", message);
        formatter.addItem("Error Type", errorType);
        formatter.addItemIfNotNull("ID", id);
        formatter.addItemIfNotNull("Constraint", constraint);
        formatter.addItemIfNotNull("Validator", validator);
        formatter.addItemIfNotNull("Exception", exception);

        return formatter.getFormatedObject();
    }
}
