/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dem.objects.parameters.rules;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.dem.interfaces.IStageInstance;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.dem.objects.parameters.IParameter;
import pt.digitalis.dif.exception.objects.ParameterException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.utils.common.StringUtils;

/**
 * A base implementation for all parameter rules
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @param <T>
 *            the parameter base type
 * @created Mar 31, 2009
 */
public abstract class AbstractParameterRule<T> implements IParameterRule<T> {

    /** The rule action */
    private ParameterRuleAction action;

    /** The description for end value */
    private String descriptionEndValue = null;

    /** The description for start value */
    private String descriptionStartValue = null;

    /** The description value */
    private String descriptionValue = null;

    /** The end value for validations */
    private String endValue = null;

    /** The parameter that this rule is associated to */
    protected String parameterID;

    /** the rule parameters */
    private List<String> parameters = new ArrayList<String>();

    /** The start value for validations */
    private String startValue = null;

    /** The value for validation */
    private String value = null;

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getAction()
     */
    public ParameterRuleAction getAction()
    {
        return action;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getDescriptionEndValue()
     */
    public String getDescriptionEndValue()
    {
        return descriptionEndValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getDescriptionStartValue()
     */
    public String getDescriptionStartValue()
    {
        return descriptionStartValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getDescriptionValue()
     */
    public String getDescriptionValue()
    {
        return descriptionValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getEndValue()
     */
    public String getEndValue()
    {
        return endValue;
    }

    /**
     * Returns the list of messages available for the parameter rule
     * 
     * @param stage
     *            the stage instance that the parameter is associated to
     * @return the parameter rule Messages
     */
    protected Map<String, String> getMessages(IStageInstance stage)
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class)
                .collectEntityMessagesFromRepository(this.getClass()).getMessages(stage.getContext().getLanguage());
    }

    /**
     * Inspector for the 'parameter' attribute.
     * 
     * @param stage
     *            the stage instance that the parameter is associated to
     * @return the parameter value
     * @throws ParameterException
     *             if the parameter does not exist
     */
    @SuppressWarnings("unchecked")
    public IParameter<T> getParameter(IStageInstance stage) throws ParameterException
    {
        return (IParameter<T>) stage.getParameters().getAllAvailableParameters().getParameter(parameterID);
    }

    /**
     * Get's the name of a given parameter from the stage messages. When missing the parameter ID is returned (camel
     * case interpreted)
     * 
     * @param stage
     *            the parameter's current stage
     * @param parameterID
     *            the parameter ID to fetch the name
     * @return the parameter name
     */
    public String getParameterName(IStageInstance stage, String parameterID)
    {
        String message = stage.getMessage(parameterID);

        if (message == null)
            message = StringUtils.camelCaseToString(parameterID);

        return message;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getParameters()
     */
    public List<String> getParameters()
    {
        return parameters;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getStartValue()
     */
    public String getStartValue()
    {
        return startValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#getValue()
     */
    public String getValue()
    {
        return value;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#init(java.lang.String, java.lang.String,
     *      pt.digitalis.dif.dem.objects.parameters.rules.ParameterRuleAction, java.lang.String, java.lang.String,
     *      java.lang.String)
     */
    public AbstractParameterRule<T> init(String parameterID, String parameters, ParameterRuleAction action,
            String value, String first, String last)
    {
        this.parameterID = parameterID;
        this.action = action;

        if (parameters != null && !"".equals(parameters))
        {
            List<String> parametersPassed = Arrays.asList(parameters.split(","));

            for (String parameterPassed: parametersPassed)
                this.parameters.add(parameterPassed.trim().toLowerCase());
        }

        if (value != null && !value.contains(AnnotationTags.NONE.toString()) && value.contains(":"))
        {
            String[] splitCommaValues = value.split(",");
            for (String v: splitCommaValues)
            {
                String[] splitValue = v.split(":");
                if (this.value != null)
                {
                    this.value += ",";
                }
                else
                {
                    this.value = "";
                }
                this.value += splitValue[0];

                if (this.descriptionValue != null)
                {
                    this.descriptionValue += ",";
                }
                else
                {
                    this.descriptionValue = "";
                }
                this.descriptionValue += splitValue[1];
            }
        }
        else
        {
            this.value = value;
        }

        if (first != null && !first.contains(AnnotationTags.NONE.toString()) && first.contains(":"))
        {
            String[] splitValue = first.split(":");
            this.startValue = splitValue[0];
            this.descriptionStartValue = splitValue[1];
        }
        else
        {
            this.startValue = first;
        }

        if (last != null && !last.contains(AnnotationTags.NONE.toString()) && last.contains(":"))
        {
            String[] splitValue = last.split(":");
            this.endValue = splitValue[0];
            this.descriptionEndValue = splitValue[1];
        }
        else
        {
            this.endValue = last;
        }

        return this;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setDescriptionEndValue(java.lang.String)
     */
    public void setDescriptionEndValue(String descriptionEndValue)
    {
        this.descriptionEndValue = descriptionEndValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setDescriptionStartValue(java.lang.String)
     */
    public void setDescriptionStartValue(String descriptionStartValue)
    {
        this.descriptionStartValue = descriptionStartValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setDescriptionValue(java.lang.String)
     */
    public void setDescriptionValue(String descriptionValue)
    {
        this.descriptionValue = descriptionValue;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setEndValue(java.lang.String)
     */
    public AbstractParameterRule<T> setEndValue(String endValue)
    {
        this.endValue = endValue;

        return this;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setStartValue(java.lang.String)
     */
    public AbstractParameterRule<T> setStartValue(String startValue)
    {
        this.startValue = startValue;

        return this;
    }

    /**
     * @see pt.digitalis.dif.dem.objects.parameters.rules.IParameterRule#setValue(java.lang.String)
     */
    public AbstractParameterRule<T> setValue(String value)
    {
        this.value = value;

        return this;
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString()
    {
        ObjectFormatter formatter = new ObjectFormatter();
        formatter.addItem("Action", this.action);
        formatter.addItem("End Value", this.endValue);
        formatter.addItem("Parameter ID", this.parameterID);
        formatter.addItem("Parameters", this.parameters);
        formatter.addItem("Start Value", this.startValue);
        formatter.addItem("Value", this.value);
        formatter.addItem("Description Value", this.descriptionValue);
        formatter.addItem("Description Start Value", this.descriptionStartValue);
        formatter.addItem("Description End Value", this.descriptionEndValue);

        return formatter.getFormatedObject();
    }
}
