/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.sanitycheck.manager;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.sanitycheck.ExecutionResult;
import pt.digitalis.dif.sanitycheck.ISanityCheckTestSuite;
import pt.digitalis.dif.sanitycheck.SanityCheckResult;
import pt.digitalis.dif.sanitycheck.TestResult;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.log.LogLevel;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 14 de Jul de 2011
 */
public class SanityCheckManager {

    /** Sanity check Logger Message Prefix */
    protected static final String _SANITY_CHECK_LOGGER_PREFIX = "Sanity Check: ";

    /** Sanity check Logger Message */
    protected static final String CIRCULAR_DEPENDENCY = _SANITY_CHECK_LOGGER_PREFIX
            + "Circular dependency detected on TestSuite \"%1\" and \"%2\"";

    /** the test execution order according to the declared dependencies */
    private static List<String> testSuiteExecutionOrder = null;

    /** the test suite ordered list after parsing all contributed ISanityCheckTestSuite */
    static private Map<String, SanityCheckTestSuiteDefinition> testSuites = null;

    /**
     * Inspector for the 'testSuiteExecutionOrder' attribute.
     * 
     * @return the testSuiteExecutionOrder value
     */
    private static List<String> getTestSuiteExecutionOrder()
    {
        if (testSuiteExecutionOrder == null)
            initialize();

        return testSuiteExecutionOrder;
    }

    /**
     * Inspector for the 'testSuites' attribute.
     * 
     * @return the testSuites value
     */
    public static Map<String, SanityCheckTestSuiteDefinition> getTestSuites()
    {
        if (testSuites == null)
            initialize();

        return testSuites;
    }

    /**
     * Initializes the manager. <br/>
     * Will read all declared {@link ISanityCheckTestSuite} classes and mount the test suites and their attributes and
     * dependencies.
     */
    private static void initialize()
    {
        // Test suite registry holder
        testSuites = new HashMap<String, SanityCheckTestSuiteDefinition>();

        // Get all ISanityCheckTestSuite contributions from the IoC
        Map<String, ISanityCheckTestSuite> suiteEntries = DIFIoCRegistry.getRegistry().getImplementationsMap(
                ISanityCheckTestSuite.class);

        // Add all to the registry attribute "testSuites"
        for (Entry<String, ISanityCheckTestSuite> suiteEntry: suiteEntries.entrySet())
        {
            // Create a definition object, that will parse and determine execution for each test suite
            if (suiteEntry.getValue() instanceof ISanityCheckTestSuite && suiteEntry.getValue().isActive())
                testSuites.put(suiteEntry.getKey(),
                        new SanityCheckTestSuiteDefinition(suiteEntry.getKey(), suiteEntry.getValue()));
            else
                DIFLogger.getLogger().warn(
                        _SANITY_CHECK_LOGGER_PREFIX + "Class \"" + suiteEntry.getValue().getClass().getSimpleName()
                                + "\" must implement " + ISanityCheckTestSuite.class.getSimpleName());
        }

        // Determine the running order according to the determined dependencies
        testSuiteExecutionOrder = new ArrayList<String>();

        // For all existing parse recursively through all dependencies
        for (SanityCheckTestSuiteDefinition testSuite: testSuites.values())
            if (!testSuiteExecutionOrder.contains(testSuite.getId()))
                // New testSuite. Add all dependencies which are not already declared
                testSuiteExecutionOrder = parseInheritedDependencies(testSuiteExecutionOrder, testSuite);
    }

    /**
     * Parses for all dependent test suites and their recursive dependencies.<br/>
     * Will detect and ignore circular dependencies.<br/>
     * <br/>
     * The provided temporaryDependencies attribute is used to detect already declared testSuites and as such no need to
     * crawl inside their dependency tree.
     * 
     * @param temporaryDependencies
     *            currently already parsed dependencies
     * @param stackDependencyParsing
     *            a stack of the current dependencies in recursive analisis. Used to prevent circular dependencies
     * @param suiteDefinition
     *            the suite to check dependencies
     * @return the current test suite recursive dependencies list
     */
    private static List<String> parseInheritedDependencies(List<String> temporaryDependencies,
            List<String> stackDependencyParsing, SanityCheckTestSuiteDefinition suiteDefinition)
    {
        for (SanityCheckTestSuiteDefinition dependentSuide: suiteDefinition.getDependencies())
        {
            if (stackDependencyParsing.contains(dependentSuide.getId()))
            {
                // The dependency is already referenced in the currently executing stack. Circular dependency.
                // Report and ignore
                DIFLogger.getLogger().warn(
                        CIRCULAR_DEPENDENCY.replace("%1", suiteDefinition.getId())
                                .replace("%2", dependentSuide.getId()));

            }
            else if (!temporaryDependencies.contains(dependentSuide.getId()))
            {
                // Add recursive dependencies
                temporaryDependencies = parseInheritedDependencies(temporaryDependencies, dependentSuide);

                // Add the current dependency
                temporaryDependencies.add(dependentSuide.getId());
            }
        }

        temporaryDependencies.add(suiteDefinition.getId());

        return temporaryDependencies;
    }

    /**
     * Parses for all dependent test suites and their recursive dependencies.<br/>
     * Will detect and ignore circular dependencies.<br/>
     * <br/>
     * The provided temporaryDependencies attribute is used to detect already declared testSuites and as such no need to
     * crawl inside their dependency tree.
     * 
     * @param temporaryDependencies
     *            currently already parsed dependencies
     * @param suiteDefinition
     *            the suite to check dependencies
     * @return the current test suite recursive dependencies list
     */
    private static List<String> parseInheritedDependencies(List<String> temporaryDependencies,
            SanityCheckTestSuiteDefinition suiteDefinition)
    {
        return parseInheritedDependencies(temporaryDependencies, new ArrayList<String>(), suiteDefinition);
    }

    /**
     * Execute all tests
     * 
     * @param context
     *            the current DIF context
     * @return T if all tests ran successfully
     */
    public static boolean runAllTests(IDIFContext context)
    {
        return runAllTests(context, false);
    }

    /**
     * Execute all tests.
     * 
     * @param context
     *            the current DIF context
     * @param testOnlyErrors
     *            the test only errors
     * @return T if all tests ran successfully
     */
    public static boolean runAllTests(IDIFContext context, boolean testOnlyErrors)
    {
        boolean result = true;

        DIFLogger.getLogger().info(_SANITY_CHECK_LOGGER_PREFIX + "Starting Sanity Check tests...");

        // Execute all testSuites, by the correct order
        for (String testSuiteID: getTestSuiteExecutionOrder())
        {
            SanityCheckTestSuiteDefinition testSuide = getTestSuites().get(testSuiteID);
            ExecutionResult testsResult = runTestSuiteTests(testSuide, context).getExecutionsResult();
            boolean testsOK = (testOnlyErrors ? (testsResult != ExecutionResult.FAILED)
                    : (testsResult == ExecutionResult.PASSED));

            // First error, and log is not in INFO, so start tests message not rendered previously.
            // Print warn heading...
            if (!DIFLogger.getLogger().isInfoEnabled() && result && !testsOK)
                DIFLogger.getLogger().warn(_SANITY_CHECK_LOGGER_PREFIX + "Starting Sanity Check tests...");

            // Log according to result...
            DIFLogger.getLogger().log(testsOK ? LogLevel.INFO : LogLevel.WARN,
                    "     " + testSuide.getName() + " (" + testSuiteID + "): " + testsResult.toString());

            result = result && testsOK;
            if (testOnlyErrors && !result)
            {
                break;
            }
        }

        // Feedback, according to the result
        if (result)
            DIFLogger.getLogger().info(_SANITY_CHECK_LOGGER_PREFIX + "All Sanity Checks ran with Success.");
        else
            DIFLogger.getLogger().warn(_SANITY_CHECK_LOGGER_PREFIX + "Errors while running Sanity Checks.");

        return result;
    }

    /**
     * Run all tests of the {@link ISanityCheckTestSuite}
     * 
     * @param suiteDefinition
     *            the test suite to run the tests
     * @param context
     *            the current DIF context
     * @return the executing result
     */
    private static SanityCheckResult runTestSuiteTests(SanityCheckTestSuiteDefinition suiteDefinition,
            IDIFContext context)
    {
        suiteDefinition.setExecutionResult(new SanityCheckResult(suiteDefinition, ExecutionResult.EXECUTING));
        suiteDefinition.getExecutionResult().setStartTime(new Date());

        try
        {
            suiteDefinition.runTestsSetup();

            boolean warnings = false;
            boolean errors = false;

            // Run all tests, adding each result to the execution result...
            for (TestMethodDefinition test: suiteDefinition.getTestMethods())
            {
                TestResult testResult = test.run(context);

                // Set the test name if empty to the test method
                if (testResult.getName() == null)
                    testResult.setName(test.getName());

                errors = errors || testResult.getExecutionResult() == ExecutionResult.FAILED;
                warnings = warnings || testResult.getExecutionResult() == ExecutionResult.WARNING;

                suiteDefinition.getExecutionResult().addTestResult(testResult);
            }

            suiteDefinition.runTestsFinalize();

            suiteDefinition.getExecutionResult().setExecutionsResult(
                    errors ? ExecutionResult.FAILED : warnings ? ExecutionResult.WARNING : ExecutionResult.PASSED);
        }
        catch (Exception e)
        {
            e.printStackTrace();

            suiteDefinition.getExecutionResult().setExecutionsResult(ExecutionResult.FAILED);
            suiteDefinition.getExecutionResult().setExecutionError(e.getMessage());
        }
        suiteDefinition.getExecutionResult().setEndTime(new Date());

        return suiteDefinition.getExecutionResult();
    }
}
