/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.startup;

import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Stack;

import pt.digitalis.dif.codegen.DIFCodeGenerator;
import pt.digitalis.dif.controller.interfaces.IModelManager;
import pt.digitalis.dif.controller.security.managers.IAuthenticationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.impl.IdentityManagerStaticImpl;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IApplicationConfiguration;
import pt.digitalis.dif.dem.interfaces.IApplicationPrivate;
import pt.digitalis.dif.dem.interfaces.IDIFAPI;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.dem.managers.impl.UsageIssuesManagerImpl;
import pt.digitalis.dif.dem.objects.issues.UsageIssue;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.exception.codegen.DIFCodeGenerationException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.logging.AuditContext;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;
import pt.digitalis.utils.ioc.IIoCRegistry;

/**
 * This class contains the initialization logic to set the framework ready to serve requests. Listeners must check if
 * the framework is ready to serve the request before submitting a new request for service.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Oct 8, 2007
 */
final public class DIFInitializer {

    /** The indentation character. */
    final static private String INDENTATION = "  ";

    /** Initialization state. */
    static private boolean initialized = false;

    /**
     * Adds the issues of a given location to the buffer.
     * 
     * @param location
     *            the location
     * @param buffer
     *            the buffer
     * @param indentPrefix
     *            the indentation prefix string or output
     */
    static private void addIssuesToBuffer(String location, StringBuffer buffer, String indentPrefix)
    {
        // Reuse var
        List<UsageIssue> issues = null;

        issues = UsageIssuesManagerImpl.getInstance().getIssues(location);

        for (UsageIssue usageIssue: issues)
            buffer.append(indentPrefix + "  => " + usageIssue.getIssueType() + ": " + usageIssue.getIssueDescription()
                    + "\n");
    }

    /**
     * Dumps the DEM.
     */
    static private void dumpDEM()
    {
        // When in debug dump the full DEM structure to the log.
        StringBuffer buffer = new StringBuffer();
        Stack<String> indentationsStack = new Stack<String>();
        String currentIndentation = null;

        buffer.append("\n\nDumping DEM structure:\n");
        buffer.append("-----------------------------------------\n");

        IDEMManager dem = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);

        Iterator<IProvider> providers = dem.getProviders().values().iterator();

        while (providers.hasNext())
        {
            IProvider provider = providers.next();

            if (providers.hasNext())
            {
                currentIndentation = INDENTATION + "|   ";
                buffer.append(INDENTATION + "|-- Provider: " + provider.getName() + " (" + provider.getID() + ")\n");
            }
            else
            {
                currentIndentation = INDENTATION + "    ";
                buffer.append(INDENTATION + "`-- Provider: " + provider.getName() + " (" + provider.getID() + ")\n");
            }

            if (UsageIssuesManagerImpl.getInstance().hasIssues(provider.getOriginalClassName()))
                addIssuesToBuffer(provider.getOriginalClassName(), buffer, currentIndentation);

            // Applications iteration
            Iterator<IApplication> applications = provider.getApplications().values().iterator();

            while (applications.hasNext())
            {
                IApplication application = applications.next();
                indentationsStack.add(currentIndentation);

                if (applications.hasNext())
                {
                    buffer.append(currentIndentation + INDENTATION + "|-- Application: " + application.getName() + " ("
                            + application.getID() + ")\n");
                    currentIndentation += INDENTATION + "|   ";
                }
                else
                {
                    buffer.append(currentIndentation + INDENTATION + "`-- Application: " + application.getName() + " ("
                            + application.getID() + ")\n");
                    currentIndentation += INDENTATION + "    ";
                }

                if (UsageIssuesManagerImpl.getInstance().hasIssues(application.getOriginalClassName()))
                    addIssuesToBuffer(application.getOriginalClassName(), buffer, currentIndentation);

                // Services iteration
                Iterator<IService> services = application.getServices().values().iterator();

                while (services.hasNext())
                {
                    IService service = services.next();
                    indentationsStack.add(currentIndentation);

                    if (services.hasNext())
                    {
                        buffer.append(currentIndentation + INDENTATION + "|-- Service: " + service.getName() + " ("
                                + service.getID() + ")\n");
                        currentIndentation += INDENTATION + "|   ";
                    }
                    else
                    {
                        buffer.append(currentIndentation + INDENTATION + "`-- Service: " + service.getName() + " ("
                                + service.getID() + ")\n");
                        currentIndentation += INDENTATION + "    ";
                    }

                    if (UsageIssuesManagerImpl.getInstance().hasIssues(service.getOriginalClassName()))
                        addIssuesToBuffer(service.getOriginalClassName(), buffer, currentIndentation);

                    // Stages iteration
                    Iterator<IStage> stages = service.getStages().values().iterator();

                    while (stages.hasNext())
                    {
                        IStage stage = stages.next();
                        indentationsStack.add(currentIndentation);

                        if (stages.hasNext())
                        {
                            buffer.append(currentIndentation + INDENTATION + "|-- " + stage.getName() + " ("
                                    + stage.getID() + ")\n");
                            currentIndentation += INDENTATION + "|   ";
                        }
                        else
                        {
                            buffer.append(currentIndentation + INDENTATION + "`-- " + stage.getName() + " ("
                                    + stage.getID() + ")\n");
                            currentIndentation += INDENTATION + "    ";
                        }

                        if (UsageIssuesManagerImpl.getInstance().hasIssues(stage.getOriginalClassName()))
                            addIssuesToBuffer(stage.getOriginalClassName(), buffer, currentIndentation);

                        currentIndentation = indentationsStack.pop();
                    }

                    buffer.append(currentIndentation + "\n");
                    currentIndentation = indentationsStack.pop();
                }

                currentIndentation = indentationsStack.pop();
            }
        }

        buffer.append("-----------------------------------------\n\n");

        if (DIFStartupConfiguration.getLogLevel().equals(LogLevel.DEBUG))
            DIFLogger.getLogger().debug(buffer.toString());
        else
            DIFLogger.getLogger().info(buffer.toString());
    }

    /**
     * DIF initialization method. Defines the order of the initialization steps.
     * 
     * @param initDEMInstances
     *            if T searches for DEM instances and initializes then
     * @param upgradeModelManagers
     *            if T upgrades all Model Manager contributions
     * @return T if initialization has successfully completed (or if has been previously done), F otherwise
     */
    synchronized static public boolean initialize(boolean initDEMInstances, boolean upgradeModelManagers)
    {
        if (!isInitialized())
        {
            try
            {
                // IoC initialization (includes module discovery)
                IIoCRegistry iocRegistry = DIFIoCRegistry.getRegistry();

                // Startup the Identity and Authorization since they will be called form the CG process of the DEM
                iocRegistry.getImplementation(IIdentityManager.class);
                iocRegistry.getImplementation(IAuthenticationManager.class);

                /*
                 * ATTENTION !!! ......................................................................................
                 * The model upgrade must be before all things, mainly because:........................................
                 * - code generation may need to invoke identity manager ..............................................
                 * - application init usually initializes all models ..................................................
                 */
                if (upgradeModelManagers)
                {
                    // Upgrades all database model, declared to the current application
                    DIFLogger.getLogger().info("Initializing Model Manager upgrades...");
                    DIFLogger.getLogger().increaseIndentation();
                    for (IModelManager modelManager: DIFIoCRegistry.getRegistry().getImplementations(
                            IModelManager.class))
                    {
                        if (modelManager.isEnabled())
                        {
                            if (!modelManager.isUpToDate())
                            {
                                String currentVersion = modelManager.getCurrentVersion();
                                DIFLogger.getLogger().info(
                                        "Model '" + modelManager.getSchema() + "': Upgrading from " + currentVersion);

                                modelManager.updateVersion();
                                DIFLogger.getLogger().info(
                                        "Model '" + modelManager.getSchema() + "': Upgraded to "
                                                + modelManager.getCurrentVersion());
                            }
                            else
                            {
                                DIFLogger.getLogger().info("Model '" + modelManager.getSchema() + "': Up to date");
                            }
                        }
                    }
                    DIFLogger.getLogger().decreaseIndentation();
                }

                // Create the DIF code generator and inject it's dependencies into it
                DIFCodeGenerator codeGen = new DIFCodeGenerator();
                iocRegistry.injectDependencies(codeGen);

                if (initDEMInstances)
                {
                    // Discover all registered DEM entities packages
                    codeGen.collectRegisteredPackages();

                    // Search registered entities
                    codeGen.searchRegisteredPackages();

                    // Enhance the registered entities
                    codeGen.enhanceDEMClasses();
                }

                // Log all issues if in debug
                if (DIFStartupConfiguration.getLogLevel().equals(LogLevel.DEBUG))
                    UsageIssuesManagerImpl.getInstance().logAllIssuesByType();

                // Clean up all temporary data
                codeGen.cleanUp();

                // Initialize the declared APIs
                initializeAPIs();

                // Run the applications initialization procedures
                initializeApplications();

                // If the identityManager is the static implementation, call the static users initialization
                IIdentityManager identityManager = DIFIoCRegistry.getRegistry().getImplementation(
                        IIdentityManager.class);
                if (identityManager instanceof IdentityManagerStaticImpl)
                    ((IdentityManagerStaticImpl) identityManager).initializeStaticCustomUsers();

                initialized = true;

                // Dump DEM structure with associated issues (if any), if DiF is running in DEBUG mode or if there are
                // issues
                if (DIFStartupConfiguration.getLogLevel().equals(LogLevel.DEBUG)
                        || UsageIssuesManagerImpl.getInstance().hasIssues()
                        || DIFStartupConfiguration.getDeveloperMode() || DIFStartupConfiguration.getTestingMode())
                {
                    dumpDEM();
                }

            }
            catch (ResourceNotFoundException resourceNotFoundException)
            {
                resourceNotFoundException.printStackTrace();

            }
            catch (CodeGenerationException codeGenerationException)
            {
                codeGenerationException.printStackTrace();
            }
            catch (DIFCodeGenerationException e)
            {
                e.printStackTrace();
            }
            catch (InternalFrameworkException e)
            {
                throw new RuntimeException(e);
            }

        }
        return initialized;
    }

    /**
     * Initialize APIs.
     * 
     * @throws InternalFrameworkException
     */
    private static void initializeAPIs() throws InternalFrameworkException
    {
        List<IDIFAPI> list = DIFIoCRegistry.getRegistry().getImplementations(IDIFAPI.class);

        Collections.sort(list, new Comparator<IDIFAPI>() {

            public int compare(IDIFAPI a, IDIFAPI b)
            {
                return a.order().compareTo(b.order());
            }
        });

        for (IDIFAPI difApi: list)
        {
            difApi.initialize();
        }

    }

    /**
     * Initialize all registered DIF Applications.
     */
    synchronized static public void initializeApplications()
    {
        List<IApplicationConfiguration> appConfImpls = DIFIoCRegistry.getRegistry().getImplementations(
                IApplicationConfiguration.class);
        if (appConfImpls.size() > 0)
        {
            appConfImpls.get(0).processConfigurations();
            DIFLogger.getLogger()
                    .info("Initializing Applications Configurations using: "
                            + appConfImpls.get(0).getClass().getSimpleName());
        }

        IDEMManager demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);
        DIFLogger.getLogger().info("Initializing Applications...");

        for (IProvider provider: demManager.getProviders().values())
            for (IApplication application: provider.getApplications().values())
            {
                AuditContext.setUserForCurrentThread("DIF");
                AuditContext.setProcessNameForCurrentThread("DIFInitialize:" + application.getName());

                ((IApplicationPrivate) application).__CG__initialize();
            }
    }

    /**
     * Initialization state inspector.
     * 
     * @return T if the framework has been initialized
     */
    synchronized static public boolean isInitialized()
    {
        return initialized;
    }
}
