/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.utils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import pt.digitalis.utils.common.StringUtils;

/**
 * Helps to format an object for printing in human readable form
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Dec 7, 2007
 */
public class ObjectFormatter {

    /** The prefix to format the lines with */
    final static private String PREFIX = "  | ";

    /** The prefix to format the lines with */
    final static private String SEPARATOR = ": ";

    /** The generated buffer to return. Will be built incrementally */
    private StringBuffer buffer = new StringBuffer();

    /** the dumpd object list to keep track (avoid circular dump and stack overflow) */
    private List<Object> dumpedObjects = new ArrayList<Object>();

    /**
     * Adds an item to the buffer
     * 
     * @param key
     *            the key that identifies the value
     * @param value
     *            the value
     * @return the updated instance (fluent interface)
     */
    public ObjectFormatter addItem(String key, Object value)
    {
        if (!dumpedObjects.contains(value)
                || (value != null && value.getClass().getPackage().getName().startsWith("java.lang")))
        {
            if (key.contains("pass"))
                value = "*****";

            String line = PREFIX + key + SEPARATOR;
            line += toStringParser(value, line.length());

            if (!"".equals(buffer))
                buffer.append("\n");

            buffer.append(line);
        }

        return this;
    }

    /**
     * Adds an item to the buffer if the value is not null
     * 
     * @param key
     *            the key that identifies the value
     * @param value
     *            the value
     * @return the updated instance (fluent interface)
     */
    public ObjectFormatter addItemIfNotNull(String key, Object value)
    {
        if (value != null)
            addItem(key, value);

        return this;
    }

    /**
     * Formats an iterator in human readable form
     * 
     * @param obj
     *            the list to convert
     * @return the formatted list
     */
    private String formatIterator(Iterable<?> obj)
    {
        if (!dumpedObjects.contains(obj))
        {
            dumpedObjects.add(obj);

            Iterator<?> iterator = ((Iterable<?>) obj).iterator();

            if (!iterator.hasNext())
                return "[]";
            else
            {
                String temp = "[\n";
                int pos = 1;

                while (iterator.hasNext())
                {
                    Object object = iterator.next();
                    String ident = " #" + StringUtils.fillStringLeft(Integer.toString(pos++), 2, " ") + " ";
                    temp += ident + toStringParserWithPrefix(object, ident.length(), " ") + "\n";
                }

                return temp + "]";
            }
        }
        else
            return "see above";
    }

    /**
     * Formats a map in human readable form
     * 
     * @param map
     *            the map to convert
     * @return the formatted map
     */
    private String formatMap(Map<?, ?> map)
    {
        if (!dumpedObjects.contains(map))
        {
            dumpedObjects.add(map);

            if (map.size() == 0)
                return "[]";
            else
            {
                String temp = "[\n";
                int pos = 1;

                for (Object key: map.keySet())
                {
                    String ident = " " + StringUtils.fillStringLeft("#" + Integer.toString(pos++), 3, " ") + " " + key
                            + " = ";
                    Object value;

                    if (key != null && key.toString().contains("pass"))
                        value = "*****";
                    else
                        value = map.get(key);

                    if (value instanceof String)
                        value = "\"" + value + "\"";

                    temp += ident + toStringParserWithPrefix(value, ident.length(), " ") + "\n";
                }

                return temp + "]";
            }
        }
        else
            return "see above";
    }

    /**
     * @return the formatted object
     */
    public String getFormatedObject()
    {
        return buffer.toString();
    }

    /**
     * Parses a string and tries to format it for including has a value in a line.
     * 
     * @param obj
     *            the object to format
     * @param prefixSize
     *            the size of the necessary prefix
     * @return the formatted string
     */
    private String toStringParser(Object obj, int prefixSize)
    {
        return toStringParserWithPrefix(obj, prefixSize, PREFIX);
    }

    /**
     * Parses a string and tries to format it for including has a value in a line.
     * 
     * @param obj
     *            the object to format
     * @param prefixSize
     *            the size of the necessary prefix
     * @param prefix
     *            the prefix to use to pad the string
     * @return the formatted string
     */
    private String toStringParserWithPrefix(Object obj, int prefixSize, String prefix)
    {
        if (obj == null)
            return null;
        else
        {
            String temp;

            if (obj instanceof Map)
                temp = formatMap((Map<?, ?>) obj);

            else if (obj instanceof Iterable)
                temp = formatIterator(((Iterable<?>) obj));

            else
                temp = obj.toString();

            dumpedObjects.add(obj);

            return temp.replace("\n", "\n" + StringUtils.fillString(prefix, prefixSize, " "));
        }
    }
}
