/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.extensions.cms;

import java.util.List;
import java.util.Set;

import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentItemNotFoundException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentItemWithDuplicateNameAndParentNodeException;
import pt.digitalis.dif.utils.extensions.cms.exception.ContentManagerException;
import pt.digitalis.dif.utils.extensions.cms.exception.InvalidNameException;
import pt.digitalis.dif.utils.extensions.cms.exception.InvalidParentNodeException;
import pt.digitalis.dif.utils.extensions.cms.exception.InvalidPathException;
import pt.digitalis.dif.utils.extensions.cms.exception.NoAccessException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeNotFoundException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeWithDuplicatePathException;
import pt.digitalis.dif.utils.extensions.cms.exception.NodeWithNodesException;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Jun 5, 2009
 */
public interface IContentManager {

    /**
     * Adds a new content to the repository.<br>
     * Will assign the ID internally
     * 
     * @param content
     *            the content to add in the repository
     * @return the updated content
     * @throws InvalidParentNodeException
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentItemWithDuplicateNameAndParentNodeException
     * @throws ContentManagerException
     */
    public ContentItem addContent(ContentItem content) throws InvalidParentNodeException, NodeNotFoundException,
            NoAccessException, ContentItemWithDuplicateNameAndParentNodeException, ContentManagerException;

    /**
     * Adds a new content to the repository<br>
     * Will assign the ID internally
     * 
     * @param node
     *            the node to add
     * @return the added node
     * @throws InvalidPathException
     * @throws InvalidNameException
     * @throws NoAccessException
     * @throws InvalidParentNodeException
     * @throws NodeNotFoundException
     * @throws NodeWithDuplicatePathException
     * @throws ContentManagerException
     */
    public Node addNode(Node node) throws InvalidPathException, InvalidNameException, NoAccessException,
            InvalidParentNodeException, NodeNotFoundException, NodeWithDuplicatePathException, ContentManagerException;

    /**
     * Begins a Content Manager transaction
     */
    public void beginTransaction();

    /**
     * Commit's a Content Manager transaction
     */
    public void commitTransaction();

    /**
     * Deletes an existing content from the repository
     * 
     * @param id
     *            the content to delete
     * @param user
     *            the user that is deleting the content
     * @return T if all went well
     * @throws ContentItemNotFoundException
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentManagerException
     */
    public boolean deleteContent(String id, IDIFUser user) throws ContentItemNotFoundException, NodeNotFoundException,
            NoAccessException, ContentManagerException;

    /**
     * Deletes an existing node from the repository
     * 
     * @param id
     *            the node to delete
     * @param user
     *            the user that is deleting the node
     * @return T if all went well
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws NodeWithNodesException
     * @throws ContentManagerException
     */
    public boolean deleteNode(Long id, IDIFUser user) throws NodeNotFoundException, NoAccessException,
            NodeWithNodesException, ContentManagerException;

    /**
     * Deletes an existing node from the repository. <br>
     * Will launch an exception if the node is not empty
     * 
     * @param id
     *            the node to delete
     * @param user
     *            the user that is deleting the node
     * @param cascadeDelete
     *            if T will delete all inner nodes and content, if F will launch an exception if the node is not empty
     * @return T if all went well
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws NodeWithNodesException
     * @throws ContentManagerException
     */
    public boolean deleteNode(Long id, IDIFUser user, boolean cascadeDelete) throws NodeNotFoundException,
            NoAccessException, NodeWithNodesException, ContentManagerException;

    /**
     * Searches content items by description
     * 
     * @param description
     *            the content id
     * @param user
     *            the user who's searching
     * @return the content list
     * @throws ContentManagerException
     */
    public List<ContentItem> getContentByDescription(String description, IDIFUser user) throws ContentManagerException;

    /**
     * Searches a content item by it's unique identifier
     * 
     * @param id
     *            the content id
     * @param user
     *            the user who's searching
     * @return the content
     * @throws ContentItemNotFoundException
     * @throws NoAccessException
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public ContentItem getContentById(String id, IDIFUser user) throws ContentItemNotFoundException, NoAccessException,
            NodeNotFoundException, ContentManagerException;

    /**
     * Searches content items by name
     * 
     * @param name
     *            the content id
     * @param user
     *            the user who's searching
     * @return the content list
     * @throws ContentManagerException
     */
    public List<ContentItem> getContentByName(String name, IDIFUser user) throws ContentManagerException;

    /**
     * Searches content items by parent node
     * 
     * @param nodeId
     *            the parent node id
     * @param user
     *            the user who's searching
     * @return the content list
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public List<ContentItem> getContentByParentNode(Long nodeId, IDIFUser user) throws NodeNotFoundException,
            ContentManagerException;

    /**
     * Searches content items by name and node path
     * 
     * @param nodeFullPath
     *            the node fullPath from the content will be retrieve
     * @param name
     *            the content id
     * @param user
     *            the user who's searching
     * @return the content object
     * @throws ContentItemNotFoundException
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentManagerException
     */
    public ContentItem getContentFromNodePathByName(String nodeFullPath, String name, IDIFUser user)
            throws ContentItemNotFoundException, NodeNotFoundException, NoAccessException, ContentManagerException;

    /**
     * get content item ACL
     * 
     * @param id
     *            the content item ID
     * @return the ACL
     * @throws ContentManagerException
     */
    public List<ACLEntry> getContentItemACL(String id) throws ContentManagerException;

    /**
     * get node ACL
     * 
     * @param id
     *            the node ID
     * @return the ACL
     * @throws ContentManagerException
     */
    public List<ACLEntry> getNodeACL(Long id) throws ContentManagerException;

    /**
     * Searches a node by unique identifier
     * 
     * @param id
     *            the node ID
     * @param user
     *            the user who's searching
     * @return a list of all existing root nodes
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentManagerException
     */
    public Node getNodeById(Long id, IDIFUser user) throws NodeNotFoundException, NoAccessException,
            ContentManagerException;

    /**
     * Searches a node by path
     * 
     * @param fullPath
     *            the full path of the node the node ID
     * @param user
     *            the user who's searching
     * @return a list of all existing root nodes
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentManagerException
     */
    public Node getNodeByPath(String fullPath, IDIFUser user) throws NodeNotFoundException, NoAccessException,
            ContentManagerException;

    /**
     * Searches nodes by description
     * 
     * @param description
     *            the description of the node to search
     * @param user
     *            the user who's searching
     * @return a list of all existing nodes with the description
     * @throws ContentManagerException
     */
    public List<Node> getNodesByDescription(String description, IDIFUser user) throws ContentManagerException;

    /**
     * Searches nodes by description
     * 
     * @param basePathToSearch
     *            the path to search from
     * @param description
     *            the name of the node to search
     * @param user
     *            the user who's searching
     * @return a list of all existing nodes with the description
     * @throws ContentManagerException
     */
    public List<Node> getNodesByDescription(String basePathToSearch, String description, IDIFUser user)
            throws ContentManagerException;

    /**
     * Searches nodes by name
     * 
     * @param name
     *            the name of the node to search
     * @param user
     *            the user who's searching
     * @return a list of all existing root nodes. May exist with same name in different parent nodes
     * @throws ContentManagerException
     */
    public List<Node> getNodesByName(String name, IDIFUser user) throws ContentManagerException;

    /**
     * Searches nodes by name
     * 
     * @param basePathToSearch
     *            the path to search from
     * @param name
     *            the name of the node to search
     * @param user
     *            the user who's searching
     * @return a list of all existing root nodes. May exist with same name in different parent nodes
     * @throws ContentManagerException
     */
    public List<Node> getNodesByName(String basePathToSearch, String name, IDIFUser user)
            throws ContentManagerException;

    /**
     * Searches nodes by parent node
     * 
     * @param nodeId
     *            the parent node id
     * @param user
     *            the user who's searching
     * @return the content list
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public List<Node> getNodesByParentNode(Long nodeId, IDIFUser user) throws NodeNotFoundException,
            ContentManagerException;

    /**
     * Searches root nodes that user has access
     * 
     * @param user
     *            the user who's searching
     * @return a list of all existing root nodes
     * @throws ContentManagerException
     */
    public List<Node> getRootNodes(IDIFUser user) throws ContentManagerException;

    /**
     * Grants group access to content item
     * 
     * @param contentId
     *            the id of the content to grant access
     * @param groupId
     *            the group to grant access
     * @return T if access was granted, F otherwise.
     * @throws ContentItemNotFoundException
     * @throws ContentManagerException
     */
    public boolean grantContentAccessToGroup(String contentId, String groupId) throws ContentManagerException;

    /**
     * Grants user access to content item
     * 
     * @param contentId
     *            the id of the content to grant access
     * @param user
     *            the group to grant access
     * @return T if access was granted, F otherwise.
     * @throws ContentItemNotFoundException
     * @throws ContentManagerException
     */
    public boolean grantContentAccessToUser(String contentId, IDIFUser user) throws ContentItemNotFoundException,
            ContentManagerException;

    /**
     * Grants group access to node
     * 
     * @param nodeId
     *            the id of the node to grant access
     * @param groupId
     *            the group to grant access
     * @return T if access was granted, F otherwise.
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean grantNodeAccessToGroup(Long nodeId, String groupId) throws NodeNotFoundException,
            ContentManagerException;

    /**
     * Grants user access to node
     * 
     * @param nodeId
     *            the id of the node to grant access
     * @param user
     *            the user to grant access
     * @return T if access was granted, F otherwise.
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean grantNodeAccessToUser(Long nodeId, IDIFUser user) throws NodeNotFoundException,
            ContentManagerException;

    /**
     * Grants user access to node
     * 
     * @param nodePath
     *            the path of the node to grant access
     * @param user
     *            the user to grant access
     * @return T if access was granted, F otherwise.
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean grantNodeAccessToUser(String nodePath, IDIFUser user) throws ContentManagerException;

    /**
     * Checks if group has access to the content
     * 
     * @param contentId
     *            the id of the content to check access
     * @param groupId
     *            the group to check access
     * @return T if group has access, F otherwise.
     * @throws ContentItemNotFoundException
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean hasContentAccessGroup(String contentId, String groupId) throws ContentItemNotFoundException,
            NodeNotFoundException, ContentManagerException;

    /**
     * Checks if user has access to the content
     * 
     * @param content
     *            the content to check access
     * @param user
     *            the user to check access
     * @return T if user has access, F otherwise.
     * @throws ContentItemNotFoundException
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean hasContentAccessUser(ContentItem content, IDIFUser user) throws ContentItemNotFoundException,
            NodeNotFoundException, ContentManagerException;

    /**
     * Checks if user has access to the content
     * 
     * @param contentId
     *            the id of the content to check access
     * @param user
     *            the user to check access
     * @return T if user has access, F otherwise.
     * @throws ContentItemNotFoundException
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean hasContentAccessUser(String contentId, IDIFUser user) throws ContentItemNotFoundException,
            NodeNotFoundException, ContentManagerException;

    /**
     * Checks if one of the group has access to the node
     * 
     * @param nodeId
     *            the id of the node to check access
     * @param groups
     *            the group list to check access
     * @return T if group has access, F otherwise.
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean hasNodeAccessGroups(Long nodeId, Set<String> groups) throws NodeNotFoundException,
            ContentManagerException;

    /**
     * Checks if user has access to the node
     * 
     * @param nodeId
     *            the id of the node to check access
     * @param user
     *            the user to check access
     * @return T if user has access, F otherwise.
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean hasNodeAccessUser(Long nodeId, IDIFUser user) throws NodeNotFoundException, ContentManagerException;

    /**
     * Moves content to another node
     * 
     * @param contentID
     *            the content ID to move
     * @param destinationNodeId
     *            the destination node to move to
     * @param user
     *            the user request the move
     * @return T if all went well
     * @throws ContentItemNotFoundException
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentManagerException
     */
    public boolean moveContent(String contentID, Long destinationNodeId, IDIFUser user)
            throws ContentItemNotFoundException, NodeNotFoundException, NoAccessException, ContentManagerException;

    /**
     * Moves a node to another parent node
     * 
     * @param nodeID
     *            the original node id
     * @param destinationNodeId
     *            the destination node id
     * @param user
     *            the user requesting the move operation
     * @return T if all went well
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentManagerException
     */
    public boolean moveNode(Long nodeID, Long destinationNodeId, IDIFUser user) throws NodeNotFoundException,
            NoAccessException, ContentManagerException;

    /**
     * Revokes group access to content item
     * 
     * @param contentId
     *            the id of the content to revoke access
     * @param groupId
     *            the group to revoke access
     * @return T if access was revoked, F otherwise.
     * @throws ContentItemNotFoundException
     * @throws ContentManagerException
     */
    public boolean revokeContentAccessToGroup(String contentId, String groupId) throws ContentItemNotFoundException,
            ContentManagerException;

    /**
     * Revokes user access to content item
     * 
     * @param contentId
     *            the id of the content to revoke access
     * @param user
     *            the group to revoke access
     * @return T if access was revoked, F otherwise.
     * @throws ContentItemNotFoundException
     * @throws ContentManagerException
     */
    public boolean revokeContentAccessToUser(String contentId, IDIFUser user) throws ContentItemNotFoundException,
            ContentManagerException;

    /**
     * Revokes group access to node
     * 
     * @param nodeId
     *            the id of the node to revoke access
     * @param groupId
     *            the group to revoke access
     * @return T if access was revoked, F otherwise.
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean revokeNodeAccessToGroup(Long nodeId, String groupId) throws NodeNotFoundException,
            ContentManagerException;

    /**
     * Revokes user access to node
     * 
     * @param nodeId
     *            the id of the node to revoke access
     * @param user
     *            the user to revoke access
     * @return T if access was revoked, F otherwise.
     * @throws NodeNotFoundException
     * @throws ContentManagerException
     */
    public boolean revokeNodeAccessToUser(Long nodeId, IDIFUser user) throws NodeNotFoundException,
            ContentManagerException;

    /**
     * Roll back the a Content Manager transaction
     */
    public void rollbackTransaction();

    /**
     * Updates content in the repository
     * 
     * @param content
     *            the content to update in the repository
     * @return the updated content
     * @throws ContentItemNotFoundException
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentItemWithDuplicateNameAndParentNodeException
     * @throws ContentManagerException
     */
    public ContentItem updateContent(ContentItem content) throws ContentItemNotFoundException, NodeNotFoundException,
            NoAccessException, ContentItemWithDuplicateNameAndParentNodeException, ContentManagerException;

    /**
     * Updates content in the repository
     * 
     * @param node
     *            the node to update in the repository
     * @return the updated content
     * @throws NodeNotFoundException
     * @throws NoAccessException
     * @throws ContentManagerException
     */
    public Node updateNode(Node node) throws NodeNotFoundException, NoAccessException, ContentManagerException;

}
