/**
 * 2015, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.jobs;

import java.util.Date;

import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.startup.DIFInitializer;
import pt.digitalis.dif.utils.logging.AuditContext;
import pt.digitalis.log.LogLevel;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 24/11/2015
 */
public abstract class DIFJob extends Thread {

    /**  */
    protected boolean isActive = true;

    /**  */
    protected Date lastRunDate = null;

    /** */
    protected Long runIntervalInSeconds = null;

    /**  */
    protected String userName = null;

    /**
     * @param session
     *            the current dif session. Null if none available (in this case any DB operations will be audited like
     *            DIF)
     */
    public DIFJob(IDIFSession session)
    {
        if (session == null || !session.isLogged())
            userName = "DIF";
        else
            userName = session.getUser().getID();
    }

    /**
     * @return T to state that the run was successful
     * @throws Exception
     */
    protected abstract boolean executeEachTime() throws Exception;

    /**
     * @return should provide the default run interval for when no custom configuration exists
     */
    protected abstract Long getDefaultRunIntervalInSeconds();

    /**
     * Inspector for the 'jobType' attribute.
     * 
     * @return the jobType value
     */
    public abstract JobType getJobType();

    /**
     * Inspector for the 'lastRunDate' attribute.
     * 
     * @return the lastRunDate value
     */
    public Date getLastRunDate()
    {
        return lastRunDate;
    }

    /**
     * Inspector for the 'runIntervalInSeconds' attribute.<br/>
     * run every 5 minutes (default)
     * 
     * @return the runIntervalInSeconds value
     */
    public Long getRunIntervalInSeconds()
    {
        if (runIntervalInSeconds == null)
        {
            if (getDefaultRunIntervalInSeconds() == null)
                return 300L;
            else
                return getDefaultRunIntervalInSeconds();
        }

        return runIntervalInSeconds;
    }

    /**
     * Inspector for the 'isActive' attribute.
     * 
     * @return the isActive value
     */
    public boolean isActive()
    {
        return isActive;
    }

    /**
     * @see java.lang.Thread#run()
     */
    @Override
    public final void run()
    {
        while (true)
        {
            try
            {
                if (!DIFInitializer.isInitialized())
                {
                    synchronized (this)
                    {
                        wait(10000); // wait 10s until DIF initializes
                    }
                }
                else
                {
                    AuditContext.setProcessNameForCurrentThread("Job:" + this.getClass().getSimpleName());
                    AuditContext.setUserForCurrentThread(userName);

                    // This is started statically and so the DIF might not be initialized yet. Must not trigger an
                    // initialize, will wait for it to happen in due time The 0 check is to allow disabling of the job
                    // by setting this to 0
                    if (this.getJobType() == JobType.SINGLE_EXECUTION || this.getRunIntervalInSeconds() != 0)
                        try
                        {
                            if (isActive())
                            {
                                executeEachTime();
                                this.setLastRunDate(new Date());
                            }
                        }
                        catch (Exception e)
                        {
                            // TODO: ComQuest: Viegas: 0. Report to ErrorLog
                            new BusinessException(e).log(LogLevel.WARN);
                        }

                    if (this.getJobType() == JobType.SINGLE_EXECUTION)
                        break;
                    else
                    {
                        // Wait
                        synchronized (this)
                        {
                            wait((this.getRunIntervalInSeconds() == 0 ? 60 : this.getRunIntervalInSeconds()) * 1000);
                        }
                    }
                }
            }
            catch (Exception e)
            {
                new BusinessException(e).log(LogLevel.WARN);
            }
        }
    }

    /**
     * Modifier for the 'isActive' attribute.
     * 
     * @param isActive
     *            the new isActive value to set
     */
    public void setActive(boolean isActive)
    {
        this.isActive = isActive;
    }

    /**
     * Modifier for the 'lastRunDate' attribute.
     * 
     * @param lastRunDate
     *            the new lastRunDate value to set
     */
    public void setLastRunDate(Date lastRunDate)
    {
        this.lastRunDate = lastRunDate;
    }

    /**
     * Modifier for the 'runIntervalInSeconds' attribute.
     * 
     * @param runIntervalInSeconds
     *            the new runIntervalInSeconds value to set
     */
    public void setRunIntervalInSeconds(Long runIntervalInSeconds)
    {
        this.runIntervalInSeconds = runIntervalInSeconds;
    }

}
