/**
 * - Digitalis Internal Framework v2.0 -
 *
 * (C) 2007, Digitalis Informatica. 
 * 
 * Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1
 * 2780-666 Paco de Arcos
 * Telefone: (351) 21 4408990
 * Fax: (351) 21 4408999
 * http://www.digitalis.pt 
 */
package pt.digitalis.dif.utils.logging;

import pt.digitalis.utils.ioc.IIoCRegistry;

/**
 * Contains the logging logic to be applied on the IoC container
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Oct 3, 2007
 * 
 */
final public aspect DIFIoCLogAspect extends DIFLogAspect {

    /** Holds the start time for the IoC engine */
    private long iocStart;
    
    /** Saves if the IoC has been started previously */
    static private boolean iocStarted = false; 

    /**
     * Defines the pointcut for the registry instantiation and it's return.
     */
    protected pointcut IoCRegistryConstructor() : call (* pt.digitalis.dif.ioc.DIFIoCRegistry.getRegistry(..)) && excludeLoggingAppliances();

    /**
     * Defines the pointcut for service resolution
     * 
     * @param clazz
     *            the class that defines the interface param obj the object instantiated
     */
    protected pointcut IoCRegistryGetService(Class clazz) : call (* pt.digitalis.utils.ioc.IIoCRegistry+.getImplementation(Class)) && args(clazz) && excludeLoggingAppliances();

    /**
     * Defines the pointcut for each discovered module
     */
    protected pointcut IoCRegistryDiscoverModules() : execution (* pt.digitalis.utils.ioc.IIoCRegistry+.discoverModules()) && excludeLoggingAppliances();

    /**
     * Captures the discovered config files for module discovery
     */
    private pointcut IoCConfigFilesFound() : withincode (* pt.digitalis.utils.ioc.AbstractIoCRegistryImpl+.discoverModules()) && call (* pt.digitalis.utils.inspection.ClasspathUtils.getClasspathFileNamesWithExtension(..)) && excludeLoggingAppliances();

    /**
     * Logs the start of the startup process of the IoC
     */
    before() : IoCRegistryConstructor()
    {
        if (!iocStarted) {
            iocStart = System.currentTimeMillis();

            // Log the message
            super.getLogger().info("IoC engine: Initializing...");
        }
    }

    /**
     * Logs the end of the startup process of the IoC
     * 
     * @param registry
     *            the registry instantiated
     */
    after() returning (IIoCRegistry registry) : IoCRegistryConstructor()
    {
        if (!iocStarted) {
            // Log the message
            super.getLogger().info(
                    "IoC engine: \"" + registry.getClass().getSimpleName() + "\" implementation started in "
                    + (System.currentTimeMillis() - iocStart) + "ms");
            iocStarted = true;
        }
    }

    /**
     * Logs each resolution made by the IoC
     * 
     * @param clazz
     *            the class that defines the service to look for
     * @param obj
     *            the implementation resolved
     * 
     */
    after(Class clazz) returning (Object obj) : IoCRegistryGetService(clazz)
    {
        // Log the message
        super.getLogger().debug(
                "IoC resolved: \"" + clazz.getSimpleName() + "\" -> \"" + obj.getClass().getSimpleName() + "\"");
    }
}
