/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.templates;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.IBeanAttributes;
import pt.digitalis.utils.common.StringUtils;

/**
 * Utilities for template loading and parsing
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Mar 3, 2009
 */
public class TemplateUtils {

    /**
     * Get a Remote template
     * 
     * @param url
     *            the template location the language tranlation
     * @return content to String
     * @throws Exception
     *             if an Exception occurrs
     */
    static public String getRemoteTemplate(String url) throws Exception
    {
        URL website;

        website = new URL(url);

        URLConnection connection = website.openConnection();
        BufferedReader in = new BufferedReader(new InputStreamReader(connection.getInputStream()));

        StringBuilder response = new StringBuilder();
        String inputLine;

        while ((inputLine = in.readLine()) != null)
            response.append(inputLine);

        in.close();

        return response.toString();
    }

    /**
     * Get the remote template for language
     * 
     * @param url
     *            the url
     * @param language
     *            the language
     * @return the content
     */
    static public String getRemoteTemplate(String url, String language)
    {
        String result = null;

        if (language == null)
        {
            try
            {
                return getRemoteTemplate(url);
            }
            catch (Exception e)
            {
                DIFLogger.getLogger().debug(e);
                return null;
            }
        }

        // Try to get content for specific language
        try
        {
            result = getRemoteTemplate(url + "." + language);
        }
        catch (Exception e)
        {
            DIFLogger.getLogger().debug(e);
        }

        if (result != null)
        {
            return result;
        }
        else
        {
            // Try to get content for default language
            try
            {
                result = getRemoteTemplate(url + "."
                        + DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage());
            }
            catch (Exception e)
            {
                DIFLogger.getLogger().debug(e);
            }
        }

        if (result != null)
        {
            return result;
        }
        else
        {
            // Try to get content without language
            try
            {
                result = getRemoteTemplate(url);
            }
            catch (Exception e)
            {
                DIFLogger.getLogger().debug(e);
            }
        }

        return result;
    }

    /**
     * Reads a given template and it's content
     * 
     * @param inputStream
     *            the input stream for the resource
     * @param params
     *            the parameters to parse the template and replace their values
     * @return the resource content
     * @throws IOException
     *             if any error happened when reading the resource
     */
    static private StringBuffer getTemplateContent(InputStream inputStream, Map<String, String> params)
            throws IOException
    {
        BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(inputStream));

        StringBuffer buffer = new StringBuffer();
        String line = parseTemplateLine(bufferedReader.readLine(), params);
        while (line != null)
        {
            buffer.append(line);
            line = parseTemplateLine(bufferedReader.readLine(), params);
        }

        return buffer;
    }

    /**
     * Reads a given template and it's content
     * 
     * @param resource
     *            the resource name
     * @param params
     *            the parameters to parse the template and replace their values
     * @return the content of the given resource
     * @throws IOException
     *             if any error happened when reading the resource
     */
    static public StringBuffer getTemplateContent(String resource, Map<String, String> params) throws IOException
    {
        return getTemplateContent(getTemplateStream(resource), null);
    }

    /**
     * Reads a given template and it's content
     * 
     * @param resource
     *            the resource name
     * @param language
     *            the default language to get the template in
     * @param params
     *            the parameters to parse the template and replace their values
     * @return the content of the given resource
     * @throws IOException
     *             if any error happened when reading the resource
     */
    static public StringBuffer getTemplateContent(String resource, String language, Map<String, String> params)
            throws IOException
    {

        if (language != null)
        {
            InputStream fileInput = Thread.currentThread().getContextClassLoader()
                    .getResourceAsStream(resource + "." + language);

            if (fileInput == null)
                fileInput = Thread
                        .currentThread()
                        .getContextClassLoader()
                        .getResourceAsStream(
                                resource + "." + DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage());

            if (fileInput == null)
                fileInput = Thread.currentThread().getContextClassLoader().getResourceAsStream(resource);

            return getTemplateContent(fileInput, params);
        }
        else
            return getTemplateContent(resource, params);
    }

    /**
     * Returns a given template input stream
     * 
     * @param resource
     *            the resource name
     * @return the stream
     * @throws IOException
     *             if any error happened when reading the resource
     */
    static public InputStream getTemplateStream(String resource) throws IOException
    {
        // TODO: Must define the correct codepage to load. Is not picking up the ISO-8859-1!!!
        return Thread.currentThread().getContextClassLoader().getResourceAsStream(resource);
    }

    /**
     * Parses a given line for parameters, and replaces them with their respective values
     * 
     * @param template
     *            the line to parse and replace parameter values
     * @param businessClass
     *            the business class to extract the attribute values from
     * @return the resource content
     */
    public static String parseTemplateLine(String template, IBeanAttributes businessClass)
    {
        return parseTemplateLine(template, businessClass, null);
    }

    /**
     * Parses a given line for parameters, and replaces them with their respective values
     * 
     * @param template
     *            the line to parse and replace parameter values
     * @param businessClass
     *            the business class to extract the attribute values from
     * @param values
     *            additional values to replace
     * @return the resource content
     */
    public static String parseTemplateLine(String template, IBeanAttributes businessClass, Map<String, String> values)
    {
        String workString = template;

        if (values == null)
            values = new HashMap<String, String>();

        // Specify the expression for ${xxx}
        Pattern p = Pattern.compile("\\$\\{([^\\}]*)\\}.*");
        Matcher m = p.matcher(workString);
        int lasFindIndex = -1;

        // Extract all entries and replace them with the value from the business class
        while (m.find())
        {
            if (lasFindIndex == m.start())
                break;

            lasFindIndex = m.start();

            // Get the expression occurrence
            String group = m.group(1);
            String value = null;

            if (businessClass.getAttribute(group) != null)
                value = businessClass.getAttributeAsString(group);
            else
                value = values.get(group);

            if (value != null)
                // Replace it (and all equals occurrences) with the attribute value from the bean
                workString = workString.replaceAll("\\$\\{" + group + "\\}", value);

            // Find the next occurrence
            m = p.matcher(workString);
        }

        return workString;
    }

    /**
     * Parses a given line for parameters, and replaces them with their respective values
     * 
     * @param line
     *            the line to parse and replace parameter values
     * @param params
     *            the parameters to parse the template and replace their values
     * @return the resource content
     */
    static public String parseTemplateLine(String line, Map<String, String> params)
    {
        if (line != null && !"".equals(line) && params != null)
        {
            for (Entry<String, String> param: params.entrySet())
            {
                String key = "${" + param.getKey() + "}";
                // String.replaceAll wasn't used due to regular expressions errors into the value field. If the value
                // contained "$" an IllegalArgumentException is throwed.
                line = StringUtils.replace(line, key, param.getValue());
            }
        }

        return line;
    }
}
