/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.translator;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.List;
import java.util.Map.Entry;

import pt.digitalis.dif.translator.exception.TranslatorException;
import pt.digitalis.utils.common.IBeanAttributes;

/**
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @author Galaio da Silva <a href="mailto:jgalaio@digitalis.pt">jgalaio@digitalis.pt</a><br/>
 * @param <T>
 * @created Dec 16, 2009
 */
public class TranslatorEngine<T extends Object> {

    /** The exporter */
    private ITranslatorExporter<T> exporter = null;

    /** The mappings to use in the translation */
    private TranslatorMappings mappings = null;

    /**
     * Constructor
     * 
     * @param mappings
     *            the mappings
     * @param exporter
     *            the exporter
     */
    public TranslatorEngine(TranslatorMappings mappings, ITranslatorExporter<T> exporter)
    {
        this.mappings = mappings;
        this.exporter = exporter;

    }

    /**
     * Automatically build mappings
     * 
     * @param listBeans
     */
    @SuppressWarnings("unchecked")
    private void buildMappings(List<? extends IBeanAttributes> listBeans)
    {
        this.mappings = new TranslatorMappings();

        HashMap<String, String> mappingMap = new HashMap<String, String>();
        FieldsDictionary dictionary = new FieldsDictionary();

        for (IBeanAttributes bean: listBeans)
        {
            T theClass = (T) bean;
            for (Field field: theClass.getClass().getDeclaredFields())
            {
                mappingMap.put(field.getName(), field.getName());
            }
        }
        this.mappings.setMappings(mappingMap);
        this.mappings.setFieldsDictionary(dictionary);
    }

    /**
     * Execute the translation
     * 
     * @param listBeans
     *            the list of beans to translate
     * @return the result value
     * @throws TranslatorException
     *             if a Translator Exception occurs
     */
    public T execute(List<? extends IBeanAttributes> listBeans) throws TranslatorException
    {
        if (this.mappings == null)
            this.buildMappings(listBeans);

        FieldsDictionary fieldsDictionary = this.mappings.getFieldsDictionary();
        this.exporter.init(this.mappings); // exporter initialization

        for (IBeanAttributes bean: listBeans)
        {
            HashMap<String, String> destinationValues = new HashMap<String, String>();

            for (Entry<String, String> mapping: this.mappings.getMappings().entrySet())
            {
                // Get the original value from bean
                String originalValue = bean.getAttributeAsString(mapping.getKey());

                String destinationValue;

                // Tesst if a mapping has translation
                if (fieldsDictionary.fieldHasDictionary(mapping.getKey()))
                {
                    destinationValue = fieldsDictionary.getValue(mapping.getKey(), originalValue);

                    /*
                     * if a field has translation but the original value was not translated gives an error
                     */
                    if (destinationValue == null && !"".equals(originalValue))
                        throw new TranslatorException(TranslatorException.TRANSLATION_VALUE_NOT_FOUND
                                .replace("#key#", mapping.getKey()).replace("#value#", originalValue));
                }
                else
                {
                    /*
                     * In case of a value has no translation, the input value will be the output
                     */
                    destinationValue = originalValue;
                }
                destinationValues.put(mapping.getValue(), destinationValue); // places the translation result in the
                                                                             // map
            }
            this.exporter.processValues(destinationValues); // The exporter will process a new line
        }
        this.exporter.terminate(); // exporter finalization

        return this.exporter.getResult(); // the exporter will process the final result
    }

}
