/**
 * First created between November and December 2003 1994-2003 Digitalis Informatica. All rights reserved. Distribuicao e
 * Gestao de Informatica, Lda. Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21
 * 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package util.collection;

import java.util.ArrayList;
import java.util.List;

import util.Filter;
import util.ObjectCompare;

// TODO: Auto-generated Javadoc
/**
 * Set of utility methods to manipulate collections, lists and relatated objects.
 * 
 * @author Daniel Alexandre Campelo <a href="mailto:dcampelo@digitalis.pt">dcampelo@digitalis.pt</a><br />
 */
public class CollectionUtil {

    /**
     * Compares the itens of two lists. If the number of itens is diferente then the list is considered to be diferent.
     * The order by which the itens are in the list is not taken into account
     * <p>
     * <b>NOTE</b> : The objects are not cloned.
     * 
     * @param <L1>
     *            the generic type
     * @param <L2>
     *            the generic type
     * @param _list1
     *            the _list1
     * @param _list2
     *            the _list2
     * @param objComparer
     *            the obj comparer
     * @return A new list with the <code>Objects</code> that fullfill the criteria
     */
    public static <L1 extends Object, L2 extends Object> boolean compareLists(List<L1> _list1, List<L2> _list2,
            ObjectCompare<L1, L2> objComparer)
    {
        if ((_list1 == null || _list1.isEmpty()) && (_list2 == null || _list2.isEmpty()))
        {
            return true;
        }

        if (_list1.size() != _list2.size())
        {
            return false;
        }

        boolean found = false;
        for (int i = 0; i < _list1.size(); i++)
        {

            found = false;
            for (int j = 0; j < _list2.size() && !found; j++)
            {
                found = (objComparer.compare(_list1.get(i), _list2.get(j)));
            }
            if (found == false)
            {
                break;
            }
        }

        return found;
    }

    /**
     * Filters a list for the specified criteria. Returns the first Object that fullfills the specified criteria
     * <p>
     * <b>NOTE</b> : The object is not cloned.
     * 
     * @param <L>
     *            the generic type
     * @param _list
     *            the list to filter
     * @param _filter
     *            the filter
     * @return The first <code>Object</code> found that fullfills the criteria, <code>null</code> if none found
     */
    public static <L extends Object> L filterFirst(List<L> _list, Filter<L> _filter)
    {
        L res = null;
        for (int i = 0; i < _list.size() && res == null; i++)
        {
            if (_filter.accept(_list.get(i)))
            {
                res = _list.get(i);
            }
        }
        return res;
    }

    /**
     * Filters a list for the specified criteria. The objects that fullfill the criteria are to be reference by the
     * filtered list.
     * <p>
     * <b>NOTE</b> : The objects are not cloned.
     * 
     * @param <L>
     *            the generic type
     * @param _list
     *            the list to filter
     * @param _filter
     *            the filter
     * @return A new list with the <code>Objects</code> that fullfill the criteria
     */
    public static <L extends Object> List<L> filterList(List<L> _list, Filter<L> _filter)
    {
        ArrayList<L> _resList = new ArrayList<L>();
        for (int i = 0; i < _list.size(); i++)
        {
            if (_filter.accept(_list.get(i)))
            {
                _resList.add(_list.get(i));
            }
        }
        return _resList;
    }

    /**
     * Merges two lists. All instances that are diferent will be refered by the new List. The first list will take
     * precedence if there are 'equal' objects.
     * <p>
     * The compartion between each object of the lists will be the responsability of the comparator.
     * <p>
     * <b>NOTE</b> : The objects are not cloned.
     * 
     * @param <L>
     *            the generic type
     * @param _list1
     *            list to be merged
     * @param _list2
     *            list to be merged
     * @param _comp
     *            the comparator, responsable for the comparation between the objects af the two lists
     * @return The list that will contain all different instances belonged to the lists.
     */
    public static <L extends Object> List<L> merge(List<L> _list1, List<L> _list2, ObjectCompare<L, L> _comp)
    {
        if (_list1 == null)
        {
            if (_list2 == null)
            {
                return new ArrayList<L>();
            }
            else
            {
                return new ArrayList<L>(_list2);
            }
        }
        else if (_list2 == null)
        {
            return new ArrayList<L>(_list1);
        }

        ArrayList<L> res = new ArrayList<L>(_list1);
        // Searchs all the instances of the second list
        for (int i = 0; i < _list2.size(); i++)
        {
            boolean found = false;

            // Checks if exists on the result list
            for (int j = 0; j < res.size() && !found; j++)
            {

                // Compares using the specified criteria
                found = (_comp.compare(res.get(j), _list2.get(i)));
            }

            // If it does not exist in the list will add it
            if (!found)
            {
                res.add(_list2.get(i));
            }
        }
        return res;
    }

    /**
     * Merges two lists. All instances that are diferent will be refered by the new List. The condition used is the
     * {@link java.util.ArrayList#contains(java.lang.Object)} method, as a consequence the 'equals' will be used as
     * check for each instance.
     * <p>
     * The first list will take precedence.
     * <p>
     * <b>NOTE</b> : The objects are not cloned.
     * 
     * @param <T>
     *            the generic type
     * @param _list1
     *            list to be merged
     * @param _list2
     *            list to be merged
     * @return The list that will contain all different instances between the lists
     */
    public static <T extends Object> List<T> merge(List<T> _list1, List<T> _list2)
    {
        if (_list1 == null)
        {
            if (_list2 == null)
            {
                return new ArrayList<T>();
            }
            else
            {
                return new ArrayList<T>(_list2);
            }
        }
        else if (_list2 == null)
        {
            return new ArrayList<T>(_list1);
        }

        ArrayList<T> res = new ArrayList<T>(_list1);

        for (int i = 0; i < _list2.size(); i++)
        {

            if (!res.contains(_list2.get(i)))
            {
                res.add(_list2.get(i));
            }
        }
        return res;
    }

    /**
     * Removes a <code>Object</code> that is accepted by the specified criteria in the {@link Filter}.<br>
     * Only one <code>Object</code> is removed, if more than one <code>Object</code> may corresponde to the criteria
     * defined in the <code>Filter</code> then {@link #removeAll(List, Filter) should be used unstead}.
     * 
     * @param <L>
     *            the generic type
     * @param _list
     *            the list to remove the item from
     * @param _filter
     *            the filter where the search criteria is defined
     * @return the removed <code>Object</code>
     */
    public static <L extends Object> L remove(List<L> _list, Filter<L> _filter)
    {
        if (_list == null || _list.isEmpty())
        {
            return null;
        }
        L removed = null;

        for (int i = 0; i < _list.size() && removed == null; i++)
        {
            if (_filter.accept(_list.get(i)))
            {
                removed = _list.remove(i);
            }
        }

        return removed;
    }

    /**
     * Removes all itens that are accepted by the specified criteria in the {@link Filter}.<br>
     * If only on item is to be ermoved then the {@link #remove(List, Filter)} should be used unstead due to performance
     * reasons.
     * 
     * @param <L>
     *            the generic type
     * @param _list
     *            the list to remove the itens from
     * @param _filter
     *            the filter where the search criteria is defined
     * @return a list of the removed itens
     */
    public static <L extends Object> List<L> removeAll(List<L> _list, Filter<L> _filter)
    {
        if (_list == null || _list.isEmpty())
        {
            return null;
        }
        List<L> removedItens = new ArrayList<L>();

        for (int i = 0; i < _list.size(); i++)
        {
            if (_filter.accept(_list.get(i)))
            {
                removedItens.add(_list.remove(i));
            }
        }

        return removedItens;
    }

}