/**
 * First created between November and December 2003
 *
 * 1994-2003 Digitalis Informatica. All righsts reserved.
 * 
 * Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1
 * 2780-666 Paco de Arcos
 * Telefone: (351) 21 4408990
 * Fax: (351) 21 4408999
 * http://www.digitalis.pt 
 */
package util.sql;

import java.util.ArrayList;
import java.util.ListIterator;

// TODO: Auto-generated Javadoc
/**
 * Utility class used to process a query to add an order by, and optionally, a
 * paging system.
 * 
 * @author Ricardo Correia de Oliveira <a
 *         href="mailto:roliveira@digitalis.pt">roliveira@digitalis.pt</a><br />
 * @author Daniel Alexandre Campelo <a
 *         href="mailto:dcampelo@digitalis.pt">dcampelo@digitalis.pt</a><br />
 * 
 *         Created on Jan 20, 2004
 */
public abstract class OrderByClause {

	/**
	 * Used to map the attribute with it's mode in the list.
	 * 
	 * @author Daniel Alexandre Campelo <a
	 *         href="mailto:dcampelo@digitalis.pt">dcampelo@digitalis.pt</a><br />
	 *         Created : 3/Jun/2005
	 */
	class NameModeValue {
		/**
		 * The property name.
		 */
		private String attr = null;
		/**
		 * The mode.
		 * 
		 * @see OrderByClause#DESCEND
		 * @see OrderByClause#ASCEND
		 */
		private String mode = null;

		/**
		 * Creates a new <i>node</i>.
		 * 
		 * @param mode
		 *            the mode
		 * @param attr
		 *            the attribute
		 */
		public NameModeValue(String mode, String attr) {
			this.setAttr(attr);
			this.setMode(mode);
		}

		/**
		 * Gets the attribute.
		 * 
		 * @return the attribute
		 */
		public String getAttr() {
			return attr;
		}

		/**
		 * Gets the mode.
		 * 
		 * @return the mode
		 */
		public String getMode() {
			return mode;
		}

		/**
		 * Sets the attribute.
		 * 
		 * @param _attr
		 *            the attribute
		 */
		public void setAttr(String _attr) {
			attr = _attr;
		}

		/**
		 * Sets the mode.
		 * 
		 * @param string
		 *            the mode
		 */
		public void setMode(String string) {
			mode = string;
		}
	}

	/** The Constant serialVersionUID. */
	static final long serialVersionUID = -9373411620714L;
	/**
	 * Flag that marks if the paging system is to used.
	 */
	protected boolean addPager = true;
	/**
	 * Order by ascendent mode.
	 */
	protected String ASCEND = "ASC";
	/**
	 * Order by descendent mode.
	 */
	protected String DESCEND = "DESC";
	/**
	 * The number of pages, only considerer if the paging system is active (
	 * {@link #addPager}).
	 */
	private int numPages = 0;
	/**
	 * The list of atributes to add to the <i>order by</i> clause. The list is
	 * order by the order in which the attributes are to be set in the <i>order
	 * by</i> clause.
	 */
	protected ArrayList<NameModeValue> props = null;

	/**
	 * The number of itens per page, only considerer if the paging system is
	 * active ({@link #addPager}).
	 */
	private int rowsPerPage = 0;

	/**
	 * Adds a property at the end of the list. Does not consider the order
	 * 
	 * @param propName
	 *            the name of the property
	 * @param mode
	 *            the mode by which the attribute will be used on the <i>order
	 *            by</i> clause. The modes are {@link #DESCEND} and
	 *            {@link #ASCEND}.
	 */
	public synchronized void addProperty(String propName, String mode) {
		// Adds a new element
		this.addProperty(propName, mode, -1);
	}

	/**
	 * Adds a property at the given position on the list.
	 * 
	 * @param propName
	 *            the name of the property
	 * @param mode
	 *            the mode by which the attribute will be used on the <i>order
	 *            by</i> clause. The modes are {@link #DESCEND} and
	 *            {@link #ASCEND}.
	 * @param index
	 *            the position where to place the property
	 */
	public synchronized void addProperty(String propName, String mode, int index) {
		if (mode != null && (mode.equals(DESCEND) || mode.equals(ASCEND))) {
			NameModeValue nameMode = new NameModeValue(
					this.translateMode(mode), propName);
			if (index >= 1) {
				for (int i = props.size() - 1; i < index; i++) {
					props.add(null);
				}
				this.props.add(index - 1, nameMode);
			} else {
				this.props.add(nameMode);
			}
		}
	}

	/**
	 * Adds a property at the given position on the list.
	 * 
	 * @param propName
	 *            the name of the property
	 * @param mode
	 *            the mode by which the attribute will be used on the <i>order
	 *            by</i> clause. The modes are {@link #DESCEND} and
	 *            {@link #ASCEND}.
	 * @param index
	 *            the position where to place the property
	 * @see #addProperty(String, String, int)
	 */
	public void addProperty(String propName, String mode, String index) {
		try {
			this.addProperty(propName, mode, Integer.parseInt(index));
		} catch (NumberFormatException nfe) {
			this.addProperty(propName, mode);
		}
	}

	/**
	 * Disables the paging system.
	 */
	public void disablePager() {
		addPager = false;
	}

	/**
	 * Gets the total number of pages.
	 * 
	 * @return the total number of pages
	 */
	public int getNumPages() {
		return numPages;
	}

	/**
	 * Creates a pager system based on the properties of this instance.
	 * 
	 * @return a new pager system
	 */
	public abstract PagerQuery getPagerQuery();

	/**
	 * Creates a pager system with the given properties.
	 * 
	 * @param numPagesAux
	 *            the num pages aux
	 * @param rowsPerPageAux
	 *            the rows per page aux
	 * @return a new pager system
	 */
	public abstract PagerQuery getPagerQuery(int numPagesAux, int rowsPerPageAux);

	/**
	 * Gets the <i>order by</i> clause from a query.
	 * 
	 * @param query
	 *            the query to get the <i>order by</i> clause from
	 * @return the <i>order by</i> of the query or an empty <code>String</code>
	 *         if none defined
	 */
	protected abstract String getQueryOrderByClause(String query);

	/**
	 * Gets the number of itens per page.
	 * 
	 * @return the number of itens per page
	 */
	public int getRowsPerPage() {
		return rowsPerPage;
	}

	/**
	 * Checks if the given query has defined the <i>order by</i> calse.
	 * 
	 * @param query
	 *            the query to check
	 * @return <code>true</code> if the query has the <i>order by</i> clause
	 *         defined, <code>false</code> otherwise
	 */
	protected abstract boolean hasOrderByClause(String query);

	/**
	 * Process the given query by adding the <i>order by</i> clause and, if
	 * active, the paging system. If the query has already defined a <i>order
	 * by</i> clause, the properties are added to it. If not, adds a new
	 * <i>order by</i> clause with the properties.
	 * 
	 * @param query
	 *            to add the <i>order by</i> clause and the paging system
	 * @return the processed query
	 */
	public String prepareQuery(String query) {
		String orderByRes;
		// Verifica se a query tem orderBy
		if (this.hasOrderByClause(query)) {
			// Obtem a orderBy da query
			String queryOrderBy = getQueryOrderByClause(query);
			// Adiciona novas propriedades a um orderBy existente
			orderByRes = processOrderByClause(queryOrderBy);

		} else { // No tem clausula de order by
					// Cria uma nova clausula de order by, com as propriedades
			orderByRes = processOrderByClause();
		}
		// Devolve a query com a nova clausula de order by
		String queryRes = this.stripOrderByClause(query) + orderByRes;

		// Verifica se  para adicionar controlo de paginao
		if (this.addPager) {
			// Obtem o paginador
			PagerQuery pagerQ = this.getPagerQuery();
			// Introduz o controlo de paginao  query
			queryRes = pagerQ.processQuery(queryRes);
		}
		return queryRes;
	}

	/**
	 * Creates a <i>order by</i> clause based on the properties specified.
	 * 
	 * @return a <i>order by</i> clause with the properties
	 */
	public abstract String processOrderByClause();

	/**
	 * Adds the properties to the specified <i>order by</i> clause.
	 * 
	 * @param orderBy
	 *            the <i>order by</i> clause to add the properties to
	 * @return the given <i>order by</i> clause with the properties
	 */
	public abstract String processOrderByClause(String orderBy);

	/**
	 * Removes a property with the given name.
	 * 
	 * @param propName
	 *            the name of the property to remove
	 */
	public synchronized void removeProperty(String propName) {
		if (propName != null) {
			boolean found = false;
			ListIterator<NameModeValue> lIter = props.listIterator();
			// Itera sobre todos os elementos ou ate encontrar o elemento para
			// remover
			while (lIter.hasNext() && !found) {
				NameModeValue nmValue = lIter.next();
				// Verifica se encontrou o registo a remover
				if (nmValue != null && nmValue.getAttr().equals(propName)) {
					// Remove o registo e sai do ciclo
					lIter.remove();
					found = true;
				}
			}
		}
	}

	/**
	 * Sets the total number of pages.
	 * 
	 * @param i
	 *            the total number of pages
	 */
	public void setNumPages(int i) {
		numPages = i;
	}

	/**
	 * Sets the total number of pages.
	 * 
	 * @param i
	 *            the total number of pages
	 */
	public void setNumPages(String i) {
		try {
			setNumPages(Integer.parseInt(i));
		} catch (NumberFormatException nfe) {
			setNumPages(1);
		}
	}

	/**
	 * Gets the number of itens per page.
	 * 
	 * @param i
	 *            the number of itens per page
	 */
	public void setRowsPerPage(int i) {
		rowsPerPage = i;
	}

	/**
	 * Gets the number of itens per page.
	 * 
	 * @param i
	 *            the number of itens per page
	 */
	public void setRowsPerPage(String i) {
		try {
			setRowsPerPage(Integer.parseInt(i));
		} catch (NumberFormatException nfe) {
			setRowsPerPage(1);
		}
	}

	/**
	 * Removes the <i>order by</i> calse from a query.
	 * 
	 * @param query
	 *            the query to get the <i>order by</i> clause stripped
	 * @return the query without the <i>order by</i> clause
	 */
	protected abstract String stripOrderByClause(String query);

	/**
	 * Translates the given ordering mode into SQL.
	 * 
	 * @param modeCod
	 *            the mode to get the corresponding SQL
	 * @return the SQL ordering mode text
	 * @see #DESCEND
	 * @see #ASCEND
	 */
	protected abstract String translateMode(String modeCod);
}
