/**
 * First created between November and December 2003
 *
 * 1994-2003 Digitalis Informatica. All rights reserved.
 * 
 * Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1
 * 2780-666 Paco de Arcos
 * Telefone: (351) 21 4408990
 * Fax: (351) 21 4408999
 * http://www.digitalis.pt 
 * 
 */

package util.system;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

// TODO: Auto-generated Javadoc
/**
 * Set of utility methods to help the manipulation of the system data.
 * 
 * @author Daniel Alexandre Campelo  <a href="mailto:dcampelo@digitalis.pt">dcampelo@digitalis.pt</a><br />
 * Created : 15/Mar/2005
 */
public class SystemUtil {

	/**
	 * Gets a resource as a <code>File</code> instance. It uses the <code>ClassLoader</code> defined for the <i>_class</i> Class parameter 
	 * or the <code>ClassLoader</code> defined for this class (<i>SystemUtil</i>) if the <i>_class</i> parameter is <code>null</code>.
	 * <p>
	 * If the resource is not found using the specified <code>Class</code> <code>ClassLoader</code> the system default <code>ClassLoder</code> is used.
	 * @param _resource the resource to load
	 * @param _class    the Class to use the <code>ClassLoader</code> from
	 * @return a File instance representing the resource
	 * @throws IOException if the resource is not found or an error occurs while loading it
	 */
	public static File getResourceAsFile(String _resource, Class<?> _class) throws IOException {
		ClassLoader loader = null;
		if(_class == null){
			loader = SystemUtil.class.getClassLoader();
		} else {
			loader = _class.getClassLoader();
		}

		URL resUrl = null;
		if (loader != null){
			resUrl = loader.getResource( _resource);
		}

		if(resUrl == null) {
			resUrl = ClassLoader.getSystemResource( _resource);
		}
		if ( resUrl == null){
			throw new IOException("Could not find resource " + _resource);
		}
		return new File( resUrl.getPath());
	}

	/**
	 * Loads a properties resource from the classpath using the <code>ClassLoader</code> defined for this class (<code>SystemUtil</code>). 
	 * @param _resource  the properties resource to load
	 * @return the properties resource
	 * @throws IOException if an error occurs while loading the properties resource or if it is not found
	 */
	public static Properties getResourceAsProperties(String _resource) throws IOException {
		Properties prop = new Properties();
		prop.load( SystemUtil.getResourceStream( _resource));
		return prop;
	}

	/**
	 * Loads a properties resource from the classpath using the <code>ClassLoader</code> defined for the specified <code>Class</code>.
	 * @param _resource  the properties resource to load
	 * @param _class     the <code>Class</code> to use the <code>ClassLoader</code> from
	 * @return  the properties resource
	 * @throws IOException if an error occurs while loading the properties resource or if it is not found
	 */
	public static Properties getResourceAsProperties(String _resource, Class<?> _class) throws IOException {
		Properties prop = new Properties();
		InputStream in = SystemUtil.getResourceStream( _resource, _class);

		prop.load( in);

		in.close();
		return prop;
	}


	/**
	 * Gets the Stream of a resource from the classpath using the <code>ClassLoader</code> defined for this class (<code>SystemUtil</code>).
	 * @param _resource  the resource to get the stream
	 * @return  the stream of the specified resource
	 * @throws IOException if an error occurs while opening the stream of the resource or if it is not found
	 */
	public static InputStream getResourceStream(String _resource) throws IOException {

		return getResourceStream(_resource, SystemUtil.class);
	}

	/**
	 * Gets a Stream of a resource from the classpath using the <code>ClassLoader</code> defined for the specified <code>Class</code>.
	 * @param _resource  the resource to get the stream
	 * @param _class     the <code>Class</code> to use the <code>ClassLoader</code> from
	 * @return  the stream of the specified resource
	 * @throws IOException if an error occurs opening the stream of the resource or if it is not found
	 */
	public static InputStream getResourceStream(String _resource, Class<?> _class) throws IOException {
		InputStream in = null;
		ClassLoader loader = SystemUtil.class.getClassLoader();
		if (loader != null)
			in = loader.getResourceAsStream( _resource);
		if (in == null)
			in = ClassLoader.getSystemResourceAsStream(_resource);
		if (in == null)
			throw new IOException("Could not find resource " + _resource);

		return in;
	}
}