package pt.digitalis.utils.documents.excel;

import pt.digitalis.utils.documents.IDocument;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Galaio da Silva <a href="mailto:jgalaio@digitalis.pt">jgalaio@digitalis.pt</a><br/>
 * @created Dec 16, 2009
 */
public interface IExcelDirectDocument extends IDocument
{

    /**
     * Add a value to a specific cell in Excel Document
     *
     * @param sheetNumber  the sheet number
     * @param rowPosition  the row position
     * @param cellPosition the cell position
     * @param value        the value to add
     */
    public void addCellValue(Integer sheetNumber, Integer rowPosition, Integer cellPosition, String value);

    /**
     * Add data to Excel Document. <br>
     * Each position in the List represents a row <br>
     * Each field in the bean object of the list represents a column
     *
     * @param data      the data to write.
     * @param fieldsMap the fields mapping key=bean field name, value=cell position in excel document
     */
    public void addData(List<?> data, LinkedHashMap<String, Integer> fieldsMap);

    /**
     * Add a list of data to Excel. <br>
     * Each position in the List represents a row <br>
     * propertie in the bean object of the list represents a column
     *
     * @param data   the data to write
     * @param fields the list representig the fields of the objects in the data list
     */
    public void addData(List<?> data, String[] fields);

    /**
     * Add a list of data to Excel. <br>
     * Each position in the List represents a row <br>
     * Each field in the bean object of the list represents a column
     *
     * @param data      the data to write.
     * @param fields    the list representig the fields of the objects in the data list
     * @param startRow  the start row position.
     * @param startCell the start cell position in row.
     */
    public void addData(List<?> data, String[] fields, Integer startRow, Integer startCell);

    /**
     * add a list of data to Excel. <br>
     * Each position in the List represents a row <br>
     * Each field in the bean object of the list represents a column
     *
     * @param data        the data to write.
     * @param fields      the list representig the fields of the objects in the data list
     * @param sheetNumber the sheet number.
     * @param startRow    the start row position in Excel.
     * @param startCell   the start cell position in row.
     */
    public void addData(List<?> data, String[] fields, Integer sheetNumber, Integer startRow, Integer startCell);

    /**
     * Add data for one line.
     *
     * @param data      the map with data.
     * @param fieldsMap the fields map (field name and column for each entry)
     */
    public void addData(Map<String, String> data, Map<String, Integer> fieldsMap);

    /**
     * Copy a number of rows to a target start position in Excel Document. <br>
     * The copyed block will start in the target position
     *
     * @param sheetNumber       the sheet number
     * @param startPosition     the start block row
     * @param endPosition       the end block row
     * @param targetRowPosition the target start position
     */
    public void copyBlockOfRows(int sheetNumber, int startPosition, int endPosition, int targetRowPosition);

    /**
     * Copy a row from on position to another in the Excel Documen. <br>
     *
     * @param sheetNumber            the sheet number
     * @param sourceRowPosition      the source row position
     * @param destinationRowPosition the destination row position
     */
    public void copyRow(int sheetNumber, int sourceRowPosition, int destinationRowPosition);

    /**
     * Delete a specific cell content and style in Excel Document
     *
     * @param sheetNumber  the sheet number
     * @param rowPosition  the row position
     * @param cellPosition the cell position
     */
    public void deleteCell(Integer sheetNumber, Integer rowPosition, Integer cellPosition);

    /**
     * Get the list of data containing the passed class. <br>
     * Each field of the passed class represents a column Each position of the list represents a row
     *
     * @param <T>         the generic.
     * @param theClass    the class to pass.
     * @param fields      the fields.
     * @param sheetNumber the sheet number
     * @param startRow    the start row
     * @param endRow      the end row
     * @param startCell   the start cell
     *
     * @return the list of data retrieved
     */
    public <T extends Object> List<T> getData(Class<T> theClass, String[] fields, Integer sheetNumber, Integer startRow,
            Integer endRow, Integer startCell);

    /**
     * Get the data for the range between start and end row in the cell specified.
     *
     * @param sheetNumber the sheet number in Excel.
     * @param startRow    the start row.
     * @param endRow      the end row.
     * @param cellNumber  the cell number.
     *
     * @return the list of data.
     */
    public List<String> getData(Integer sheetNumber, Integer startRow, Integer endRow, Integer cellNumber);

    /**
     * Get the data in the range of rows and cells
     *
     * @param sheetNumber the sheet number in Excel
     * @param startRow    the Start Row in Excel
     * @param endRow      the End Row in Excel
     * @param startCell   the Start Cell in Excel
     * @param endCell     the the End Cell in Excel
     *
     * @return the data
     */
    public List<String[]> getData(Integer sheetNumber, Integer startRow, Integer endRow, Integer startCell,
            Integer endCell);

    /**
     * Inspector for the 'rowPosition' attribute.
     *
     * @return the rowPosition value
     */
    public Integer getRowPosition();

    /**
     * Modifier for the 'rowPosition' attribute.
     *
     * @param rowPosition the new rowPosition value to set
     */
    public void setRowPosition(Integer rowPosition);

    /**
     * Inspector for the 'sheetNumber' attribute.
     *
     * @return the sheetNumber value
     */
    public Integer getSheetNumber();

    /**
     * Modifier for the 'sheetNumber' attribute.
     *
     * @param sheetNumber the new sheetNumber value to set
     */
    public void setSheetNumber(Integer sheetNumber);

    /**
     * Modifier for the 'cellPosition' attribute.
     *
     * @param cellPosition the new cellPosition value to set
     */
    public void setCellPosition(Integer cellPosition);

    /**
     * Set a value to a specific cell in Excel Document
     *
     * @param sheetNumber  the sheet number
     * @param rowPosition  the row position
     * @param cellPosition the cell position
     * @param value        the value to add
     */
    public void setCellValue(Integer sheetNumber, Integer rowPosition, Integer cellPosition, String value);

    /**
     * The setter for template row
     *
     * @param sheetNumber
     * @param rowNumber
     */
    public void setTemplateRow(Integer sheetNumber, Integer rowNumber);
}
