package pt.digitalis.utils.documents.template;

import freemarker.cache.FileTemplateLoader;
import freemarker.cache.StringTemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.DefaultObjectWrapper;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.config.FreeMarkerConfig;
import pt.digitalis.utils.documents.IDocument;
import pt.digitalis.utils.documents.exception.DocumentException;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Map;

/**
 * Runs an Freemarker script
 *
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a>
 * @created 2011 /06/19
 */
public class FreemarkerDocument implements IDocument
{

    /** Freemarker configuration */
    private final Configuration cfg;

    /** The report properties */
    private final Map<String, Object> properties;

    /** The template file name */
    private final String templateName;

    /**
     * The Number format.
     */
    private final String NUMBER_FORMAT = "0.############";

    /** File document name */
    private String name;

    /**
     * Constructor
     *
     * @param templatePath the template complete file path
     * @param properties   the document encoding
     *
     * @exception IOException if an IO Exception occurrs
     */
    public FreemarkerDocument(String templatePath, Map<String, Object> properties) throws IOException
    {
        StringBuffer templateBaseDir = new StringBuffer();
        // if (!templatePath.startsWith("/"))
        // {
        // templateBaseDir.append("/");
        // }

        templateBaseDir.append(templatePath.substring(0, templatePath.lastIndexOf("/")));
        String templateName = templatePath.substring(templatePath.lastIndexOf("/") + 1, templatePath.length());

        this.templateName = templateName;
        this.properties = properties;
        this.cfg = new Configuration();
        cfg.setNumberFormat(NUMBER_FORMAT);

        // Specify the data source where the template files come from.
        FileTemplateLoader templateLoader = new FileTemplateLoader(new File(
                Thread.currentThread().getContextClassLoader().getResource(templateBaseDir.toString()).getPath()));

        this.cfg.setTemplateLoader(templateLoader);
        // Specify how templates will see the data model. This is an advanced topic...
        // but just use this:
        this.cfg.setObjectWrapper(new DefaultObjectWrapper());
    }

    /**
     * Constructor for FreeMarker
     *
     * @param templateName the tamplate name
     * @param template     the Freemarker Template
     * @param properties   the rootMap
     *
     * @exception IOException if an IO Exception occurrs
     */
    public FreemarkerDocument(String templateName, String template, Map<String, Object> properties) throws IOException
    {
        this.templateName = templateName;
        this.properties = properties;
        this.cfg = new Configuration();
        cfg.setNumberFormat(NUMBER_FORMAT);

        StringTemplateLoader stringLoader = new StringTemplateLoader();
        stringLoader.putTemplate(templateName, template);

        this.cfg.setTemplateLoader(stringLoader);
        this.cfg.setObjectWrapper(new DefaultObjectWrapper());
    }

    /**
     * Export as output stream byte array output stream.
     *
     * @return the byte array output stream
     *
     * @see pt.digitalis.utils.documents.IDocument#exportAsOutputStream() pt.digitalis.utils.documents.IDocument#exportAsOutputStream()
     */
    @Override
    public ByteArrayOutputStream exportAsOutputStream()
    {
        ByteArrayOutputStream output = new ByteArrayOutputStream();

        try
        {
            String encoding = FreeMarkerConfig.getInstance().getEncoding();
            Writer out = null;

            if (encoding != null && !"".equals(encoding))
            {
                out = new OutputStreamWriter(output, encoding);
            }
            else
            {
                out = new OutputStreamWriter(output);
            }

            processTemplate(out);
            out.flush();
            output.close();
        }
        catch (IOException e)
        {
            output = null;
            e.printStackTrace();
        }
        catch (TemplateException e)
        {
            output = null;
            e.printStackTrace();
        }
        catch (ConfigurationException e)
        {
            output = null;
            e.printStackTrace();
        }

        return output;
    }

    /**
     * Get the processed template result as string
     *
     * @return the result
     *
     * @exception DocumentException if an error occurs
     */
    public String exportAsString() throws DocumentException
    {
        String output = null;
        Writer out = new StringWriter();

        try
        {
            processTemplate(out);
            out.flush();
            output = out.toString();
        }
        catch (IOException e)
        {
            throw new DocumentException(e);
        }
        catch (TemplateException e)
        {
            throw new DocumentException(e);
        }
        catch (ConfigurationException e)
        {
            throw new DocumentException(e);
        }

        return output;
    }

    /**
     * Get bytes byte [ ].
     *
     * @return the byte [ ]
     *
     * @see pt.digitalis.utils.documents.IDocument#getBytes() pt.digitalis.utils.documents.IDocument#getBytes()
     */
    @Override
    public byte[] getBytes()
    {
        return exportAsOutputStream().toByteArray();
    }

    /**
     * Gets name.
     *
     * @return the name
     *
     * @see pt.digitalis.utils.documents.IDocument#getName() pt.digitalis.utils.documents.IDocument#getName()
     */
    @Override
    public String getName()
    {
        return this.name;
    }

    /**
     * Sets name.
     *
     * @param name the name
     *
     * @see pt.digitalis.utils.documents.IDocument#setName(java.lang.String) pt.digitalis.utils.documents.IDocument#setName(java.lang.String)
     */
    @Override
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * Process a freemarker template merging the giving map with it and returning the generated file to the
     * generatedFilePath
     *
     * @param out the writer
     *
     * @exception IOException            Exception thrown when the generated file cannot be created or it cannot write
     *                                   in
     *                                   the generated file.
     * @exception TemplateException      Thrown when the template directory cannot be read.
     * @exception ConfigurationException the configuration exception
     */
    private void processTemplate(Writer out) throws IOException, TemplateException, ConfigurationException
    {

        String encoding = FreeMarkerConfig.getInstance().getEncoding();
        Template temp = null;
        if (encoding != null && !"".equals(encoding))
        {
            temp = cfg.getTemplate(templateName, encoding);
        }
        else
        {
            temp = cfg.getTemplate(templateName);
        }
        /* Merge data model with template */
        temp.process(properties, out);
    }

    /**
     * Save to file.
     *
     * @param fileName the file name
     *
     * @exception IOException the io exception
     * @see pt.digitalis.utils.documents.IDocument#saveToFile(java.lang.String) pt.digitalis.utils.documents.IDocument#saveToFile(java.lang.String)
     */
    @Override
    public void saveToFile(String fileName) throws IOException
    {

        BufferedWriter out;

        try
        {
            String encoding = FreeMarkerConfig.getInstance().getEncoding();
            if (encoding != null && !"".equals(encoding))
                out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(fileName), encoding));
            else
                out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(fileName)));

            processTemplate(out);
            out.flush();
            out.close();
        }
        catch (TemplateException e)
        {
            throw new IOException(e.getMessage());
        }
        catch (ConfigurationException e)
        {
            throw new IOException(e.getMessage());
        }
    }
}
