/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.utils.reporting;

import pt.digitalis.utils.reporting.exception.ReportingException;

import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Connection;
import java.util.Map;

/**
 * Defines the expected behavior for the reporting engines implementations.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2008/12/09
 */
public interface IReport
{

    /**
     * Compiles the report.
     *
     * @exception ReportingException if the report can't be compiled
     */
    public void compileReport() throws ReportingException;

    /**
     * Exports the report to a given format and to a file with a given name.
     *
     * @param outputFileName the output file name
     *
     * @exception ReportingException if the report can't be exported
     */
    public void exportToFile(String outputFileName) throws ReportingException;

    /**
     * Exports the report on a given format to a given output stream.
     *
     * @param outputStream the output stream
     *
     * @exception ReportingException if the report can't be exported
     */
    public void exportToStream(OutputStream outputStream) throws ReportingException;

    /**
     * Fills the report.
     *
     * @exception ReportingException if the report can't be filled
     */
    public void fillReport() throws ReportingException;

    /**
     * Fills the report with data from obtained through a JDBC connection.
     *
     * @param jdbcConnection the JDBC Connection to fill the report from
     *
     * @exception ReportingException if the report can't be filled
     */
    public void fillReportFromConnection(Connection jdbcConnection) throws ReportingException;

    /**
     * Fills the report with data from a data source.
     *
     * @param dataSource the data source used to populate the report
     *
     * @exception ReportingException if the report can't be filled
     */
    public void fillReportFromDataSource(IReportDataSource dataSource) throws ReportingException;

    /**
     * Generates a report from the given parameters. Combines the three previous operations in a single step.
     *
     * @param jdbcConnection the connection
     * @param outputStream   the output stream
     *
     * @exception ReportingException if the report can't be generated
     */
    public void generateReport(Connection jdbcConnection, OutputStream outputStream) throws ReportingException;

    /**
     * Generates a report from the given parameters. Combines the three previous operations in a single step.
     *
     * @param jdbcConnection the connection
     * @param outputFileName the output file name
     *
     * @exception ReportingException if the report can't be generated
     */
    public void generateReport(Connection jdbcConnection, String outputFileName) throws ReportingException;

    /**
     * Generates a report from the given parameters. Combines the three previous operations in a single step.
     *
     * @param dataSource   the data source used to populate the report
     * @param outputStream the output stream
     *
     * @exception ReportingException if the report can't be generated
     */
    public void generateReport(IReportDataSource dataSource, OutputStream outputStream) throws ReportingException;

    /**
     * Generates a report from the given parameters. Combines the three previous operations in a single step.
     *
     * @param dataSource     the data source used to populate the report
     * @param outputFileName the output file name
     *
     * @exception ReportingException if the report can't be generated
     */
    public void generateReport(IReportDataSource dataSource, String outputFileName) throws ReportingException;

    /**
     * Generates a report from the given parameters. Combines the three previous operations in a single step.
     *
     * @param outputFileName the output file name
     *
     * @exception ReportingException if the report can't be generated
     */
    public void generateReport(String outputFileName) throws ReportingException;

    /**
     * Returns the defined report export format.
     *
     * @return the defined report export format
     */
    public ReportExportFormat getExportFormat();

    /**
     * Changes the defined report export format.
     *
     * @param exportFormat the new exportFormat value to set
     */
    public void setExportFormat(ReportExportFormat exportFormat);

    /**
     * Returns the defined parameters.
     *
     * @return the defined parameters
     */
    public Map<String, Object> getParameters();

    /**
     * Changes the defined parameters.
     *
     * @param parameters the new parameters value to set
     */
    public void setParameters(Map<String, Object> parameters);

    /**
     * Returns the defined template input stream.
     *
     * @return the defined template path
     */
    public InputStream getTemplateInputStream();

    /**
     * Changes the defined template input stream.
     *
     * @param templateInputStream the new templateInputStream value to set
     */
    public void setTemplateInputStream(InputStream templateInputStream);

    /**
     * Returns the defined template path.
     *
     * @return the defined template path
     */
    public String getTemplatePath();

    /**
     * Changes the defined template path.
     *
     * @param templatePath the new templatePah value to set
     */
    public void setTemplatePath(String templatePath);
}
