/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.bytecode.holders;

import java.util.Map;

import javassist.CtField;
import javassist.NotFoundException;
import pt.digitalis.utils.CodeGenUtil4Javassist;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

/**
 * An helper class that helps manage the temporary attributes in the code generation process. Provides managing methods
 * and caches the details read from the attribute.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Sep 23, 2007
 */
public class AttributeHolder {

    /** The attribute object. */
    private CtField field;

    /** The attribute's parent class name. */
    private String parentClassName;

    /**
     * Constructs an holder from a parent class and a field object.
     * 
     * @param clazz
     *            the parent class
     * @param field
     *            the field/attribute to manage
     * @throws ResourceNotFoundException
     *             if the class can't be read
     */
    public AttributeHolder(ClassHolder clazz, CtField field) throws ResourceNotFoundException
    {
        this.parentClassName = clazz.getUniqueID();
        this.field = field;
    }

    /**
     * Returns the attribute's parent class.
     * 
     * @return the parent class
     * @throws ResourceNotFoundException
     *             if the parent class can't be read
     */
    public ClassHolder getParentClass() throws ResourceNotFoundException
    {
        return new ClassHolder(HolderRepository.getClass(parentClassName));
    }

    /**
     * Returns the attribute's parent class name.
     * 
     * @return the parent class name
     */
    public String getParentClassName()
    {
        return parentClassName;
    }

    /**
     * Returns a unique ID for this attribute in it's context (class/method/attribute).
     * 
     * @return the unique ID
     */
    public String getUniqueID()
    {
        return parentClassName + "." + getName();
    }

    /**
     * Returns the attribute name.
     * 
     * @return the attribute name
     */
    public String getName()
    {
        return field.getName();
    }

    /**
     * Returns the attribute signature.
     * 
     * @return the attribute signature
     */
    public String getSignature()
    {
        return field.getSignature();
    }

    /**
     * Returns the annotations of the managed method.
     * 
     * @return the annotation list
     * @throws ResourceNotFoundException
     *             if annotations can't be read
     */
    public Map<String, AnnotationHolder> getAnnotations() throws ResourceNotFoundException
    {
        // Get the attribute's annotations
        Map<String, AnnotationHolder> attributeAnnotations = HolderRepository.getAnnotations(getUniqueID());

        // Lazy loading for the attribute's annotations...
        if (attributeAnnotations == null)
        {
            // Get attribute's annotations from javassist
            attributeAnnotations = CodeGenUtil4Javassist.getFieldAnnotations(this);
            // Add the annotations to the repository
            HolderRepository.addAnnotations(getUniqueID(), attributeAnnotations);
        }

        return attributeAnnotations;
    }

    /**
     * Validates if a given annotation is used on the attribute
     * 
     * @param annotationName
     *            the name of the annotation to search
     * @return T if the annotation is used, F otherwise
     * @throws ResourceNotFoundException
     *             if annotations can't be read
     */
    public boolean containsAnnotation(String annotationName) throws ResourceNotFoundException
    {
        return getAnnotations().containsKey(annotationName);
    }

    /**
     * Returns the attribute object
     * 
     * @return the attribute object
     */
    public CtField getManagedAttribute()
    {
        return this.field;
    }

    /**
     * Returns the attribute class type
     * 
     * @return the class type
     */
    public String getAttributeType()
    {
        try
        {
            return getManagedAttribute().getType().getName();
        }
        catch (NotFoundException e)
        {

            // Should never reach here, so return null. The exception will be captured by the exception aspect
            return null;
        }
    }
}