/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.bytecode.holders;

import javassist.bytecode.annotation.Annotation;
import pt.digitalis.utils.CodeGenUtil4Javassist;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.util.Map;

/**
 * An helper class that helps to manage the temporary annotations in the code generation process. Provides managing
 * methods and caches the details read from the annotation. This class allows storing of type, method or field
 * annotations. Constructors and getters are provided for each kind of use.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Sep 23, 2007
 */
public class AnnotationHolder
{

    /** The annotation to object. */
    private Annotation annotation;

    /** The annotation's parent class name. */
    private String parentClassName;

    /** The annotation's parent method name. */
    private String parentMethodName;

    /** The annotation's parent method name. */
    private String parentAttributeName;

    /** If T the repository will not be used */
    private boolean noCacheMode = false;

    /**
     * Constructs an holder from a parent class and an annotation
     *
     * @param clazz       the parent class
     * @param annotation  the annotation to manage
     * @param noCacheMode if T the repository will not be used
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    public AnnotationHolder(ClassHolder clazz, Annotation annotation, boolean noCacheMode)
            throws ResourceNotFoundException
    {
        this.parentClassName = clazz.getUniqueID();
        this.annotation = annotation;
        this.noCacheMode = noCacheMode;
    }

    /**
     * Constructs an holder from a parent class and an annotation
     *
     * @param clazz      the parent class
     * @param annotation the annotation to manage
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    public AnnotationHolder(ClassHolder clazz, Annotation annotation) throws ResourceNotFoundException
    {
        this(clazz, annotation, false);
    }

    /**
     * Constructs an holder from a parent method and an annotation
     *
     * @param method     the parent method
     * @param annotation the annotation to manage
     */
    public AnnotationHolder(MethodHolder method, Annotation annotation)
    {
        this.parentClassName = method.getParentClassName();
        this.parentMethodName = method.getName();
        this.annotation = annotation;
    }

    /**
     * Constructs an holder from a parent attribute and an annotation
     *
     * @param attribute  the parent attribute
     * @param annotation the annotation to manage
     */
    public AnnotationHolder(AttributeHolder attribute, Annotation annotation)
    {
        this(attribute, annotation, false);
    }

    /**
     * Constructs an holder from a parent attribute and an annotation
     *
     * @param attribute   the parent attribute
     * @param annotation  the annotation to manage
     * @param noCacheMode if T the repository will not be used
     */
    public AnnotationHolder(AttributeHolder attribute, Annotation annotation, boolean noCacheMode)
    {
        this.parentClassName = attribute.getParentClassName();
        this.parentAttributeName = attribute.getName();
        this.annotation = annotation;
        this.noCacheMode = noCacheMode;
    }

    /**
     * Returns the annotation object.
     *
     * @return the annotation object
     */
    public Annotation getManagedAnnotation()
    {
        return this.annotation;
    }

    /**
     * Returns a map with the annotation's members
     *
     * @return a map with the annotation's members
     *
     * @exception ResourceNotFoundException if the member values can't be found
     */
    public Map<String, AnnotationMemberValueHolder> getMembers() throws ResourceNotFoundException
    {

        // Get the annotation's members for this annotation
        Map<String, AnnotationMemberValueHolder> annotationMembers = null;

        if (!noCacheMode)
            annotationMembers = HolderRepository.getAnnotationMembers(getUniqueID());

        // Lazy loading for the annotation members...
        if (annotationMembers == null)
        {
            // Get annotation members from javassist
            annotationMembers = CodeGenUtil4Javassist.getAnnotationMemberValues(annotation);
            // Add the members to the repository
            HolderRepository.addAnnotationMembers(getUniqueID(), annotationMembers);
        }

        return annotationMembers;
    }

    /**
     * Returns all the meta annotations of this annotation.
     *
     * @return a map with all meta-annotations
     *
     * @exception ResourceNotFoundException if the annotation isn't annotated
     */
    public Map<String, AnnotationHolder> getMetaAnnotations() throws ResourceNotFoundException
    {
        return new ClassHolder(this.getName()).getAnnotations();
    }

    /**
     * Returns the annotation name.
     *
     * @return the annotation name
     */
    public String getName()
    {
        return this.annotation.getTypeName();
    }

    /**
     * The annotation's parent attribute
     *
     * @return the parent attribute if it exists or null if it's a class or method annotation
     */
    public AttributeHolder getParentAttribute()
    {
        AttributeHolder parentAttribute = null;

        // If it's an attribute annotation...
        if (parentAttributeName != null)
        {
            // Get attributes from the annotation's parent class
            Map<String, AttributeHolder> classAttributes = HolderRepository.getAttributes(parentClassName);

            // If there are any attributes, find the annotation parent method
            if (classAttributes != null)
                parentAttribute = classAttributes.get(parentAttributeName);
        }

        return parentAttribute;
    }

    /**
     * Returns the annotation's parent class. <b>Note:</b> Even if this annotation refers to a method or attribute it
     * will return the method/attribute parent class.
     *
     * @return the parent class
     *
     * @exception ResourceNotFoundException it the parent class can't be found
     */
    public ClassHolder getParentClass() throws ResourceNotFoundException
    {
        return new ClassHolder(parentClassName);
    }

    /**
     * Returns the annotation's parent class name.
     *
     * @return the parent class name
     */
    public String getParentClassName()
    {
        return parentClassName;
    }

    /**
     * Returns the annotation's parent method.
     *
     * @return the parent method if it exists or null if it's a class or attribute annotation
     *
     * @exception ResourceNotFoundException
     */
    public MethodHolder getParentMethod() throws ResourceNotFoundException
    {
        MethodHolder parentMethod = null;

        // If it's a method annotation...
        if (parentMethodName != null)
        {
            // Get methods from the annotation's parent class
            Map<String, MethodHolder> classMethods = HolderRepository.getMethods(parentClassName);

            // If there are any methods,find the annotation parent method
            if (classMethods != null)
                parentMethod = classMethods.get(parentMethodName);
        }

        return parentMethod;
    }

    /**
     * Returns a unique ID for this annotation in it's context (class/method/attribute). T
     *
     * @return the unique ID
     */
    public String getUniqueID()
    {

        String id = null;

        if (isClassAnnotation())
            id = parentClassName + "." + getName();
        else if (isMethodAnnotation())
            id = parentClassName + "." + parentMethodName + "." + getName();
        else if (isAttributeAnnotation())
            id = parentClassName + "." + parentAttributeName + "." + getName();
        return id;
    }

    /**
     * Checks if this annotation refers to an attribute.
     *
     * @return T if it's an attribute annotation, F otherwise
     */
    public boolean isAttributeAnnotation()
    {
        return (parentAttributeName != null);
    }

    /**
     * Checks if this annotation refers to a class.
     *
     * @return T if it's a class annotation, F otherwise
     */
    public boolean isClassAnnotation()
    {
        return (parentAttributeName == null && parentMethodName == null);
    }

    /**
     * Checks if this annotation refers to a method.
     *
     * @return T if it's a method annotation, F otherwise
     */
    public boolean isMethodAnnotation()
    {
        return (parentMethodName != null);
    }
}