/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.bytecode.holders;

import javassist.CtClass;
import pt.digitalis.utils.CodeGenUtil4Javassist;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * This class will manage the repository of CtClass objects and all its methods, attributes, and annotations. The
 * CtClass objects are updated during the code generation process. The other objects are just cached to reduce loading
 * time. This is possible because the enhancement takes place on the CtClass object level. The other objects are only
 * read to DEM building and validation processes.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Sep 25, 2007
 */
public class HolderRepository
{

    /**
     * Repository of class/method annotations member values. Members are NOT updated and serve as a cache for the
     * initial snapshot before code generation takes place. <K,<K, V>> = <parent class name, <member name, member
     * object>>
     */
    static private Map<String, Map<String, AnnotationMemberValueHolder>> annotationMembers =
            new ConcurrentHashMap<String, Map<String, AnnotationMemberValueHolder>>();

    /**
     * Repository of class/method annotations. Annotation are NOT updated and serve as a cache for the initial snapshot
     * before code generation takes place. <K,<K, V>> = <parent class name, <annotation name, annotation object>>
     */
    static private Map<String, Map<String, AnnotationHolder>> annotations =
            new ConcurrentHashMap<String, Map<String, AnnotationHolder>>();

    /**
     * Repository of class attributes. Attributes are NOT updated and serve as a cache for the initial snapshot before
     * code generation takes place. <K,<K, V>> = <parent class name, <attribute name, attribute object>>
     */
    static private Map<String, Map<String, AttributeHolder>> attributes =
            new ConcurrentHashMap<String, Map<String, AttributeHolder>>();

    /**
     * CtClass central repository for class enhancement. The CtClass objects are updated when the code enhancement takes
     * place. <K,V> = <class name, class object>
     */
    static private Map<String, CtClass> classes = new ConcurrentHashMap<String, CtClass>();

    /**
     * Repository of class methods. Methods are NOT updated and serve as a cache for the initial snapshot before code
     * generation takes place. <K,<K, V>> = <parent class name, <method name, method object>>
     */
    static private Map<String, Map<String, MethodHolder>> methods =
            new ConcurrentHashMap<String, Map<String, MethodHolder>>();

    /**
     * Adds class/method/attribute annotation members to the repository.
     *
     * @param id                the id of the class/method/attribute
     * @param annotationMembers the annotation members to set
     */
    static public void addAnnotationMembers(String id, Map<String, AnnotationMemberValueHolder> annotationMembers)
    {
        HolderRepository.annotationMembers.put(id, annotationMembers);
    }

    /**
     * Adds class/method/attribute annotations to the repository.
     *
     * @param id          the id of the class/method/attribute
     * @param annotations the annotations to set
     */
    static public void addAnnotations(String id, Map<String, AnnotationHolder> annotations)
    {
        HolderRepository.annotations.put(id, annotations);
    }

    /**
     * Adds a given class's attributes to the repository.
     *
     * @param id         the class's id
     * @param attributes the attributes to set
     */
    public static void addAttributes(String id, Map<String, AttributeHolder> attributes)
    {
        HolderRepository.attributes.put(id, attributes);
    }

    /**
     * Adds a given class to the repository.
     *
     * @param clazz the class to store
     */
    static public void addClass(CtClass clazz)
    {
        classes.put(clazz.getName(), clazz);
    }

    /**
     * Adds a given class's methods to the repository.
     *
     * @param id      the class's id
     * @param methods the methods to set
     */
    public static void addMethods(String id, Map<String, MethodHolder> methods)
    {
        HolderRepository.methods.put(id, methods);
    }

    /**
     * Returns T if the class exists on the repository, F otherwise
     *
     * @param id the class id
     *
     * @return T if the class exists, F otherwise
     */
    static public boolean classExists(String id)
    {
        if (classes == null)
            return false;
        else
            return (classes.get(id) != null) ? true : false;
    }

    /**
     * Makes internal data containers eligible for GC when they are no longer needed. Must be called explicitly since
     * this is a static class with static attributes for temporary data.
     */
    static public void cleanUp()
    {
        classes = null;
        methods = null;
        attributes = null;
        annotations = null;
        annotationMembers = null;
    }

    /**
     * Removes all annotations cached with the given ID.
     *
     * @param id the id that aggregates the annotations to remove
     */
    private static void deleteAnnotations(String id)
    {
        // Get the annotations with the given id
        Map<String, AnnotationHolder> elementAnnotations = annotations.get(id);

        // If there are some, free their members...
        if (elementAnnotations != null)
            for (AnnotationHolder annotation : elementAnnotations.values())
            {
                annotationMembers.remove(annotation.getUniqueID());
            }

        // Remove the annotations from cache
        annotations.remove(id);
    }

    /**
     * Removes from the repository all the previously cached entries referring this class
     *
     * @param id the id of the class to remove
     */
    public static void deleteClassAndReferences(String id)
    {

        // Get class methods and attributes...
        Map<String, MethodHolder> classMethods = methods.get(id);
        Map<String, AttributeHolder> classAttributes = attributes.get(id);

        // If there are any cached methods, remove it's annotations
        if (classMethods != null)
            for (MethodHolder method : classMethods.values())
            {
                deleteAnnotations(method.getUniqueID());
            }

        // If there are any cached attributes, remove it's annotations
        if (classAttributes != null)
            for (AttributeHolder attribute : classAttributes.values())
            {
                deleteAnnotations(attribute.getUniqueID());
            }

        // Remove class annotations
        deleteAnnotations(id);

        // Remove class's methods and attributes
        methods.remove(id);
        attributes.remove(id);

        // remove the class itself
        classes.remove(id);
    }

    /**
     * Returns the annotation member list for the given class/method/attribute id.
     *
     * @param id the id of the class/method/attribute
     *
     * @return the annotation members
     */
    static public Map<String, AnnotationMemberValueHolder> getAnnotationMembers(String id)
    {
        return annotationMembers.get(id);
    }

    /**
     * Returns the class/method/attribute annotation list for the given id that are currently in the repository. Null if
     * not loaded yet. Note: This method does not load the annotation. It is simply a getter method. The caller must add
     * the annotation on first access in they don't yet exist.
     *
     * @param id the id of the class/method/attribute set
     *
     * @return the list of the class/method/attribute annotations present in the repository
     *
     * @see ClassHolder
     * @see MethodHolder
     * @see AttributeHolder
     */
    static public Map<String, AnnotationHolder> getAnnotations(String id)
    {

        return annotations.get(id);
    }

    /**
     * Returns a list of attributes belonging to the class given.
     *
     * @param id the class's id
     *
     * @return a list with the class's attributes
     */
    public static Map<String, AttributeHolder> getAttributes(String id)
    {
        return attributes.get(id);
    }

    /**
     * Return the CtClass object
     *
     * @param className the class name to search
     *
     * @return the class
     *
     * @exception ResourceNotFoundException if the class can't be loaded
     */
    static public CtClass getClass(String className) throws ResourceNotFoundException
    {

        CtClass clazz = classes.get(className);

        if (clazz == null)
            addClass(CodeGenUtil4Javassist.getClass(className));

        return clazz;
    }

    /**
     * Returns a list of methods belonging to the class given.
     *
     * @param id the class's id
     *
     * @return a list with the class's methods
     *
     * @exception ResourceNotFoundException if class can't be found
     */
    static public Map<String, MethodHolder> getMethods(String id) throws ResourceNotFoundException
    {

        return methods.get(id);
    }

    /**
     * Initializes the internal objects. Useful for testing purposes after cleanUp has been called
     */
    static public void initializeInternals()
    {
        if (classes == null)
            classes = new HashMap<String, CtClass>();

        if (methods == null)
            methods = new HashMap<String, Map<String, MethodHolder>>();

        if (attributes == null)
            attributes = new HashMap<String, Map<String, AttributeHolder>>();

        if (annotations == null)
            annotations = new HashMap<String, Map<String, AnnotationHolder>>();

        if (annotationMembers == null)
            annotationMembers = new HashMap<String, Map<String, AnnotationMemberValueHolder>>();
    }

    /**
     * Updates a given class in the repository. Add the class to the repository if it doesn't exists yet.
     *
     * @param clazz the class to store
     */
    static public void updateClass(CtClass clazz)
    {
        /*
         * Implementation Note: since the Java's Map interface 'put' method replaces an object with a given key if the
         * key already exists, the HolderRepository#addClass(CtClass) method can be used to perform the task. From a
         * user's point of view it is clearer to separate the update and addition operations. By providing two separate
         * operations it is possible to achieve better code legibility and a more suited interface decomposition.
         */
        addClass(clazz);
    }
}
