/**
 * (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos num.9 - Piso -1
 * 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.utils;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtField;
import javassist.CtMethod;
import javassist.CtNewMethod;
import javassist.LoaderClassPath;
import javassist.NotFoundException;
import javassist.bytecode.AnnotationsAttribute;
import javassist.bytecode.ClassFile;
import javassist.bytecode.ConstPool;
import javassist.bytecode.FieldInfo;
import javassist.bytecode.MethodInfo;
import javassist.bytecode.annotation.Annotation;
import javassist.bytecode.annotation.AnnotationMemberValue;
import javassist.bytecode.annotation.ArrayMemberValue;
import javassist.bytecode.annotation.BooleanMemberValue;
import javassist.bytecode.annotation.ByteMemberValue;
import javassist.bytecode.annotation.CharMemberValue;
import javassist.bytecode.annotation.ClassMemberValue;
import javassist.bytecode.annotation.DoubleMemberValue;
import javassist.bytecode.annotation.EnumMemberValue;
import javassist.bytecode.annotation.FloatMemberValue;
import javassist.bytecode.annotation.IntegerMemberValue;
import javassist.bytecode.annotation.LongMemberValue;
import javassist.bytecode.annotation.MemberValue;
import javassist.bytecode.annotation.ShortMemberValue;
import javassist.bytecode.annotation.StringMemberValue;
import pt.digitalis.utils.bytecode.ByteCodeStartupConfiguration;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.bytecode.holders.AnnotationHolder;
import pt.digitalis.utils.bytecode.holders.AnnotationMemberValueHolder;
import pt.digitalis.utils.bytecode.holders.AttributeHolder;
import pt.digitalis.utils.bytecode.holders.ClassHolder;
import pt.digitalis.utils.bytecode.holders.MethodHolder;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This class presents a set of methods and constants to aid the code generation. The methods presented here use the
 * javassist API. Some of these methods present similar functionality to those on CodeGenUtil4Reflection but the
 * underlying technology is different. Methods from both classes can't coexist on the same class file due to the fact
 * that they share types with the same name and if they're mixed together a name clash occurs.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created 2007/08/30
 */
public class CodeGenUtil4Javassist
{

    /**
     * Adds code to a given method's body after the existing code.
     *
     * @param methodSourceToAdd the source code to inject on the method
     * @param method            the method to enhance
     *
     * @return the method with the source added
     *
     * @exception CodeGenerationException if method's body cannot be replaced
     */
    static public CtMethod addCodeAfterMethodBody(String methodSourceToAdd, CtMethod method)
            throws CodeGenerationException
    {

        long initTime = System.currentTimeMillis();

        if (ByteCodeStartupConfiguration.isShowSource())
        // LOG Execution times code
        {
            System.out.println("Compiling source for method \"" + method.getName() + "\": \n     | " +
                               methodSourceToAdd.replace("\n", "\n    | "));
        }

        try
        {
            method.insertAfter(methodSourceToAdd);
        }
        catch (CannotCompileException cannotCompileException)
        {
            throw new CodeGenerationException(
                    "Could not compile source code for method " + method.getName() + " ( Cause: " +
                    cannotCompileException.getReason() + " Source code: " + methodSourceToAdd + ")",
                    cannotCompileException);
        }

        if (ByteCodeStartupConfiguration.isTrackTimes())
        // LOG Execution times code
        {
            System.out.println(
                    method.getDeclaringClass().getSimpleName() + " - Added after Method: " + method.getName() + " (" +
                    (System.currentTimeMillis() - initTime) + "ms)");
        }

        return method;
    }

    /**
     * Adds code to a given method's body before the existing code.
     *
     * @param methodSourceToAdd the source code to inject on the method
     * @param method            the method to enhance
     *
     * @return the method with the source added
     *
     * @exception CodeGenerationException if method's body cannot be replaced
     */
    static public CtMethod addCodeBeforeMethodBody(String methodSourceToAdd, CtMethod method)
            throws CodeGenerationException
    {

        long initTime = System.currentTimeMillis();

        if (ByteCodeStartupConfiguration.isShowSource())
        // LOG Execution times code
        {
            System.out.println("Compiling source for method \"" + method.getName() + "\": \n     | " +
                               methodSourceToAdd.replace("\n", "\n    | "));
        }

        try
        {
            method.insertBefore(methodSourceToAdd);
        }
        catch (CannotCompileException cannotCompileException)
        {
            throw new CodeGenerationException(
                    "Could not compile source code for method " + method.getName() + " ( Cause: " +
                    cannotCompileException.getReason() + " Source code: " + methodSourceToAdd + ")",
                    cannotCompileException);
        }

        if (ByteCodeStartupConfiguration.isTrackTimes())
        // LOG Execution times code
        {
            System.out.println(
                    method.getDeclaringClass().getSimpleName() + " - Added before Method: " + method.getName() + " (" +
                    (System.currentTimeMillis() - initTime) + "ms)");
        }

        return method;
    }

    /**
     * Adds an interface to a given class.
     *
     * @param interfaceClass   the interface
     * @param destinationClass the class to implement the interface
     *
     * @return the enhanced class with the interface
     */
    static public CtClass addInterface(CtClass interfaceClass, CtClass destinationClass)
    {
        if (destinationClass.isFrozen())
        {
            destinationClass.defrost();
        }

        destinationClass.addInterface(interfaceClass);

        return destinationClass;
    }

    /**
     * Adds a method source code to a given class.
     *
     * @param methodSource the method's source code
     * @param clazz        the class to add the method to
     *
     * @return the class with the added method
     *
     * @exception CodeGenerationException if the method's source code can't be compiled
     */
    static public CtClass addMethodSourceToClass(String methodSource, CtClass clazz) throws CodeGenerationException
    {
        return addMethodSourceToClass(methodSource, clazz, false);
    }

    /**
     * Adds a method source code to a given class.
     *
     * @param methodSource     the method's source code
     * @param clazz            the class to add the method to
     * @param overrideExisting if T will override any existing method with the same name. if F will keep the existing
     *                         method and
     *                         ignore the new one
     *
     * @return the class with the added method
     *
     * @exception CodeGenerationException if the method's source code can't be compiled
     */
    static public CtClass addMethodSourceToClass(String methodSource, CtClass clazz, boolean overrideExisting)
            throws CodeGenerationException
    {
        long initTime = System.currentTimeMillis();

        if (clazz.isFrozen())
        {
            clazz.defrost();
        }

        // Create method holder...
        CtMethod method = null;

        if (ByteCodeStartupConfiguration.isShowSource())
        // LOG Execution times code
        {
            System.out.println("Compiling new Method: \n     | " + methodSource.replace("\n", "\n    | "));
        }

        try
        {
            // Create method (compile source)
            method = CtNewMethod.make(methodSource, clazz);

            if (overrideExisting || !methodExists(method.getName(), clazz))
            {
                clazz.addMethod(method);
            }
        }
        catch (CannotCompileException cannotCompileException)
        {
            throw new CodeGenerationException(
                    "Could not compile source code for initialization method on class " + clazz.getName() +
                    " (Source err?) [source = " + methodSource + "]", cannotCompileException);
        }

        if (ByteCodeStartupConfiguration.isTrackTimes())
        // LOG Execution times code
        {
            System.out.println(clazz.getSimpleName() + " - Added Method: " + methodSource.substring(0, 50) + " (" +
                               (System.currentTimeMillis() - initTime) + "ms)");
        }

        return clazz;
    }

    /**
     * Searches a given class for a given method name.
     *
     * @param attributeName the method's name
     * @param clazz         the class to search
     *
     * @return T if the class already contains the method, F otherwise
     */
    static public boolean attributeExists(String attributeName, CtClass clazz)
    {
        boolean result = false;

        // Get class's methods...
        CtField[] attributes = clazz.getDeclaredFields();

        /*
         * Implementation Note: doesn't use Javassist's getDeclaredField(String) because it throws an exception if
         * method's not found.
         */

        // Check each method's name for a match with the given name
        for (int i = 0; i < attributes.length; i++)
        {
            if (attributes[i].getName().equals(attributeName))
            {
                result = true;
                break;
            }
        }

        return result;
    }

    /**
     * Converts a given array member value to an array of annotations.
     *
     * @param arrayMemberValue the array member value
     *
     * @return a string array with the array member value elements
     */
    static public AnnotationMemberValue[] convertMemberValueToAnnotationArray(ArrayMemberValue arrayMemberValue)
    {
        return (AnnotationMemberValue[]) arrayMemberValue.getValue();
    }

    /**
     * Converts a given member value to Boolean.
     *
     * @param memberValue the member value to convert
     *
     * @return a String representation of the memberValue
     */
    static public Boolean convertMemberValueToBoolean(MemberValue memberValue)
    {
        return ((BooleanMemberValue) memberValue).getValue();
    }

    /**
     * Converts a given member value to String.
     *
     * @param memberValue the member value to convert
     *
     * @return a String representation of the memberValue
     */
    static public String convertMemberValueToString(MemberValue memberValue)
    {
        return ((StringMemberValue) memberValue).getValue();
    }

    /**
     * Converts a given array member value to an array of strings.
     *
     * @param arrayMemberValue the array member value
     *
     * @return a string array with the array member value elements
     */
    static public String[] convertMemberValueToStringArray(ArrayMemberValue arrayMemberValue)
    {
        String[] entries = null;

        return entries;
    }

    /**
     * Converts an annotation member value into a MemberValue type. The conversion to ArrayMemberValue is not supported
     * since the method parameter is not an array.
     *
     * @param originalValue the original value to convert from
     *
     * @return a MemberValue converted type
     */
    static private MemberValue convertToMemberValue(Object originalValue)
    {

        // Get the original runtime type...
        Class<?> originalValueType = originalValue.getClass();

        // ...perform the appropriate cast based on the rt type.
        if (originalValueType == Annotation.class)
        {
            AnnotationMemberValue convertedValue =
                    new AnnotationMemberValue(new ConstPool("javassist.bytecode.AnnotationMemberValue"));
            convertedValue.setValue((Annotation) originalValue);

            return convertedValue;
        }
        else if (originalValueType == Boolean.class)
        {
            BooleanMemberValue convertedValue =
                    new BooleanMemberValue(new ConstPool("javassist.bytecode.BooleanMemberValue"));
            convertedValue.setValue(((Boolean) originalValue));

            return convertedValue;
        }
        else if (originalValueType == Byte.class)
        {
            ByteMemberValue convertedValue = new ByteMemberValue(new ConstPool("javassist.bytecode.ByteMemberValue"));
            convertedValue.setValue(((Byte) originalValue));

            return convertedValue;
        }
        else if (originalValueType == Character.class)
        {
            CharMemberValue convertedValue = new CharMemberValue(new ConstPool("javassist.bytecode.CharMemberValue"));
            convertedValue.setValue((Character) originalValue);

            return convertedValue;
        }
        else if (originalValueType == Class.class)
        {
            ClassMemberValue convertedValue =
                    new ClassMemberValue(new ConstPool("javassist.bytecode.ClassMemberValue"));
            convertedValue.setValue(((Class<?>) originalValue).getCanonicalName());

            return convertedValue;
        }
        else if (originalValueType == Double.class)
        {
            DoubleMemberValue convertedValue =
                    new DoubleMemberValue(new ConstPool("javassist.bytecode.DoubleMemberValue"));
            convertedValue.setValue((Double) originalValue);

            return convertedValue;
        }
        else if (originalValueType == Enum.class)
        {
            EnumMemberValue convertedValue = new EnumMemberValue(new ConstPool("javassist.bytecode.EnumMemberValue"));
            convertedValue.setValue(((Enum<?>) originalValue).name());

            return convertedValue;
        }
        else if (originalValueType == Float.class)
        {
            FloatMemberValue convertedValue =
                    new FloatMemberValue(new ConstPool("javassist.bytecode.FloatMemberValue"));
            convertedValue.setValue((Float) originalValue);

            return convertedValue;
        }
        else if (originalValueType == Integer.class)
        {
            IntegerMemberValue convertedValue =
                    new IntegerMemberValue(new ConstPool("javassist.bytecode.IntegerMemberValue"));
            convertedValue.setValue((Integer) originalValue);

            return convertedValue;
        }
        else if (originalValueType == Long.class)
        {
            LongMemberValue convertedValue = new LongMemberValue(new ConstPool("javassist.bytecode.LongMemberValue"));
            convertedValue.setValue((Long) originalValue);

            return convertedValue;
        }
        else if (originalValueType == Short.class)
        {
            ShortMemberValue convertedValue =
                    new ShortMemberValue(new ConstPool("javassist.bytecode.ShortMemberValue"));
            convertedValue.setValue((Short) originalValue);

            return convertedValue;
        }
        else if (originalValueType == String.class)
        {
            StringMemberValue convertedValue =
                    new StringMemberValue(new ConstPool("javassist.bytecode.StringMemberValue"));
            convertedValue.setValue(((String) originalValue));

            return convertedValue;
        }
        else
        {
            return null;
        }
    }

    /**
     * Copies the class attributes to another class.
     *
     * @param sourceClass      the source class
     * @param destinationClass the destination class
     *
     * @return the destination class with the source class body
     *
     * @exception CodeGenerationException if the copied element already exists on the destination class OR if the copied
     *                                    element can't be
     *                                    compiled
     */
    static public CtClass copyClassAttributes(CtClass sourceClass, CtClass destinationClass)
            throws CodeGenerationException
    {

        // Get source class attributes
        CtField[] sourceClassAttributes = sourceClass.getDeclaredFields();

        // For each retrieved attribute...
        for (int i = 0; i < sourceClassAttributes.length; i++)
        {
            // Check if it already exists on destination class
            if (!attributeExists(sourceClassAttributes[i].getName(), destinationClass))
            {
                try
                {
                    // If not add it...
                    CtField attributeByteCode = new CtField(sourceClassAttributes[i], destinationClass);
                    destinationClass.addField(attributeByteCode);
                }
                catch (CannotCompileException cannotCompileException)
                {
                    throw new CodeGenerationException(
                            "Could not add attribute " + sourceClassAttributes[i].getName() + " to class " +
                            destinationClass.getName() + "(" + cannotCompileException.getReason() + ")",
                            cannotCompileException);
                }
            }
        }

        // Return enhanced class
        return destinationClass;
    }

    /**
     * Copies the class interfaces to another class.
     *
     * @param sourceClass      the source class
     * @param destinationClass the destination class
     *
     * @return the destination class with the added interfaces
     *
     * @exception CodeGenerationException   if the copied element already exists on the destination class OR if the
     *                                      copied element can't be
     *                                      compiled
     * @exception ResourceNotFoundException if the interfaces could not be located
     */
    static public CtClass copyClassInterfaces(CtClass sourceClass, CtClass destinationClass)
            throws CodeGenerationException, ResourceNotFoundException
    {

        CtClass[] interfaces = null;

        // Get the source class interfaces
        try
        {
            interfaces = sourceClass.getInterfaces();
        }
        catch (NotFoundException notFoundException)
        {
            throw new ResourceNotFoundException(
                    "Could not get interfaces from template class " + sourceClass.getName() + " to copy!",
                    notFoundException);
        }

        // Add each discovered interface from the source class
        for (int i = 0; i < interfaces.length; i++)
        {
            CodeGenUtil4Javassist.addInterface(interfaces[i], destinationClass);
        }

        // Return enhanced class
        return destinationClass;
    }

    /**
     * Copies the class method to another class.
     *
     * @param sourceClass      the source class
     * @param destinationClass the destination class
     * @param methodName       the name of the method to copy
     *
     * @return the destination class with the source class body
     *
     * @exception CodeGenerationException if the copied element already exists on the destination class OR if the copied
     *                                    element can't be
     *                                    compiled
     */
    static public CtClass copyClassMethod(CtClass sourceClass, CtClass destinationClass, String methodName)
            throws CodeGenerationException
    {
        // Check if it already exists on destination class
        if (!methodExists(methodName, destinationClass))
        {
            try
            {
                CtMethod methodToCopy = null;
                for (CtMethod ctMethod : sourceClass.getMethods())
                {
                    if (ctMethod.getName().equals(methodName))
                    {
                        methodToCopy = ctMethod;
                    }
                }
                if (methodToCopy == null)
                {
                    throw new CodeGenerationException(
                            "Could not add method " + methodName + " to class " + destinationClass.getName() + "()",
                            null);
                }
                // If not add it...
                CtMethod methodByteCode = new CtMethod(methodToCopy, destinationClass, null);
                destinationClass.addMethod(methodByteCode);
            }
            catch (CannotCompileException cannotCompileException)
            {
                throw new CodeGenerationException(
                        "Could not add method " + methodName + " to class " + destinationClass.getName() + "(" +
                        cannotCompileException.getReason() + ")", cannotCompileException);
            }
            /*
             * catch (NotFoundException notFoundException) { throw new CodeGenerationException("Could not add method " +
             * methodName + " to class " + destinationClass.getName() + "(" + notFoundException.getMessage() + ")",
             * notFoundException); }
             */
        }

        // Return enhanced class
        return destinationClass;
    }

    /**
     * Copies the class method to another class.
     *
     * @param sourceClass      the source class
     * @param destinationClass the destination class
     * @param methodName       the name of the method to copy
     * @param methodSignature
     *
     * @return the destination class with the source class body
     *
     * @exception CodeGenerationException if the copied element already exists on the destination class OR if the copied
     *                                    element can't be
     *                                    compiled
     */
    static public CtClass copyClassMethod(CtClass sourceClass, CtClass destinationClass, String methodName,
            String methodSignature) throws CodeGenerationException
    {
        // Check if it already exists on destination class
        if (!methodExists(methodName, destinationClass))
        {
            try
            {
                // If not add it...
                CtMethod sourceMethod = sourceClass.getMethod(methodName, methodSignature);
                CtMethod methodByteCode = new CtMethod(sourceMethod, destinationClass, null);

                //Copy annotations
                AnnotationsAttribute sourceAnotations = (AnnotationsAttribute) sourceMethod.getMethodInfo()
                        .getAttribute(AnnotationsAttribute.visibleTag);
                if (sourceAnotations != null)
                {
                    methodByteCode.getMethodInfo().addAttribute(sourceAnotations);
                }

                destinationClass.addMethod(methodByteCode);
            }
            catch (CannotCompileException cannotCompileException)
            {
                throw new CodeGenerationException(
                        "Could not add method " + methodName + " to class " + destinationClass.getName() + "(" +
                        cannotCompileException.getReason() + ")", cannotCompileException);
            }
            catch (NotFoundException notFoundException)
            {
                throw new CodeGenerationException(
                        "Could not add method " + methodName + " to class " + destinationClass.getName() + "(" +
                        notFoundException.getMessage() + ")", notFoundException);
            }
        }

        // Return enhanced class
        return destinationClass;
    }

    /**
     * Copies the class methods to another class.
     *
     * @param sourceClass      the source class
     * @param destinationClass the destination class
     *
     * @return the destination class with the source class body
     *
     * @exception CodeGenerationException if the copied element already exists on the destination class OR if the copied
     *                                    element can't be
     *                                    compiled
     */
    static public CtClass copyClassMethods(CtClass sourceClass, CtClass destinationClass) throws CodeGenerationException
    {

        // Get source class methods
        CtMethod[] sourceClassMethods = sourceClass.getDeclaredMethods();

        // For each retrieved method...
        for (int i = 0; i < sourceClassMethods.length; i++)
        {
            copyClassMethod(sourceClass, destinationClass, sourceClassMethods[i].getName(),
                    sourceClassMethods[i].getSignature());
        }

        // Return enhanced class
        return destinationClass;
    }

    /**
     * Copies the class body to another class. It copies interfaces, attributes, and methods
     *
     * @param sourceClass      the source class
     * @param destinationClass the destination class
     *
     * @return the destination class with the source class body
     *
     * @exception CodeGenerationException   if the copied element already exists on the destination class OR if the
     *                                      copied element can't be
     *                                      compiled
     * @exception ResourceNotFoundException if the interfaces could not be located
     */
    static public CtClass copyClassToClass(CtClass sourceClass, CtClass destinationClass)
            throws CodeGenerationException, ResourceNotFoundException
    {

        long initTime = System.currentTimeMillis();

        destinationClass = copyClassInterfaces(sourceClass, destinationClass);
        destinationClass = copyClassAttributes(sourceClass, destinationClass);
        destinationClass = copyClassMethods(sourceClass, destinationClass);

        if (ByteCodeStartupConfiguration.isTrackTimes())
        // LOG Execution times code
        {
            System.out.println(
                    destinationClass.getSimpleName() + " - Copied all from class: " + sourceClass.getSimpleName() +
                    " (" + (System.currentTimeMillis() - initTime) + "ms)");
        }

        return destinationClass;
    }

    /**
     * Creates a new class from a class name.
     *
     * @param className the name of the class to create
     *
     * @return a new class object
     */
    static public CtClass createNewClass(String className)
    {

        loadCurrentThreadClassPathContext();

        return ClassPool.getDefault().makeClass(className);
    }

    /**
     * Creates a new class holder and the associated class.
     *
     * @param className the name of the class to create
     *
     * @return a ClassHolder object with the supplied name
     */
    static public ClassHolder createNewClassHolder(String className)
    {
        loadCurrentThreadClassPathContext();

        return new ClassHolder(createNewClass(className));
    }

    /**
     * Returns the value of an annotation member, given the annotation and the member name.
     *
     * @param annotation the annotation to search
     * @param memberName the name of the member
     *
     * @return the given annotation member value
     *
     * @exception ResourceNotFoundException if the annotation class file can't be found
     */
    static public AnnotationMemberValueHolder getAnnotationMemberValue(Annotation annotation, String memberName)
            throws ResourceNotFoundException
    {

        return getAnnotationMemberValues(annotation).get(memberName);
    }

    /**
     * Returns a map with all the member names and values of a given annotation. Albeit being a utility method it can't
     * be located on CodeGenUtil4Reflection class due to import name clash of java.lang.annotation.Annotation and
     * javassist.bytecode.annotation.Annotation.
     *
     * @param annotation the annotation whose members will be read
     *
     * @return a map with all the member names and values of the annotation,
     *
     * @exception ResourceNotFoundException if the annotation class file can't be found
     */
    static public Map<String, AnnotationMemberValueHolder> getAnnotationMemberValues(Annotation annotation)
            throws ResourceNotFoundException
    {
        // Init result
        Map<String, AnnotationMemberValueHolder> result = new HashMap<String, AnnotationMemberValueHolder>();

        // ...get declared annotation members names
        Set<?> memberNames = annotation.getMemberNames();
        // ...if the annotation has members...
        if (memberNames != null)
        {
            // ...for each member...
            for (Object memberName : memberNames)
            {
                // ...fetch it's value
                MemberValue memberValue = annotation.getMemberValue((String) memberName);
                // ...and add pair to result
                result.put(memberName.toString(), new AnnotationMemberValueHolder(memberValue));
            }
        }

        /*
         * By now, result contains the member names and values declared by the developer. The default member values,
         * although, are not part of the annotation read in runtime. As such, the following code browses through the
         * annotation class file, extracts the default member values and assigns them to result if it's not there
         * already.
         */

        // Init class holder
        Class<?> annotationObject = null;

        try
        {
            // Load annotation class file
            annotationObject = Class.forName(annotation.getTypeName().toString());
        }
        catch (ClassNotFoundException classNotFoundException)
        {
            throw new ResourceNotFoundException("Could not load class file for annotation: " + annotation.getTypeName(),
                    classNotFoundException);
        }

        // Get the methods for the annotation
        Method[] methods = annotationObject.getMethods();

        // Holds the default member values
        Map<String, Object> defaultMemberValues = new HashMap<String, Object>();

        // For each method...
        for (Method method : methods)
        {

            /*
             * ...get the default value... (see java.lang.reflect.Method#getDefaultValue())
             */
            Object defaultValue = method.getDefaultValue();

            // ...if the method has a default value...
            if (defaultValue != null)
            // ...add it to default member values map...
            {
                defaultMemberValues.put(method.getName(), defaultValue);
            }
        }

        // Browse through all the default members...
        for (String defaultMethod : defaultMemberValues.keySet())
        {
            // ...add them if they're not on the result yet
            if (!result.containsKey(defaultMethod))
            {
                result.put(defaultMethod,
                        new AnnotationMemberValueHolder(convertToMemberValue(defaultMemberValues.get(defaultMethod))));
            }
        }

        return result;
    }

    /**
     * Returns a class object for a given class.
     *
     * @param className the class's FQN
     *
     * @return the class object
     *
     * @exception ResourceNotFoundException if class can't be loaded.
     */
    static public CtClass getClass(String className) throws ResourceNotFoundException
    {

        // Disable pruning if it's active
        if (ClassPool.doPruning)
        {
            ClassPool.doPruning = false;
        }

        // Init class holder
        CtClass clazz = null;

        try
        {
            /*
             * IMPLEMENTATION NOTE: In Maven Surefire the System class path does not contain all classes. This loads the
             * classPath entries that it adds to the provided default classLoader into Javassist's Loader.
             */

            // Load the current thread class path context
            loadCurrentThreadClassPathContext();

            // Get class
            clazz = ClassPool.getDefault().get(className);
            // Disable pruning
            clazz.stopPruning(true);
        }
        catch (NotFoundException notFoundException)
        {
            throw new ResourceNotFoundException("Could not load class " + className + "!", notFoundException);
        }

        // Defrost class if needed
        if (clazz.isFrozen())
        {
            clazz.defrost();
        }

        return clazz;
    }

    /**
     * Converts an enum value to String.
     *
     * @param memberValue the member value to convert
     *
     * @return the String conversion of the member value
     */
    static public String getEnumValue(MemberValue memberValue)
    {
        return ((EnumMemberValue) memberValue).getValue();
    }

    /**
     * Returns, as a List, all annotations present on a given field.
     *
     * @param attribute the attribute to search
     *
     * @return a list with the field's annotation
     */
    static public Map<String, AnnotationHolder> getFieldAnnotations(AttributeHolder attribute)
    {

        Map<String, AnnotationHolder> annotations = new HashMap<String, AnnotationHolder>();

        Annotation[] annotationsArray = getFieldAnnotationsAsArray(attribute.getManagedAttribute());

        if (annotationsArray != null)
        {
            for (Annotation annotation : annotationsArray)
            {
                annotations.put(annotation.getTypeName(), new AnnotationHolder(attribute, annotation));
            }
        }

        return annotations;
    }

    /**
     * Returns, as an array, all annotations present on a given field.
     *
     * @param field the field to search
     *
     * @return a list with the field's annotation
     */
    static public Annotation[] getFieldAnnotationsAsArray(CtField field)
    {

        Annotation[] annotations = null;

        if (isFieldAnnotated(field))
        {
            // Get field's info...
            FieldInfo fieldInfo = field.getFieldInfo();
            // ...get the runtime-visible annotation accessor object for the
            // attribute...
            AnnotationsAttribute annotationsAttribute =
                    (AnnotationsAttribute) fieldInfo.getAttribute(AnnotationsAttribute.visibleTag);

            // ...get the annotations
            annotations = annotationsAttribute.getAnnotations();
        }

        return annotations;
    }

    /**
     * Returns, as a Map, all annotations present on a given method.
     *
     * @param method the method to search
     *
     * @return an array with a given method's annotations
     */
    static public Map<String, AnnotationHolder> getMethodAnnotations(MethodHolder method)
    {

        Map<String, AnnotationHolder> annotations = new HashMap<String, AnnotationHolder>();

        Annotation[] annotationsArray = getMethodAnnotationsAsArray(method.getManagedMethod());

        if (annotationsArray != null)
        {
            for (Annotation annotation : annotationsArray)
            {
                annotations.put(annotation.getTypeName(), new AnnotationHolder(method, annotation));
            }
        }

        return annotations;
    }

    /**
     * Returns, as an array, all annotations present on a given method.
     *
     * @param method the method to search
     *
     * @return an array with a given method's annotations
     */
    static public Annotation[] getMethodAnnotationsAsArray(CtMethod method)
    {

        Annotation[] annotations = null;

        if (isMethodAnnotated(method))
        {
            // Read method info
            MethodInfo methodInfo = method.getMethodInfo();
            // ...get the method's runtime-visible annotation accessor object
            AnnotationsAttribute annotationsAttribute =
                    (AnnotationsAttribute) methodInfo.getAttribute(AnnotationsAttribute.visibleTag);

            // ...get the annotations
            annotations = annotationsAttribute.getAnnotations();
        }

        return annotations;
    }

    /**
     * Gets the name of a class's superclass, if it exists.
     *
     * @param className the name of the class to inspect
     *
     * @return the name of the superclass, null if the class to inspect doesn't inherit from any other class
     */
    static public String getSuperClassName(String className)
    {
        loadCurrentThreadClassPathContext();

        try
        {
            return ClassPool.getDefault().get(className).getSuperclass().getName();
        }
        catch (NotFoundException notFoundException)
        {
            return null;
        }
    }

    /**
     * Searches a given class for a given type annotation.
     *
     * @param annotationName the name of the annotation to look for
     * @param className      the name of the class to search
     *
     * @return the annotation on the class, null if it wasn't present
     *
     * @exception ResourceNotFoundException if class can't be loaded.
     */
    static public Annotation getTypeAnnotation(String annotationName, String className) throws ResourceNotFoundException
    {

        // Init result
        Annotation annotation = null;

        // Fetch class annotations
        Annotation[] annotations = getTypeAnnotationsAsArray(className);

        if (annotations != null)
        {
            // Search the target annotation in class annotations
            for (int i = 0; i < annotations.length; i++)
            {
                // If the target annotation is found...
                if (annotations[i].getTypeName().equals(annotationName))
                {
                    // Set it to result, and exit
                    annotation = annotations[i];
                    break;
                }
            }
        }

        return annotation;
    }

    /**
     * Returns, as a Map, all type annotations present on a given class.
     *
     * @param clazz the class
     *
     * @return a Map with the FQNs of all type annotations found on the class and the corresponding annotation object.
     *
     * @exception ResourceNotFoundException if the class to search can't be loaded
     */
    static public Map<String, AnnotationHolder> getTypeAnnotations(ClassHolder clazz) throws ResourceNotFoundException
    {

        Map<String, AnnotationHolder> annotations = new HashMap<String, AnnotationHolder>();

        Annotation[] annotationsArray = getTypeAnnotationsAsArray(clazz.getManagedClass());

        if (annotationsArray != null)
        {
            for (Annotation annotation : annotationsArray)
            {
                annotations.put(annotation.getTypeName(), new AnnotationHolder(clazz, annotation));
            }
        }

        return annotations;
    }

    /**
     * Returns, as a List, all type annotations present on a class, given it's name.
     *
     * @param className the FQN of the class to search for annotations
     *
     * @return a list with the FQNs of all type annotations found on the class.
     *
     * @exception ResourceNotFoundException if the class to search can't be loaded
     */
    static public List<Annotation> getTypeAnnotations(String className) throws ResourceNotFoundException
    {
        // TODO: implement in the getMethodAnnotations() fashion
        return Arrays.asList(getTypeAnnotationsAsArray(className));
    }

    /**
     * Returns, as an array, all type annotations present on a given class.
     *
     * @param clazz the class
     *
     * @return an array with all type annotations found on the class
     */
    static public Annotation[] getTypeAnnotationsAsArray(CtClass clazz)
    {

        if (clazz.isFrozen())
        {
            clazz.defrost();
        }

        // Get class file from class object
        ClassFile classFile = clazz.getClassFile();

        // Get the runtime-visible annotation accessor object for the attribute
        AnnotationsAttribute annotationsAttribute =
                (AnnotationsAttribute) classFile.getAttribute(AnnotationsAttribute.visibleTag);

        // Init result
        Annotation[] annotations = null;

        // Read type annotations used on this class if there are some
        if (annotationsAttribute != null)
        {
            annotations = annotationsAttribute.getAnnotations();
        }

        return annotations;
    }

    /**
     * Returns, as an array, all type annotations present on a class, given it's name.
     *
     * @param className the FQN of the class to search for annotations
     *
     * @return an array with all type annotations found on the class
     *
     * @exception ResourceNotFoundException if the class to search can't be loaded
     */
    static public Annotation[] getTypeAnnotationsAsArray(String className) throws ResourceNotFoundException
    {

        // Get class from name
        CtClass clazz = getClass(className);

        // Get type annotations for class
        Annotation[] annotations = getTypeAnnotationsAsArray(clazz);

        return annotations;
    }

    /**
     * Returns a list (Collection) with the FQNs of all the annotations found on a given class.
     *
     * @param className the FQN of the class to search
     *
     * @return a list with all the type annotation names found on the class
     *
     * @exception ResourceNotFoundException if class can't be loaded
     */
    static public List<String> getTypeAnnotationsNames(String className) throws ResourceNotFoundException
    {

        // Get class annotations
        Annotation[] annotations = getTypeAnnotationsAsArray(className);

        // Init result holder
        List<String> annotationsNames = null;

        if (annotations != null)
        {
            annotationsNames = new ArrayList<String>();

            // Extract annotation names to result holder
            for (int i = 0; i < annotations.length; i++)
            {
                annotationsNames.add(annotations[i].getTypeName());
            }
        }

        return annotationsNames;
    }

    /**
     * Returns, as a Map, all attributes present on a given class.
     *
     * @param clazz the class to search
     *
     * @return an array with a given class's attributes
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    static public Map<String, AttributeHolder> getTypeFields(ClassHolder clazz) throws ResourceNotFoundException
    {

        Map<String, AttributeHolder> attributes = new HashMap<String, AttributeHolder>();

        for (CtField field : getTypeFieldsAsArray(clazz.getManagedClass()))
        {
            attributes.put(field.getName(), new AttributeHolder(clazz, field));
        }

        return attributes;
    }

    /**
     * Returns, as an array, all attributes present on a given class.
     *
     * @param clazz the class to search
     *
     * @return an array with a given class's attributes
     */
    static public CtField[] getTypeFieldsAsArray(CtClass clazz)
    {
        // All class fields
        List<CtField> list = new ArrayList<CtField>();
        list.addAll(Arrays.asList(clazz.getDeclaredFields()));

        // Add the public parent classes fields (must filter out duplicate ones)
        for (CtField field : clazz.getFields())
        {

            boolean exists = false;

            for (CtField current : list)
            {
                if (current.getName().equals(field.getName()))
                {
                    exists = true;
                }
            }

            if (!exists)
            {
                list.add(field);
            }
        }

        CtField[] array = new CtField[list.size()];

        return list.toArray(array);
    }

    /**
     * Returns, as a Map, all methods present on a given class.
     *
     * @param clazz     the class to search
     * @param inherited if inherited methods should be returned
     *
     * @return an array with a given class's methods
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    static public Map<String, MethodHolder> getTypeMethods(ClassHolder clazz, boolean inherited)
            throws ResourceNotFoundException
    {

        Map<String, MethodHolder> methods = new HashMap<String, MethodHolder>();

        for (CtMethod method : getTypeMethodsAsArray(clazz.getManagedClass(), inherited))
        {
            methods.put(method.getName(), new MethodHolder(clazz, method));
        }

        return methods;
    }

    /**
     * Returns, as an array, all methods present on a given class.
     *
     * @param clazz     the class to search
     * @param inherited if inherited methods should be returned
     *
     * @return an array with a given class's methods
     */
    static public CtMethod[] getTypeMethodsAsArray(CtClass clazz, boolean inherited)
    {
        // All class fields
        List<CtMethod> list = new ArrayList<CtMethod>();
        list.addAll(Arrays.asList(clazz.getDeclaredMethods()));

        if (inherited)
        {
            // Add the public parent classes fields (must filter out duplicate
            // ones)
            for (CtMethod method : clazz.getMethods())
            {

                boolean exists = false;

                for (CtMethod current : list)
                {
                    if (current.equals(method))
                    {
                        exists = true;
                        break;
                    }
                }

                if (!exists)
                {
                    list.add(method);
                }
            }
        }
        CtMethod[] array = new CtMethod[list.size()];

        return list.toArray(array);
    }

    /**
     * Returns T if a given class implements a given interface, F otherwise.
     *
     * @param clazz         the class that implements the interface
     * @param interfaceName the name of the implemented interface
     *
     * @return T if the class implements the interface, F otherwise
     *
     * @exception ResourceNotFoundException if the class or interface can't be found on the class loader
     */
    static public boolean hasInterface(ClassHolder clazz, String interfaceName) throws ResourceNotFoundException
    {
        loadCurrentThreadClassPathContext();

        return clazz.hasInterface(interfaceName);
    }

    /**
     * Returns T if a given class implements a given interface, F otherwise.
     *
     * @param className     the name of the class that implements the interface
     * @param interfaceName the name of the implemented interface
     *
     * @return T if the class implements the interface, F otherwise
     *
     * @exception NotFoundException if the class or interface can't be found on the class loader
     */
    static public boolean hasInterface(String className, String interfaceName) throws NotFoundException
    {
        loadCurrentThreadClassPathContext();

        return Arrays.asList(ClassPool.getDefault().get(className).getClassFile2().getInterfaces())
                .contains(interfaceName);
    }

    /**
     * Implements a given interface on a given class. Does not implement any interface methods.
     *
     * @param clazz         the class to alter
     * @param interfaceName the name of the interface to be implemented.
     *
     * @exception ResourceNotFoundException if the class can't be found
     */
    static public void implementInterface(ClassHolder clazz, String interfaceName) throws ResourceNotFoundException
    {
        loadCurrentThreadClassPathContext();

        try
        {
            clazz.getManagedClass().addInterface(ClassPool.getDefault().get(interfaceName));
        }
        catch (NotFoundException notFoundException)
        {
            throw new ResourceNotFoundException("Could not find interface " + interfaceName + "!", notFoundException);
        }
    }

    /**
     * Implements a given interface on a given class. Does not implement any interface methods.
     *
     * @param className     the name of the class to alter
     * @param interfaceName the name of the interface to be implemented.
     *
     * @exception NotFoundException if the class or the interface can't be found
     */
    static public void implementInterface(String className, String interfaceName) throws NotFoundException
    {
        loadCurrentThreadClassPathContext();

        ClassPool.getDefault().get(className).addInterface(ClassPool.getDefault().get(interfaceName));
    }

    /**
     * Check if a class is annotated.
     *
     * @param clazz the field to check
     *
     * @return T if the field is annotated, F otherwise
     */
    static public boolean isClassAnnotated(CtClass clazz)
    {
        return (clazz.getAvailableAnnotations().length > 0 ? true : false);
    }

    /**
     * Searches a class for a given annotation.
     *
     * @param className      the name of the class to search
     * @param annotationName the annotation to search for
     *
     * @return T if the class is annotated with the given annotation, F otherwise
     *
     * @exception ResourceNotFoundException if the class can't be found
     */
    static public boolean isClassAnnotatedWith(String className, String annotationName) throws ResourceNotFoundException
    {

        // Init result
        boolean result = false;

        // Get annotations for the class
        List<String> annotationsNames = getTypeAnnotationsNames(className);

        // If class is annotated AND contains annotation, set result to T
        if (annotationsNames != null && annotationsNames.contains(annotationName))
        {
            result = true;
        }

        return result;
    }

    /**
     * Check if a field is annotated.
     *
     * @param field the field to check
     *
     * @return T if the field is annotated, F otherwise
     */
    static public boolean isFieldAnnotated(CtField field)
    {
        return (field.getAvailableAnnotations().length > 0 ? true : false);
    }

    /**
     * Check if a method is annotated.
     *
     * @param method the method to check
     *
     * @return T if the method is annotated, F otherwise
     */
    static public boolean isMethodAnnotated(CtMethod method)
    {
        return (method.getAvailableAnnotations().length > 0 ? true : false);
    }

    /**
     * Appends the current execution thread class path to the default class pool.
     */
    static public void loadCurrentThreadClassPathContext()
    {
        ClassPool.getDefault().appendClassPath(new LoaderClassPath(Thread.currentThread().getContextClassLoader()));
    }

    /**
     * Searches a given class for a given method name.
     *
     * @param methodName the method's name
     * @param clazz      the class to search
     *
     * @return T if the class already contains the method, F otherwise
     */
    static public boolean methodExists(String methodName, CtClass clazz)
    {
        boolean result = false;

        // Get class's methods...
        CtMethod[] methods = clazz.getDeclaredMethods();

        /*
         * Implementation Note: doesn't use Javassist's getDeclaredMethod(String) because it throws an exception if
         * method's not found.
         */

        // Check each method's name for a match with the given name
        for (int i = 0; i < methods.length; i++)
        {
            if (methods[i].getName().equals(methodName))
            {
                result = true;
                break;
            }
        }

        return result;
    }

    /**
     * Replaces a given method's body.
     *
     * @param methodSource the source code to inject on the method
     * @param method       the method to enhance
     *
     * @return the method with the source added
     *
     * @exception CodeGenerationException if method's body cannot be replaced
     */
    static public CtMethod replaceMethodBody(String methodSource, CtMethod method) throws CodeGenerationException
    {

        long initTime = System.currentTimeMillis();

        if (ByteCodeStartupConfiguration.isShowSource())
        // LOG Execution times code
        {
            System.out.println("Compiling source for method \"" + method.getName() + "\": \n     | " +
                               methodSource.replace("\n", "\n    | "));
        }

        try
        {
            if (method.getDeclaringClass().isFrozen())
            {
                method.getDeclaringClass().defrost();
            }

            method.setBody(methodSource);
        }
        catch (CannotCompileException cannotCompileException)
        {
            throw new CodeGenerationException(
                    "Could not compile source code for method " + method.getName() + " ( Cause: " +
                    cannotCompileException.getReason() + " Source code: " + methodSource + ")", cannotCompileException);
        }

        if (ByteCodeStartupConfiguration.isTrackTimes())
        // LOG Execution times code
        {
            System.out.println(
                    method.getDeclaringClass().getSimpleName() + " - Updated Method: " + method.getName() + " (" +
                    (System.currentTimeMillis() - initTime) + "ms)");
        }

        return method;
    }

    /**
     * Sets the super class of a given class.
     *
     * @param clazz          the child class
     * @param superClassName the name of the super class
     *
     * @exception ResourceNotFoundException if any of the classes can't be found
     * @exception CannotCompileException    if the child class already inherits from another class
     */
    static public void setSuperClass(ClassHolder clazz, String superClassName)
            throws CannotCompileException, ResourceNotFoundException
    {
        loadCurrentThreadClassPathContext();
        try
        {
            clazz.setSuperClass(ClassPool.getDefault().get(superClassName));
        }
        catch (NotFoundException notFoundException)
        {
            throw new ResourceNotFoundException("Could not find " + superClassName, notFoundException);
        }
    }

    /**
     * Sets the super class of a given class.
     *
     * @param className      the name of the child class
     * @param superClassName the name of the super class
     *
     * @exception ResourceNotFoundException if any of the classes can't be found
     * @exception CannotCompileException    if the child class already inherits from another class
     */
    static public void setSuperClass(String className, String superClassName)
            throws CannotCompileException, ResourceNotFoundException
    {
        loadCurrentThreadClassPathContext();

        try
        {
            ClassPool.getDefault().get(className).setSuperclass(ClassPool.getDefault().get(superClassName));
        }
        catch (NotFoundException notFoundException)
        {
            throw new ResourceNotFoundException("Could not found class!", notFoundException);
        }
    }

    /**
     * Converts an enhanced class back to .class file format.
     *
     * @param clazz the class to write
     *
     * @return the generated class
     *
     * @exception CodeGenerationException
     */
    static public Class<?> writeClass(CtClass clazz) throws CodeGenerationException
    {
        try
        {
            if (clazz.isFrozen())
            {
                clazz.defrost();
            }

            // Implementation note:
            // -------------------------
            // Creates the new Class after compile by JavaAssist in the current API ClassLoader context.
            // We use the CodeGenUtil4Javassist ClassLoader since we have observed that this is normally the WebApp
            // ClassLoader, witch in the chain of ClassLoaders is the lowest level ClassLoader with all LIBs in the
            // ClassPath.
            // If the class extends another class with package private attributes, this ClassLoader MUST be in the same
            // chain as the original class.
            //
            // We cannot use the default behavior since it uses the ClassLoader of the JavaAssist LIB wich may be in
            // the App Server context and thus is not the same of the eventual super classes.

            return clazz.toClass(CodeGenUtil4Javassist.class.getClassLoader(),
                    CodeGenUtil4Javassist.class.getProtectionDomain());
        }
        catch (CannotCompileException cannotCompileException)
        {
            throw new CodeGenerationException(cannotCompileException);
        }
    }
}