/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.bytecode.holders;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.NotFoundException;
import pt.digitalis.utils.CodeGenUtil4Javassist;
import pt.digitalis.utils.CodeGenUtils;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.util.Arrays;
import java.util.Map;

/**
 * An helper class that helps manage the temporary class in the code generation process. Provides managing methods and
 * caches the details read from the class.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Sep 21, 2007
 */
public class ClassHolder
{

    /** The fully qualified name of the managed class. */
    private String classFQName;

    /**
     * Constructs a holder from a class object.
     *
     * @param clazz the class to manage
     */
    public ClassHolder(CtClass clazz)
    {
        // Add the class to repository
        if (!HolderRepository.classExists(clazz.getName()))
        {
            HolderRepository.addClass(clazz);
        }

        this.classFQName = clazz.getName();
    }

    /**
     * Constructs a holder from a class name.
     *
     * @param classFQName the fully qualified name of the class to manage
     *
     * @exception ResourceNotFoundException
     */
    public ClassHolder(String classFQName) throws ResourceNotFoundException
    {
        // Loads the CtClass prior to construction
        this(CodeGenUtil4Javassist.getClass(classFQName));
    }

    /**
     * Adds an interface to the class
     *
     * @param interfaceClass the interface class to add
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    public void addInterface(ClassHolder interfaceClass) throws ResourceNotFoundException
    {
        this.update(CodeGenUtil4Javassist.addInterface(interfaceClass.getManagedClass(), this.getManagedClass()));
    }

    /**
     * Adds a new method to the class. If it already exists nothing is done and the new method is kept.
     *
     * @param sourceCode the source code to compile into to method
     *
     * @exception ResourceNotFoundException if the class can't be read
     * @exception CodeGenerationException   if method's body can't be added
     */
    public void addMethod(String sourceCode) throws CodeGenerationException, ResourceNotFoundException
    {
        this.update(CodeGenUtil4Javassist.addMethodSourceToClass(sourceCode, this.getManagedClass()));
    }

    /**
     * Adds a new method to the class. If it already exists, replaces the old one.
     *
     * @param sourceCode the source code to compile into to method
     *
     * @exception ResourceNotFoundException if the class can't be read
     * @exception CodeGenerationException   if method's body can't be added
     */
    public void addOrReplaceMethod(String sourceCode) throws CodeGenerationException, ResourceNotFoundException
    {
        this.update(CodeGenUtil4Javassist.addMethodSourceToClass(sourceCode, this.getManagedClass(), true));
    }

    /**
     * Performs the object clean-up.
     */
    public void cleanUp()
    {
        try
        {
            this.getManagedClass().detach();
        }
        catch (ResourceNotFoundException e)
        {
            // The exception will be captured so that the process can continue.
            // No clean up is needed if the class was not found
        }
    }

    /**
     * Validates if a given annotation name is used on the class.
     *
     * @param annotationName the name of the annotation to search
     *
     * @return T if the annotation is used, F otherwise
     *
     * @exception ResourceNotFoundException if annotations can't be read
     */
    public boolean containsAnnotation(String annotationName) throws ResourceNotFoundException
    {

        return this.getAnnotations().containsKey(annotationName);
    }

    /**
     * Validates if a given method name is present on the class.
     *
     * @param methodName the name of the method to search
     *
     * @return T if the method is present, F otherwise
     *
     * @exception ResourceNotFoundException if the methods can't be read
     */
    public boolean containsMethod(String methodName) throws ResourceNotFoundException
    {
        return this.getMethods().containsKey(methodName);
    }

    /**
     * Copies a class into the current one. Copies all interfaces, fields and methods
     *
     * @param sourceTemplate the source template class to copy from
     *
     * @exception ResourceNotFoundException if the class can't be read
     * @exception CodeGenerationException   if any element of the class could not be copied
     */
    public void copyAllFromClass(ClassHolder sourceTemplate) throws CodeGenerationException, ResourceNotFoundException
    {

        // Copy template class to the class to enhance
        CtClass enhancedClass =
                CodeGenUtil4Javassist.copyClassToClass(sourceTemplate.getManagedClass(), this.getManagedClass());

        // Update the enhanced class
        this.update(enhancedClass);
    }

    /**
     * Copies a method from a class into the current one. Does not support methods with same name but different
     * signatures.
     *
     * @param sourceTemplate the source template class to copy from
     * @param methodName     the method to copy
     *
     * @exception ResourceNotFoundException if the class can't be read
     * @exception CodeGenerationException   if any element of the class could not be copied
     */
    public void copyMethodFromClass(ClassHolder sourceTemplate, String methodName)
            throws CodeGenerationException, ResourceNotFoundException
    {

        // Copy method to the class to enhance
        CtClass enhancedClass = CodeGenUtil4Javassist
                .copyClassMethod(sourceTemplate.getManagedClass(), this.getManagedClass(), methodName);

        // Update the enhanced class
        this.update(enhancedClass);
    }

    /**
     * Removes from the repository all the previously cached entries referring this class.
     *
     * @exception ResourceNotFoundException if the class cant' be read
     */
    public void deleteClassFromRepository() throws ResourceNotFoundException
    {

        HolderRepository.deleteClassAndReferences(this.getUniqueID());

        this.classFQName = null;
    }

    /**
     * Creates a generated ID which is the lower-case version of the class name.
     *
     * @return the generated ID
     *
     * @exception ResourceNotFoundException if the class can't be read for id generation
     */
    public String generateID() throws ResourceNotFoundException
    {
        return CodeGenUtils.generateID(this.getName());
    }

    /**
     * Returns the managed class's methods, including all inherited methods.
     *
     * @return a map with the methods names as K and the methods objects as V
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    public Map<String, MethodHolder> getAllMethods() throws ResourceNotFoundException
    {
        // Get the methods from repository
        Map<String, MethodHolder> classMethods = HolderRepository.getMethods(this.getUniqueID());

        // Lazy loading for methods...
        if (classMethods == null)
        {
            // Get the methods from javassist
            classMethods = CodeGenUtil4Javassist.getTypeMethods(this, true);
            // Add methods to the repository
            HolderRepository.addMethods(this.getUniqueID(), classMethods);
        }

        return classMethods;
    }

    /**
     * Returns the managed class's annotations.
     *
     * @return a map with the annotations names as K and the annotation object as V
     *
     * @exception ResourceNotFoundException if annotations can't be read
     */
    public Map<String, AnnotationHolder> getAnnotations() throws ResourceNotFoundException
    {

        // Get the type annotations from repository
        Map<String, AnnotationHolder> classAnnotations = HolderRepository.getAnnotations(this.getUniqueID());

        // Lazy loading for annotations...
        if (classAnnotations == null)
        {
            // Get annotations from javassist
            classAnnotations = CodeGenUtil4Javassist.getTypeAnnotations(this);
            // Add annotations to repository
            HolderRepository.addAnnotations(this.getUniqueID(), classAnnotations);
        }

        return classAnnotations;
    }

    /**
     * Returns the managed class's attributes.
     *
     * @return a map with the attributes names as K and the attributes object as V
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    public Map<String, AttributeHolder> getAttributes() throws ResourceNotFoundException
    {

        // Get the attributes from repository
        Map<String, AttributeHolder> classAttributes = HolderRepository.getAttributes(this.getUniqueID());

        // Lazy loading for attributes...
        if (classAttributes == null)
        {
            // Get attributes from javassist
            classAttributes = CodeGenUtil4Javassist.getTypeFields(this);
            // Add attributes to repository
            HolderRepository.addAttributes(this.getUniqueID(), classAttributes);
        }

        return classAttributes;
    }

    /**
     * @return a new instance of the enriched class
     *
     * @exception ClassNotFoundException
     * @exception IllegalAccessException
     * @exception InstantiationException
     * @exception ResourceNotFoundException
     */
    public Object getClassInstance()
            throws ClassNotFoundException, InstantiationException, IllegalAccessException, ResourceNotFoundException
    {
        return this.getManagedClass().getClassPool().getClassLoader().loadClass(this.classFQName).newInstance();
    }

    /**
     * Returns the class's Fully Qualified Name.
     *
     * @return the class's FQN
     */
    public String getFQName()
    {
        return this.classFQName;
    }

    /**
     * Returns the class object
     *
     * @return the class object
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    public CtClass getManagedClass() throws ResourceNotFoundException
    {
        return HolderRepository.getClass(this.classFQName);
    }

    /**
     * Returns the managed class's methods.
     *
     * @return a map with the methods names as K and the methods objects as V
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    public Map<String, MethodHolder> getMethods() throws ResourceNotFoundException
    {

        // Get the methods from repository
        Map<String, MethodHolder> classMethods = HolderRepository.getMethods(this.getUniqueID());

        // Lazy loading for methods...
        if (classMethods == null)
        {
            // Get the methods from javassist
            classMethods = CodeGenUtil4Javassist.getTypeMethods(this, false);
            // Add methods to the repository
            HolderRepository.addMethods(this.getUniqueID(), classMethods);
        }

        return classMethods;
    }

    /**
     * Returns the class's name.
     *
     * @return the class's name
     *
     * @exception ResourceNotFoundException if class can't be found
     */
    public String getName() throws ResourceNotFoundException
    {
        return HolderRepository.getClass(this.classFQName).getSimpleName();
    }

    /**
     * Returns the class's package name.
     *
     * @return the package name
     *
     * @exception ResourceNotFoundException if class can't be found
     */
    public String getPackageName() throws ResourceNotFoundException
    {
        return HolderRepository.getClass(this.classFQName).getPackageName();
    }

    /**
     * Returns the name of the superclass.
     *
     * @return the name of the superclass if it exists
     *
     * @exception ResourceNotFoundException if the class or the superclass can't be found
     */
    public String getSuperClass() throws ResourceNotFoundException
    {
        try
        {
            return this.getManagedClass().getSuperclass().getName();
        }
        catch (NotFoundException e)
        {
            return null;
        }
    }

    /**
     * Sets the superclass of a given class.
     *
     * @param superClass the superclass
     *
     * @exception ResourceNotFoundException if either class can't be found
     * @exception CannotCompileException    if the class already extends from another class
     */
    public void setSuperClass(CtClass superClass) throws CannotCompileException, ResourceNotFoundException
    {
        this.getManagedClass().setSuperclass(superClass);
    }

    /**
     * Sets the superclass of a given class.
     *
     * @param superClassName the superclass name
     *
     * @exception ResourceNotFoundException if either class can't be found
     * @exception CannotCompileException    if the class already extends from another class
     */
    public void setSuperClass(String superClassName) throws CannotCompileException, ResourceNotFoundException
    {
        CodeGenUtil4Javassist.setSuperClass(this.getFQName(), superClassName);
    }

    /**
     * Returns a unique ID for this class.
     *
     * @return the unique ID
     *
     * @exception ResourceNotFoundException if the class can't be read for id generation
     */
    public String getUniqueID() throws ResourceNotFoundException
    {
        return this.getFQName();
    }

    /**
     * Returns T if the class contains a given interface, F otherwise.
     *
     * @param interfaceName the name of the interface to check
     *
     * @return T if the class implements the given interface, F otherwise
     *
     * @exception ResourceNotFoundException if the class can't be found
     */
    public boolean hasInterface(String interfaceName) throws ResourceNotFoundException
    {
        return Arrays.asList(this.getManagedClass().getClassFile2().getInterfaces()).contains(interfaceName);
    }

    /**
     * Updates the managed class object. This method should be called after class enhancing takes place.
     *
     * @param clazz the updated class
     */
    public void update(CtClass clazz)
    {
        /*
         * Implementation Note: Forces the change to take place on the correct class since it reflects the change on the
         * holder if a different class is passed as argument.
         */

        try
        {
            HolderRepository.deleteClassAndReferences(this.getUniqueID());
        }
        catch (ResourceNotFoundException e)
        {
            // Do nothing. If the class was not found there was nothing to delete in any way
        }

        this.classFQName = clazz.getName();

        HolderRepository.updateClass(clazz);
    }

    /**
     * Updates a class method with the new implementation source code.
     *
     * @param methodName the method do override it's source
     * @param sourceCode the source code to compile into to method
     *
     * @exception ResourceNotFoundException if the class can't be read
     * @exception CodeGenerationException   if method's body cannot be replaced
     */
    public void updateMethodSource(String methodName, String sourceCode)
            throws CodeGenerationException, ResourceNotFoundException
    {

        MethodHolder method = this.getMethods().get(methodName);
        method.updateSource(sourceCode);
    }

    /**
     * Writes the current class so the JVM may load the new version when needed
     *
     * @return the generated class
     *
     * @exception CodeGenerationException   if the class can't be written
     * @exception ResourceNotFoundException if the class can't be read
     */
    public Class<?> writeClass() throws CodeGenerationException, ResourceNotFoundException
    {
        return CodeGenUtil4Javassist.writeClass(this.getManagedClass());
    }
}