/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.bytecode.holders;

import javassist.CtMethod;
import pt.digitalis.utils.CodeGenUtil4Javassist;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.util.Map;

/**
 * An helper class that helps manage the temporary class methods in the code generation process. Provides managing
 * methods and caches the details read from the method.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Sep 23, 2007
 */
public class MethodHolder
{

    /** The method object. */
    private CtMethod method;

    /** The method's parent class name. */
    private String parentClassName;

    /**
     * Constructs an holder from a parent class and a method object.
     *
     * @param clazz  the method's parent class
     * @param method the method to manage
     *
     * @exception ResourceNotFoundException if the class can't be read
     */
    public MethodHolder(ClassHolder clazz, CtMethod method) throws ResourceNotFoundException
    {
        this.parentClassName = clazz.getUniqueID();
        this.method = method;
    }

    /**
     * Adds source after the current source code.
     *
     * @param sourceCode the source code to compile into to method
     *
     * @exception ResourceNotFoundException if the class can't be read
     * @exception CodeGenerationException   if method's body cannot be replaced
     */
    public void addAfterSource(String sourceCode) throws CodeGenerationException, ResourceNotFoundException
    {

        method = CodeGenUtil4Javassist.addCodeAfterMethodBody(sourceCode, getManagedMethod());
    }

    /**
     * Adds source before the current source code.
     *
     * @param sourceCode the source code to compile into to method
     *
     * @exception ResourceNotFoundException if the class can't be read
     * @exception CodeGenerationException   if method's body cannot be replaced
     */
    public void addBeforeSource(String sourceCode) throws CodeGenerationException, ResourceNotFoundException
    {

        method = CodeGenUtil4Javassist.addCodeBeforeMethodBody(sourceCode, getManagedMethod());
    }

    /**
     * Validates if a given annotation is used on the method.
     *
     * @param annotationName the name of the annotation to search
     *
     * @return T if the annotation is used, F otherwise
     *
     * @exception ResourceNotFoundException if the annotations can't be read
     */
    public boolean containsAnnotation(String annotationName) throws ResourceNotFoundException
    {

        return getAnnotations().containsKey(annotationName);
    }

    /**
     * Returns the method's access flags.
     *
     * @return the method's access flags
     */
    public int getAccessFlags()
    {
        return method.getMethodInfo().getAccessFlags();
    }

    /**
     * Returns the method's annotations.
     *
     * @return a map with the annotations names as K and the annotations objects as V
     *
     * @exception ResourceNotFoundException if the annotations can't be read
     */
    public Map<String, AnnotationHolder> getAnnotations() throws ResourceNotFoundException
    {

        // Get annotations from repository
        Map<String, AnnotationHolder> methodAnnotations = HolderRepository.getAnnotations(getUniqueID());

        // Lazy-loading for the annotations
        if (methodAnnotations == null)
        {
            // Get method's annotations from javassist
            methodAnnotations = CodeGenUtil4Javassist.getMethodAnnotations(this);
            // Add method's annotations to repository
            HolderRepository.addAnnotations(getUniqueID(), methodAnnotations);
        }

        return methodAnnotations;
    }

    /**
     * Returns the managed method object.
     *
     * @return the managed method object
     */
    public CtMethod getManagedMethod()
    {
        return this.method;
    }

    /**
     * Returns the method name.
     *
     * @return the method name
     */
    public String getName()
    {
        return method.getName();
    }

    /**
     * Returns the method's parent class.
     *
     * @return the parent class object
     *
     * @exception ResourceNotFoundException if the parent class can't be read
     */
    public ClassHolder getParentClass() throws ResourceNotFoundException
    {
        return new ClassHolder(HolderRepository.getClass(parentClassName));
    }

    /**
     * Returns the method's parent class name.
     *
     * @return the parent class name
     */
    public String getParentClassName()
    {
        return parentClassName;
    }

    /**
     * Returns the method's signature.
     *
     * @return the method's signature
     */
    public String getSignature()
    {
        return method.getSignature();
    }

    /**
     * Returns a unique ID for this method in it's context (class/method/attribute).
     *
     * @return the unique ID
     */
    public String getUniqueID()
    {
        return parentClassName + "." + getName();
    }

    /**
     * Updates a method with the new implementation source code.
     *
     * @param sourceCode the source code to compile into to method
     *
     * @exception ResourceNotFoundException if the class can't be read
     * @exception CodeGenerationException   if method's body cannot be replaced
     */
    public void updateSource(String sourceCode) throws CodeGenerationException, ResourceNotFoundException
    {

        method = CodeGenUtil4Javassist.replaceMethodBody(sourceCode, getManagedMethod());
    }
}
