/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.config;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

import pt.digitalis.log.Logger;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.DateUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.annotations.ConfigDefault;
import pt.digitalis.utils.config.annotations.ConfigID;
import pt.digitalis.utils.config.annotations.ConfigIgnore;
import pt.digitalis.utils.config.annotations.ConfigKeyID;
import pt.digitalis.utils.config.annotations.ConfigLOVAjaxEvent;
import pt.digitalis.utils.config.annotations.ConfigLOVValues;
import pt.digitalis.utils.config.annotations.ConfigSectionID;
import pt.digitalis.utils.config.annotations.ConfigSectionIDGetter;
import pt.digitalis.utils.inspection.ReflectionUtils;
import pt.digitalis.utils.inspection.ResourceUtils;
import pt.digitalis.utils.inspection.exception.AuxiliaryOperationException;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

/**
 * A facilitator base implementation for IConfigurations. It implements the recurrent needs of these implementations
 * allowing the descending classes to focus only on what is specific to each implementation.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Sep 27, 2007
 */
abstract public class AbstractConfigurationsImpl implements IConfigurations {

    /** The cached configurations */
    static protected Map<String, Class<?>> cachedConfigurations = null;

    /** The cached configurations packages searched so far */
    static protected List<String> cachedConfigurationsPackages = new ArrayList<String>();

    /** The value used to save the value when the default value is active */
    public static final String DEFAULT_VALUE_KEYWORK = "#DefaultValue#";

    /** The common configuration prefix to apply to all configuration paths */
    static private String generalConfigurationPrefix = "";

    /** if T, prefixed configurations will be active */
    static private boolean prefixedConfigurationsActive = false;

    /**
     * The prefixed specific configuration exception. The path specified here will be read from the common configuration
     * node, instead of the specific one
     */
    static private Map<String, List<String>> prefixException = new HashMap<String, List<String>>();

    /**
     * Configures the general prefix
     *
     * @param configurationPrefix
     */
    static public void setGeneralPrefix(String configurationPrefix)
    {
        AbstractConfigurationsImpl.generalConfigurationPrefix = configurationPrefix;
    }

    /**
     * Constructor
     */
    public AbstractConfigurationsImpl()
    {
        // Read general configurations
        Properties props = readConfiguration("Configurations", "General", false);

        // General configurations
        prefixedConfigurationsActive = Boolean.parseBoolean(props.getProperty("PrefixedConfigurationsActive", "false"));
        props.put("PrefixedConfigurationsActive", prefixedConfigurationsActive);

        // Persist, thus creating them if it is the first time
        // TODO: ComQuest: Viegas: 1. Place code to prevent writing if the value has not changed
        writeConfiguration("Configurations", "General", props, false);

        // Custom exclusions for registered prefixes
        props = readConfiguration("Configurations", "PrefixExceptions", false);

        for (Entry<Object, Object> property: props.entrySet())
            if (property.getKey() != null && property.getValue() != null)
            {
                String prefix = property.getKey().toString();
                String[] exceptions = property.getValue().toString().split(",");

                prefixException.put(prefix, Arrays.asList(exceptions));
            }
    }

    /**
     * Constructor
     *
     * @param forcePrefixedConfigurationsActive
     *            if T will always consider that prefixed configurations is active, even if they are configured as
     *            inactive
     */
    public AbstractConfigurationsImpl(boolean forcePrefixedConfigurationsActive)
    {
        this();
        prefixedConfigurationsActive = true;
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#getCacheConfigurationPoints()
     */
    public Map<String, Class<?>> getCacheConfigurationPoints()
    {
        if (cachedConfigurations == null)
            readAllConfigurationsPoints();

        return cachedConfigurations;
    }

    /**
     * Returns the config of an annotated configuration object
     *
     * @param clazz
     *            the class to parse
     * @return the Config value if present
     */
    public String getConfigID(Class<?> clazz)
    {

        // Search for the ConfigID annotation in the object class
        ConfigID configAnnotation = clazz.getAnnotation(ConfigID.class);

        if (configAnnotation == null)
            // If we did not find it return null
            return null;
        else
            // ...else return the annotation value
            return configAnnotation.value();
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#getConfigItemsMap(java.lang.Class)
     */
    public List<ConfigItem> getConfigItemsMap(Class<?> clazz)
    {

        List<ConfigItem> configItems = new ArrayList<ConfigItem>();
        boolean booleanItem;

        // Lets parse all methods
        for (Method method: clazz.getMethods())
        {

            // Exclude inherited methods && if it's a getter method not configIgnore annotated
            if (method.getDeclaringClass() == clazz
                    && ((method.getName().startsWith("get") || method.getName().startsWith("is"))
                            && !method.isAnnotationPresent(ConfigIgnore.class)))
            {
                booleanItem = method.getName().startsWith("is");

                // This is a valid item. Let's see if it has a supported Class getter/setter.
                // Only classes that have a constructor with a String argument may be used

                String key;
                Method getter = method;
                Method setter;

                try
                {
                    if (booleanItem)
                        // Remove the "is" and add "set" prefix to determine the setter method
                        setter = ReflectionUtils.getMethod(clazz, "set" + method.getName().substring(2));
                    else
                        // Remove the "get" and add "set" prefix to determine the setter method
                        setter = ReflectionUtils.getMethod(clazz, "set" + method.getName().substring(3));

                    if (setter == null)
                        throw new ResourceNotFoundException("Could not find setter method: " + "set"
                                + method.getName().substring(3) + " for " + clazz.getSimpleName());

                    if (setter.getParameterTypes().length != 1)
                        Logger.getLogger().error("More than one parameter: method=" + setter.toGenericString());

                    else
                    {
                        Class<?> parameterClass = setter.getParameterTypes()[0];
                        Constructor<?> constructor;

                        if (booleanItem && (parameterClass != boolean.class && parameterClass != Boolean.class))
                            Logger.getLogger()
                                    .error("Prefix \"is\" for getter methods only allows for boolean attributes: method="
                                            + setter.toGenericString());
                        else
                        {
                            if (parameterClass == int.class)
                                parameterClass = Integer.class;
                            else if (parameterClass == long.class)
                                parameterClass = Long.class;
                            else if (parameterClass == double.class)
                                parameterClass = Double.class;
                            else if (parameterClass == boolean.class)
                                parameterClass = Boolean.class;

                            try
                            {
                                constructor = parameterClass.getConstructor(String.class);
                            }
                            catch (SecurityException e)
                            {
                                constructor = null;

                            }
                            catch (NoSuchMethodException e)
                            {
                                constructor = null;
                            }

                            if (constructor == null)
                            {
                                System.err.println(method.getDeclaringClass() + "#" + method.getName());
                                Logger.getLogger()
                                        .error("No constructor in the argument class for String value: method="
                                                + setter.toGenericString());
                            }

                            else
                            {
                                if (method.isAnnotationPresent(ConfigKeyID.class))
                                {

                                    // ConfigKeyID annotated getter method
                                    ConfigKeyID keyAnnotation = method.getAnnotation(ConfigKeyID.class);

                                    key = keyAnnotation.value();

                                }
                                else
                                {
                                    String methodName = method.getName();

                                    // Simple getter method not ConfigIgnore annotated
                                    if (methodName.startsWith("is"))
                                        key = method.getName().substring(2);
                                    else
                                        key = method.getName().substring(3);
                                }

                                // Read default value if annotation present
                                ConfigDefault defaultValueAnnotation = method.getAnnotation(ConfigDefault.class);

                                String defaultValue;

                                if (defaultValueAnnotation == null)
                                    defaultValue = null;
                                else
                                    defaultValue = defaultValueAnnotation.value();

                                ConfigItem configItem = new ConfigItem(key, getter, setter, constructor, defaultValue,
                                        getter.getReturnType());

                                ConfigLOVValues lovValuesAnnotation = method.getAnnotation(ConfigLOVValues.class);
                                if (lovValuesAnnotation != null && StringUtils.isNotBlank(lovValuesAnnotation.value()))
                                    configItem.setLovValues(
                                            CollectionUtils.keyValueStringToMap(lovValuesAnnotation.value()));

                                ConfigLOVAjaxEvent lovAjaxAnnotation = method.getAnnotation(ConfigLOVAjaxEvent.class);
                                if (lovAjaxAnnotation != null && StringUtils.isNotBlank(lovAjaxAnnotation.value()))
                                    configItem.setLovAjaxEvent(lovAjaxAnnotation.value());

                                // Add the item record with the read values
                                configItems.add(configItem);
                            }
                        }
                    }
                }
                catch (ResourceNotFoundException e)
                {
                    Logger.getLogger().error(e.getMessage());
                }

                // }

            }
        }

        return configItems;
    }

    /**
     * Returns the config of an annotated configuration object
     *
     * @param clazz
     *            the class to parse
     * @return the Config value if present
     */
    private String getConfigPath(Class<?> clazz)
    {
        String configID = getConfigID(clazz);
        String configSectionID = getConfigSectionID(clazz);

        if (configID != null)
        {
            if (configSectionID == null)
                return configID;
            else
                return configID + "/" + configSectionID;
        }
        else
            return null;
    }

    /**
     * Returns the configSection of an annotated configuration class
     *
     * @param clazz
     *            the class to parse
     * @return the ConfigSection value if present
     */
    public String getConfigSectionID(Class<?> clazz)
    {

        ConfigSectionID sectionAnnotation = clazz.getAnnotation(ConfigSectionID.class);

        if (sectionAnnotation == null)
        {
            return null;
        }
        else
            return sectionAnnotation.value();
    }

    /**
     * Returns the configSection of an annotated configuration object
     *
     * @param obj
     *            the object to parse
     * @return the ConfigSection value if present
     */
    public String getConfigSectionID(Object obj)
    {

        // Search for the ConfigSectionID annotation in the object class
        ConfigSectionID sectionAnnotation = obj.getClass().getAnnotation(ConfigSectionID.class);

        // If it was not found...
        if (sectionAnnotation == null)
        {

            // ConfigSectionID annotation was not used. Search for the ConfigSectionIDGetter
            ConfigSectionIDGetter sectionMethodAnnotation = null;
            Method sectionMethod = null;

            // Search ConfigSessionIDGetter annotated methods
            for (Method method: obj.getClass().getMethods())
            {
                sectionMethodAnnotation = method.getAnnotation(ConfigSectionIDGetter.class);

                // If found save it and exit the loop...
                if (sectionMethodAnnotation != null)
                {
                    sectionMethod = method;
                    break;
                }
            }

            if (sectionMethod == null)
                // If we did not found the annotation return null...
                return null;
            else
            {
                // ...else we must execute the method to get the value to use as the ID

                try
                {
                    return ReflectionUtils.invokeMethod(sectionMethod, obj).toString();
                }
                catch (Exception e)
                {

                    Logger.getLogger().error(e.getMessage());
                    return null;
                }
            }

        }
        else
            // ...else use the annotation value as the ID
            return sectionAnnotation.value();
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#getGeneralPrefix()
     */
    public String getGeneralPrefix()
    {
        return AbstractConfigurationsImpl.generalConfigurationPrefix;
    }

    /**
     * @param configPath
     *            the config path to validate
     * @return T if the specific configuration path should be perfixed for custom (prefixed) or common (not prefixed)
     *         configurations
     */
    protected boolean isPrefixedConfiguration(String configPath)
    {
        if (!isPrefixedConfigurationsActive())
            return false;
        else
        {
            List<String> exceptions = prefixException.get(generalConfigurationPrefix);

            if (exceptions == null)
                return true;
            else
            {
                for (String exceptionPath: exceptions)
                    if (configPath.startsWith(exceptionPath))
                        return false;

                return true;
            }
        }
    }

    /**
     * Inspector for the 'prefixedConfigurationsActive' attribute.
     *
     * @return the prefixedConfigurationsActive value
     */
    public boolean isPrefixedConfigurationsActive()
    {
        return prefixedConfigurationsActive;
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readAllConfigurationsPoints()
     */
    public void readAllConfigurationsPoints()
    {
        cachedConfigurationsPackages.clear();

        if (cachedConfigurations == null)
            cachedConfigurations = new HashMap<String, Class<?>>();
        else
            cachedConfigurations.clear();

        List<String> basePackages = new ArrayList<String>();

        for (Package pck: Package.getPackages())
        {
            String basePackage = pck.getName().split("\\.", 2)[0];

            if (!basePackages.contains(basePackage))
                basePackages.add(basePackage);
        }

        for (String basePackage: basePackages)
            readConfigurationsPointsForPackage(basePackage);
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readConfiguration(java.lang.Class)
     */
    public <T> T readConfiguration(Class<T> clazz)
    {

        if (clazz == null)
        {
            Logger.getLogger().error("The class cannot be null. Parse error!");
            return null;

        }
        else
        {
            String configID = getConfigID(clazz);
            String sectionID = getConfigSectionID(clazz);

            if ((configID != null) && (sectionID != null))
                return readConfiguration(configID, sectionID, clazz);
            else
                return null;
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readConfiguration(java.lang.String, java.lang.String)
     */
    public Properties readConfiguration(String configID, String sectionID)
    {
        return readConfiguration(configID, sectionID, isPrefixedConfiguration(configID + "/" + sectionID));
    }

    /**
     * Reads configurations from the persistence layer
     *
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @param usePrefix
     *            if the prefix should be used for reading
     * @return a Properties object with the key value pairs
     */
    abstract public Properties readConfiguration(String configID, String sectionID, boolean usePrefix);

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readConfiguration(java.lang.String, java.lang.String,
     *      java.lang.Class)
     */
    public <T> T readConfiguration(String configID, String sectionID, Class<T> clazz)
    {

        // Will define if the configurations should be updated
        boolean mustUpdate = false;

        if (clazz == null)
        {
            Logger.getLogger().error("The class cannot be null. Parse error!");
            return null;

        }
        else
        {

            Properties props = readConfiguration(configID, sectionID);

            T obj;
            try
            {
                obj = ReflectionUtils.instantiateObjectFromClass(clazz);

            }
            catch (AuxiliaryOperationException e)
            {
                Logger.getLogger().error("Could not create the Configuration Instance:\n" + e.getMessage());
                return null;
            }

            List<ConfigItem> configItems = getConfigItemsMap(clazz);
            Object valueAsObject;
            String value;

            for (ConfigItem item: configItems)
            {
                if (props == null)
                    value = null; // Will force the default value to be set if available
                else
                    value = props.getProperty(item.getKey()); // Read the value from the properties object

                if (value == null)
                    mustUpdate = true;

                // If not found, or defined as default, set it with the default value
                if (value == null || value.equalsIgnoreCase(DEFAULT_VALUE_KEYWORK))
                    value = item.getDefaultValue();

                if (!StringUtils.isBlank(value))
                {
                    // The argument will be created with the previously determined compatible class constructor that can
                    // receive a String argument. This will then be passed to the setter method

                    try
                    {
                        if ("null".equals(value))
                            valueAsObject = null;
                        else if (item.getItemClass() == Date.class)
                            valueAsObject = DateUtils.stringToDate(value);
                        else
                        {
                            if (item.getItemClass() == Double.class)
                            {
                                value = value.replace(',', '.');
                            }
                            valueAsObject = ReflectionUtils.instantiateObjectFromClass(item.getConstructor(), value);
                        }

                        ReflectionUtils.invokeMethod(item.getSetter(), obj, valueAsObject);

                    }
                    catch (ParseException e)
                    {
                        Logger.getLogger().error("Could not convert to Date: \"" + value + "\"\n" + e.getMessage());
                        return null;
                    }
                    catch (AuxiliaryOperationException e)
                    {
                        Logger.getLogger()
                                .error("Could not call setter: " + item.getSetter().getName() + "\n" + e.getMessage());
                        return null;
                    }
                }
            }

            // Update the configurations if any value was missing...
            if (mustUpdate)
                writeConfiguration(configID, sectionID, obj);

            return obj;
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readConfigurationAsMap(java.lang.String, java.lang.String)
     */
    public Map<String, String> readConfigurationAsMap(String configID, String sectionID)
    {
        Properties properties = readConfiguration(configID, sectionID);
        Map<String, String> propObj = new HashMap<String, String>();
        Iterator<Object> iteratorProps = properties.keySet().iterator();

        while (iteratorProps.hasNext())
        {
            Object object = iteratorProps.next();
            propObj.put(object.toString(), properties.getProperty(object.toString()));
        }
        return propObj;
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readConfigurationsPointsForPackage(java.lang.String)
     */
    public void readConfigurationsPointsForPackage(String basePackage)
    {
        if (basePackage == null || "".equals(basePackage))
            readAllConfigurationsPoints();

        else if (cachedConfigurations == null)
        {
            cachedConfigurations = new HashMap<String, Class<?>>();
            cachedConfigurationsPackages.clear();
        }

        if (!cachedConfigurationsPackages.contains(basePackage))
        {
            Logger.getLogger().debug("Reading configurations for package: \"" + basePackage + "\"");

            try
            {
                String configPath;
                List<String> classes = ResourceUtils.getClassesInDeepPackage(basePackage);

                for (String className: classes)
                {
                    try
                    {
                        Class<?> clazz = Class.forName(className);
                        configPath = getConfigPath(clazz);

                        if (configPath != null)
                            cachedConfigurations.put(configPath, clazz);
                    }
                    catch (Throwable e)
                    {
                        e.printStackTrace();
                        // Do nothing. Will discard all classes it cannot load
                    }
                }
            }
            catch (Throwable e)
            {
                // Do nothing. Will discard all classes it cannot load
            }

            cachedConfigurationsPackages.add(basePackage);
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#removeConfiguration(java.lang.Object)
     */
    public boolean removeConfiguration(Object bean) throws Exception
    {

        if (bean == null)
        {
            Logger.getLogger().error("The bean cannot be null. Parse error!");
            return false;

        }
        else
        {

            String configID = getConfigID(bean.getClass());
            String sectionID = getConfigSectionID(bean);

            if ((configID != null) && (sectionID != null))
                return removeConfiguration(configID, sectionID);
            else
                return false;
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#removeConfiguration(java.lang.String, java.lang.String)
     */
    public boolean removeConfiguration(String configID, String sectionID)
    {
        return removeConfiguration(configID, sectionID, isPrefixedConfiguration(configID + "/" + sectionID));
    }

    /**
     * Removes configurations from the persistence layer
     *
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @param usePrefix
     *            if the prefix should be used for reading
     * @return T if successful
     */
    abstract public boolean removeConfiguration(String configID, String sectionID, boolean usePrefix);

    /**
     * @see pt.digitalis.utils.config.IConfigurations#writeConfiguration(java.lang.Object)
     */
    public boolean writeConfiguration(Object annotatedPojo) throws Exception
    {

        if (annotatedPojo == null)
        {
            Logger.getLogger().error("The pojo cannot be null. Parse error!");
            return false;

        }
        else
        {

            String configID = getConfigID(annotatedPojo.getClass());
            String sectionID = getConfigSectionID(annotatedPojo);

            if ((configID != null) && (sectionID != null))
                return writeConfiguration(configID, sectionID, annotatedPojo);
            else
                return false;
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#writeConfiguration(java.lang.String, java.lang.String,
     *      java.lang.Object)
     */
    public boolean writeConfiguration(String configID, String sectionID, Object bean)
    {

        if (bean == null)
        {
            Logger.getLogger().error("The bean cannot be null. Parse error!");
            return false;
        }
        else
        {
            Properties props = new Properties();
            Object value;
            String valueString;

            List<ConfigItem> configItems = getConfigItemsMap(bean.getClass());

            try
            {
                for (ConfigItem item: configItems)
                {
                    value = ReflectionUtils.invokeMethod(item.getGetter(), bean);
                    valueString = "";

                    if (value != null)
                        if (item.getItemClass() == Date.class)
                            valueString = DateUtils.dateToString((Date) value);
                        else
                            valueString = value.toString();

                    if (valueString.equals(item.getDefaultValue()))
                        valueString = DEFAULT_VALUE_KEYWORK;

                    props.setProperty(item.getKey(), valueString);
                }

            }
            catch (AuxiliaryOperationException e)
            {
                return false;
            }

            return writeConfiguration(configID, sectionID, props);
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#writeConfiguration(java.lang.String, java.lang.String,
     *      java.util.Properties)
     */
    public boolean writeConfiguration(String configID, String sectionID, Properties values)
    {
        return writeConfiguration(configID, sectionID, values, isPrefixedConfiguration(configID + "/" + sectionID));
    }

    /**
     * Writes configurations to the persistence layer
     *
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @param values
     *            a Properties object with the key value pairs
     * @param usePrefix
     *            if the prefix should be used for writing
     * @return T if the operation was successful
     */
    abstract public boolean writeConfiguration(String configID, String sectionID, Properties values, boolean usePrefix);

    /**
     * @see pt.digitalis.utils.config.IConfigurations#writeConfigurationFromMap(java.lang.String, java.lang.String,
     *      java.util.Map)
     */
    public boolean writeConfigurationFromMap(String configID, String sectionID, Map<String, String> values)
    {
        if (values == null)
        {
            Logger.getLogger().error("The values map cannot be null. Parse error!");
            return false;
        }
        Properties props = new Properties();
        for (Entry<String, String> entrie: values.entrySet())
        {
            props.put(entrie.getKey(), entrie.getValue());
        }
        return writeConfiguration(configID, sectionID, props);
    }

}
