/**
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.config;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import pt.digitalis.log.Logger;

/**
 * The Class ConfigurationsPreferencesImpl.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Sep 27, 2007
 */
public class ConfigurationsPreferencesImpl extends AbstractConfigurationsImpl {

    /**
     * Instantiates a new configurations preferences impl.
     */
    public ConfigurationsPreferencesImpl()
    {
        super();
    }

    /**
     * Instantiates a new configurations preferences impl.
     * 
     * @param forcePrefixedConfigurationsActive
     *            the force prefixed configurations active
     */
    public ConfigurationsPreferencesImpl(boolean forcePrefixedConfigurationsActive)
    {
        super(forcePrefixedConfigurationsActive);
    }

    /**
     * Gets the childrens.
     *
     * @param configID
     *            the config ID
     * @param sectionID
     *            the section ID
     * @return the childrens
     * @throws BackingStoreException
     */
    private String[] getChildrens(String configID, String sectionID) throws BackingStoreException
    {
        boolean usePrefix = isPrefixedConfiguration(configID + "/" + sectionID);
        if (usePrefix)
            configID = getGeneralPrefix() + configID;

        if (sectionID != null && !"".equals(sectionID))
            configID += "/" + sectionID;

        if (!configID.startsWith("/"))
            configID = "/" + configID;

        Preferences systemPrefs = getPreferences(configID);

        return systemPrefs.childrenNames();
    }

    /**
     * Gets the general prefix.
     * 
     * @return the general prefix
     * @see pt.digitalis.utils.config.AbstractConfigurationsImpl#getGeneralPrefix()
     */
    @Override
    public String getGeneralPrefix()
    {
        return super.getGeneralPrefix() + "/";
    }

    /**
     * Gets the preferences.
     * 
     * @param configID
     *            the config id
     * @return the preferences
     */
    private Preferences getPreferences(String configID)
    {

        // WORKAROUND - Nodes with "_" char have the name replaced by Base64 full name encryption
        // SOURCE: Javadoc for FileSystemPreferences.isDirChar
        /**
         * Returns true if the specified character is appropriate for use in Unix directory names. A character is
         * appropriate if it's a printable ASCII character (> 0x1f && < 0x7f) and unequal to slash ('/', 0x2f), dot
         * ('.', 0x2e), or underscore ('_', 0x5f).
         */

        String node = configID.replaceAll("_", "-");

        if (System.getProperty("os.name").equalsIgnoreCase("Mac OS X"))
            return Preferences.userRoot().node(node);
        else
            return Preferences.systemRoot().node(node);
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readAllConfigurations(java.lang.String, java.lang.String)
     */
    public Map<String, Properties> readAllConfigurations(String configID, String sectionID)
    {
        Map<String, Properties> result = new HashMap<String, Properties>();

        boolean usePrefix = isPrefixedConfiguration(configID + "/" + sectionID);
        try
        {
            for (String name: getChildrens(configID, sectionID))
            {
                result.put(name, this.readConfiguration(configID, name, usePrefix));
            }
        }
        catch (BackingStoreException e)
        {
            Logger.getLogger().error("Repository not available. Configuration not loaded.\n" + e.getMessage());
        }

        return result;
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readAllConfigurations(java.lang.String, java.lang.String,
     *      java.lang.Class)
     */
    public <T> Map<String, T> readAllConfigurations(String configID, String sectionID, Class<T> clazz)
    {
        Map<String, T> result = new HashMap<String, T>();

        try
        {
            for (String name: getChildrens(configID, sectionID))
            {
                result.put(name, this.readConfiguration(configID, sectionID + "/" + name, clazz));
            }
        }
        catch (BackingStoreException e)
        {
            Logger.getLogger().error("Repository not available. Configuration not loaded.\n" + e.getMessage());
        }

        return result;
    }

    /**
     * Read configuration.
     * 
     * @param configID
     *            the config id
     * @param sectionID
     *            the section id
     * @param usePrefix
     *            the use prefix
     * @return the properties
     * @see pt.digitalis.utils.config.AbstractConfigurationsImpl#readConfiguration(java.lang.String, java.lang.String,
     *      boolean)
     */
    @Override
    public Properties readConfiguration(String configID, String sectionID, boolean usePrefix)
    {
        if (usePrefix)
            configID = getGeneralPrefix() + configID;

        if (sectionID != null && !"".equals(sectionID))
            configID += "/" + sectionID;

        if (!configID.startsWith("/"))
            configID = "/" + configID;

        Preferences systemPrefs = getPreferences(configID);
        Properties propObj = new Properties();

        try
        {
            systemPrefs.sync();
            StringBuffer out = new StringBuffer();
            for (String key: systemPrefs.keys())
                if (key != null && systemPrefs.get(key, null) != null)
                {
                    propObj.put(key, systemPrefs.get(key, null));
                    if (key.toLowerCase().contains("password"))
                    {
                        out.append("  " + key + "=[*******]\n");
                    }
                    else
                    {
                        out.append("  " + key + "=[" + propObj.getProperty(key) + "]\n");
                    }

                }
            if (out.length() > 0)
            {
                Logger.getLogger().debug("Configuration " + configID + ":\n" + out.toString());
            }
        }
        catch (BackingStoreException e)
        {
            Logger.getLogger().error("Repository not available. Configuration not loaded.\n" + e.getMessage());
        }

        return propObj;
    }

    /**
     * Removes the configuration.
     * 
     * @param configID
     *            the config id
     * @param sectionID
     *            the section id
     * @param usePrefix
     *            the use prefix
     * @return true, if successful
     * @see pt.digitalis.utils.config.AbstractConfigurationsImpl#removeConfiguration(java.lang.String, java.lang.String,
     *      boolean)
     */
    @Override
    public boolean removeConfiguration(String configID, String sectionID, boolean usePrefix)
    {
        if (usePrefix)
            configID = getGeneralPrefix() + configID;

        if (sectionID != null && !"".equals(sectionID))
            configID += "/" + sectionID;

        if (!configID.startsWith("/"))
            configID = "/" + configID;

        try
        {
            Preferences systemPrefs = getPreferences(configID);
            systemPrefs.removeNode();
            systemPrefs.flush();
        }
        catch (BackingStoreException e)
        {
            Logger.getLogger().error("Repository not available. Configuration not removed.\n" + e.getMessage());

            return false;
        }

        return true;
    }

    /**
     * Write configuration.
     * 
     * @param configID
     *            the config id
     * @param sectionID
     *            the section id
     * @param values
     *            the values
     * @param usePrefix
     *            the use prefix
     * @return true, if successful
     * @see pt.digitalis.utils.config.AbstractConfigurationsImpl#writeConfiguration(java.lang.String, java.lang.String,
     *      java.util.Properties, boolean)
     */
    @Override
    public boolean writeConfiguration(String configID, String sectionID, Properties values, boolean usePrefix)
    {
        if (usePrefix)
            configID = getGeneralPrefix() + configID;

        if (sectionID != null && !"".equals(sectionID))
            configID += "/" + sectionID;

        if (!configID.startsWith("/"))
            configID = "/" + configID;

        Preferences systemPrefs = getPreferences(configID);

        for (Entry<Object, Object> entry: values.entrySet())
            if (entry.getValue() != null)
                systemPrefs.put(entry.getKey().toString(), entry.getValue().toString());

        try
        {
            systemPrefs.flush();

        }
        catch (BackingStoreException e)
        {
            Logger.getLogger().error("Repository not available. Configuration not saved.\n" + e.getMessage());

            return false;
        }

        return true;
    }
}
