/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.config;

import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Interface that provides a common interface for configuration persistence purposes. It provides a structure for
 * configurations in the following form: configID -> sectionID -> List of "key = value"
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Sep 27, 2007
 */
public interface IConfigurations {

    /**
     * Returns the previously cached configuration points. Will call cacheAllConfigurations method if no configurations
     * are yet in cache.
     * 
     * @return the configuration map with configuration path and corresponding class
     */
    public Map<String, Class<?>> getCacheConfigurationPoints();

    /**
     * The list of the methods and keys read from the ConfigKeyID annotated methods or other getters as long as not
     * ConfigIgnore annotated. Note: Only setters that receives a class argument that has a String constructor will be
     * supported. In order to have a custom class supported just implement a constructor(String) and the toString method
     * to provide conversion to/from your own class object. Some types are automatically converted to classes: int,
     * long, double, boolean
     * 
     * @param clazz
     *            the class to parse
     * @return the list of ConfigItems
     */
    public List<ConfigItem> getConfigItemsMap(Class<?> clazz);

    /**
     * @return Returns the common configuration prefix
     */
    public String getGeneralPrefix();

    /**
     * Read all configurations inside a configID/SectionUD.
     *
     * @param configID
     *            the config ID
     * @param sectionID
     *            the section ID
     * @return the map the result where the key is the children name and the value the properties
     */
    public Map<String, Properties> readAllConfigurations(String configID, String sectionID);

    /**
     * Read all configurations inside a configID/SectionUD.
     *
     * @param <T>
     *            the generic type
     * @param configID
     *            the config ID
     * @param sectionID
     *            the section ID
     * @param clazz
     *            the class that defines the object that represents the properties to read
     * @return the map the result where the key is the children name and the value the properties
     */
    public <T> Map<String, T> readAllConfigurations(String configID, String sectionID, Class<T> clazz);

    /**
     * Parses all classes for configuration points and build a map of all configurations found
     */
    public void readAllConfigurationsPoints();

    /**
     * Reads configurations from the persistence layer. It will scan the given class for annotations that instruct how
     * to read the values. The annotations are the ones present in "pt.digitalis.dif.config.annotations" If not
     * available a simple conversion of attributes to key name will be used, as in: "method: getFullName -> key
     * FullName" Note: The ConfigID and ConfigSectionID annotations are mandatory for this method to succeed
     * 
     * @param <T>
     *            the Type to create
     * @param clazz
     *            the class that defines the object that represents the properties to read
     * @return an instance of class with the read values
     */
    public <T> T readConfiguration(Class<T> clazz);

    /**
     * Reads configurations from the persistence layer
     * 
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @return a Properties object with the key value pairs
     */
    public Properties readConfiguration(String configID, String sectionID);

    /**
     * Reads configurations from the persistence layer. It will scan the given class for annotations that instruct how
     * to read the values. The annotations are the ones present in "pt.digitalis.utils.config.annotations" If not
     * available a simple conversion of attributes to key name will be used, as in: "method: getFullName -> key
     * FullName"
     * 
     * @param <T>
     *            the Type to create
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @param clazz
     *            the class that defines the object that represents the properties to read
     * @return an instance of class with the read values
     */
    public <T> T readConfiguration(String configID, String sectionID, Class<T> clazz);

    /**
     * Reads configurations from the persistence layer
     * 
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @return a Properties object with the key value pairs
     */
    public Map<String, String> readConfigurationAsMap(String configID, String sectionID);

    /**
     * Parses all classes of the given package for configuration points and build a map of all configurations found
     * 
     * @param basePackage
     *            the base package to search configurations classes on
     */
    public void readConfigurationsPointsForPackage(String basePackage);

    /**
     * Removes configurations from the persistence layer. It will scan the given class for annotations that instruct
     * what configurations it should remove. The annotations are the ones present in
     * "pt.digitalis.dif.config.annotations" Note: The ConfigID and ConfigSectionID annotations are mandatory for this
     * method to succeed
     * 
     * @param bean
     *            the object (JavaBean) that will be read and saved
     * @return T if successful
     * @throws Exception
     *             if the return object cannot be created
     */
    public boolean removeConfiguration(Object bean) throws Exception;

    /**
     * Removes configurations from the persistence layer
     * 
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @return T if successful
     */
    public boolean removeConfiguration(String configID, String sectionID);

    /**
     * Writes configurations to the persistence layer. A simple conversion of attributes to key name will be used, as
     * in: "method: getFullName -> key FullName"
     * 
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @param bean
     *            the object (JavaBean) that will be read and saved
     * @return T if the operation was successful
     */
    public boolean writeConfiguration(String configID, String sectionID, Object bean);

    /**
     * Writes configurations to the persistence layer
     * 
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @param values
     *            a Properties object with the key value pairs
     * @return T if the operation was successful
     */
    public boolean writeConfiguration(String configID, String sectionID, Properties values);

    /**
     * Writes configurations to the persistence layer. It will scan the given class for annotations that instruct how to
     * read the values. The annotations are the ones present in "pt.digitalis.dif.config.annotations" If not available a
     * simple conversion of attributes to key name will be used, as in: "method: getFullName -> key FullName"
     * 
     * @param <T>
     *            the Type to create
     * @param annotatedPojo
     *            the annotated class object with the values to be saved
     * @return T if the operation was successful
     * @throws Exception
     *             if the bean class is not found
     */
    public <T> boolean writeConfiguration(T annotatedPojo) throws Exception;

    /**
     * Writes configurations to the persistence layer
     * 
     * @param configID
     *            the identifier of the configuration group
     * @param sectionID
     *            the identifier of the section within the group
     * @param values
     *            a Properties object with the key value pairs
     * @return T if the operation was successful
     */
    public boolean writeConfigurationFromMap(String configID, String sectionID, Map<String, String> values);

}
