package pt.digitalis.utils.config.tests;

import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import junit.framework.TestCase;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.config.IConfigurations;

/**
 * Abstract Test suite for Configuration specific implementations.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created 2007/09/13
 */
public abstract class AbstractConfigurationsSpecificImplTest extends TestCase {

    /** A sample date to use for comparison in the tests. */
    private final Date sampleDate = new GregorianCalendar(2007, 10, 2).getTime();

    /**
     * Instantiates a new abstract configurations specific impl test.
     */
    public AbstractConfigurationsSpecificImplTest()
    {
        super();
    }

    /**
     * Instantiates a new abstract configurations specific impl test.
     *
     * @param name
     *            the name
     */
    public AbstractConfigurationsSpecificImplTest(String name)
    {
        super(name);
    }

    /**
     * Gets the implementation.
     *
     * @return the implementation
     * @throws ConfigurationException
     */
    abstract protected IConfigurations getImplementation() throws ConfigurationException;

    /**
     * Tests removing and reading default configuration from an annotated pojo.
     * 
     * @throws Exception
     *             the exception
     */
    public void testDefaultRemoveConfigurationAnnotatedPojo() throws Exception
    {
        IConfigurations config = this.getImplementation();

        // Initialize a pojo
        ConfigsMockupAnnotatedPojo pojo = new ConfigsMockupAnnotatedPojo();
        pojo.setName("Abraham Lincoln");
        pojo.setAddress("Washington DC - President's Street 1");

        // Persist it to the repository in the temp config slot
        config.writeConfiguration("TestConfigurations", "temporaryAnnotatedPojo", pojo);

        // Read it again
        pojo = config.readConfiguration("TestConfigurations", "temporaryAnnotatedPojo", pojo.getClass());
        assertNotNull(pojo);
        assertTrue(pojo.getName().equals("Abraham Lincoln"));
        assertTrue(pojo.getAddress().equals("Washington DC - President's Street 1"));

        // Remove the configuration
        assertTrue(config.removeConfiguration("TestConfigurations", "temporaryAnnotatedPojo"));

        // Read it again. Should not exist so null values and defaults should be returned
        pojo = config.readConfiguration("TestConfigurations", "temporaryAnnotatedPojo", pojo.getClass());
        assertNotNull(pojo);
        assertNull(pojo.getName());
        assertTrue(pojo.getAddress().equals("Our little blue planet!"));

        // Remove the configuration using annotated configuration parameters
        assertTrue(config.removeConfiguration(pojo));

        // Read it again. Should not exist so null values and defaults should be returned
        pojo = config.readConfiguration(ConfigsMockupAnnotatedPojo.class);
        assertNotNull(pojo);
        assertNull(pojo.getName());
        assertTrue(pojo.getAddress().equals("Our little blue planet!"));
    }

    /**
     * Test read all configurations.
     *
     * @throws Exception
     *             the exception
     */
    public void testReadAllConfigurations() throws Exception
    {
        IConfigurations config = this.getImplementation();
        assertNotNull(config);

        testWriteConfigurationAnnotatedPojo();
        testWriteConfigurationBean();
        testWriteConfigurationFromMap();
        testWriteConfigurationProperties();

        Map<String, Properties> confs = config.readAllConfigurations("TestConfigurations", "");
        assertTrue(confs.size() > 4);

        assertTrue(confs.containsKey("annotatedPojo"));
        assertTrue(confs.containsKey("bean"));
        assertTrue(confs.containsKey("properties"));

        ConfigsMockupAnnotatedPojo annPojo = config.readConfiguration(ConfigsMockupAnnotatedPojo.class);
        annPojo.setName("Name#1");
        config.writeConfiguration("TestConfigurations", "annotatedPojoConfigs/1", annPojo);
        annPojo.setName("Name#2");
        config.writeConfiguration("TestConfigurations", "annotatedPojoConfigs/2", annPojo);

        Map<String, ConfigsMockupAnnotatedPojo> results = config.readAllConfigurations("TestConfigurations",
                "annotatedPojoConfigs", ConfigsMockupAnnotatedPojo.class);
        assertNotNull(results);
        assertEquals(results.size(), 2);
        assertNotNull(results.get("1"));
        assertEquals(results.get("1").getName(), "Name#1");
        assertNotNull(results.get("2"));
        assertEquals(results.get("2").getName(), "Name#2");
    }

    /**
     * Loads a configuration from an annotated pojo.
     * 
     * @throws Exception
     *             the exception
     */
    public void testReadConfigurationAnnotatedPojo() throws Exception
    {
        IConfigurations config = this.getImplementation();

        ConfigsMockupAnnotatedPojo pojo = new ConfigsMockupAnnotatedPojo();
        pojo.setName("Abraham Lincoln");
        pojo.setAddress("Washington DC - President's Street 1");
        pojo.setAge(300);
        // Persist it to the repository for reading tests
        config.writeConfiguration("TestConfigurations", "annotatedPojo", pojo);

        pojo = config.readConfiguration(ConfigsMockupAnnotatedPojo.class);
        Properties props = config.readConfiguration("TestConfigurations", "annotatedPojo");

        assertNotNull(pojo);

        assertTrue(pojo.getName().equals("Abraham Lincoln"));
        assertTrue(pojo.getAddress().equals("Washington DC - President's Street 1"));
        assertTrue(pojo.getAge() == 300);

        assertTrue(props.getProperty("FullName").equals(pojo.getName()));
        assertTrue(props.getProperty("Address").equals(pojo.getAddress()));
        assertTrue(props.size() == 4);
    }

    /**
     * Loads a configuration from a properties file to map.
     * 
     * @throws ConfigurationException
     */
    public void testReadConfigurationAsMap() throws ConfigurationException
    {
        IConfigurations config = this.getImplementation();
        Map<String, String> propsMap = config.readConfigurationAsMap("TestConfigurations", "properties");

        assertNotNull(propsMap);
        assertTrue(propsMap.size() == 10);

        for (String key: propsMap.keySet())
            assertTrue(propsMap.get(key).equals("value " + key));
    }

    /**
     * Loads a configuration from a simple bean.
     * 
     * @throws Exception
     *             the exception
     */
    public void testReadConfigurationBean() throws Exception
    {
        IConfigurations config = this.getImplementation();

        // Non-anotated pojo
        ConfigsMockupPojo pojo = config.readConfiguration("TestConfigurations", "bean", ConfigsMockupPojo.class);

        assertNotNull(pojo);

        assertTrue(pojo.getName().equals("Abraham Lincoln"));
        assertTrue(pojo.getAddress().equals("Washington DC - President's Street 1"));
        assertTrue(pojo.getAge() == 300);
        assertTrue(pojo.getBirthDate().equals(sampleDate));

        // Non-anotated pojo
        ConfigsMockupAnnotatedPojo annPojo = config.readConfiguration(ConfigsMockupAnnotatedPojo.class);

        assertNotNull(annPojo);
        assertTrue(annPojo.getName().equals("Abraham Lincoln"));
        assertTrue(annPojo.getAddress().equals("Washington DC - President's Street 1"));
        assertTrue(annPojo.getAge() == 300);

        Properties propsAnnPojo = config.readConfigurationAsProperties(ConfigsMockupAnnotatedPojo.class);
        assertNotNull(propsAnnPojo);
        assertTrue(propsAnnPojo.getProperty("FullName").equals("Abraham Lincoln"));
        assertTrue(propsAnnPojo.getProperty("Address").equals("Washington DC - President's Street 1"));
        assertTrue(propsAnnPojo.getProperty("Age").equals("300"));
    }

    /**
     * Loads a configuration from a properties file.
     * 
     * @throws ConfigurationException
     */
    public void testReadConfigurationProperties() throws ConfigurationException
    {
        IConfigurations config = this.getImplementation();
        Properties props = config.readConfiguration("TestConfigurations", "properties");

        assertNotNull(props);
        assertTrue(props.size() == 10);

        for (Object key: props.keySet())
            assertTrue(props.getProperty((String) key).equals("value " + key));
    }

    /**
     * Test read write configuration with under score char.
     * 
     * @throws ConfigurationException
     */
    public void testReadWriteConfigurationWithUnderScoreChar() throws ConfigurationException
    {

        IConfigurations config = this.getImplementation();
        Properties props = new Properties();
        String sectionID = "name_with_underscore";
        for (int i = 0; i < 10; i++)
            props.put(Integer.toString(i), "value " + i);

        props.put("valueToReturn", "MyValue");

        assertTrue(config.writeConfiguration("TestConfigurations", sectionID, props));
        props = config.readConfiguration("TestConfigurations", sectionID);
        assertNotNull(props);
        assertEquals(props.getProperty("valueToReturn"), "MyValue");

        assertTrue(config.removeConfiguration("TestConfigurations", sectionID));
        props = config.readConfiguration("TestConfigurations", sectionID);
        assertEquals(props.size(), 0);
    }

    /**
     * Saves a new configuration from an annotated pojo.
     * 
     * @throws Exception
     *             the exception
     */
    public void testWriteConfigurationAnnotatedPojo() throws Exception
    {
        IConfigurations config = this.getImplementation();
        ConfigsMockupAnnotatedPojo pojo = new ConfigsMockupAnnotatedPojo();
        pojo.setName("Abraham Lincoln");
        pojo.setAddress("Washington DC - President's Street 1");
        pojo.setAge(300);

        assertTrue(config.writeConfiguration(pojo));
    }

    /**
     * Saves a new configuration from a simple bean.
     * 
     * @throws Exception
     *             the exception
     */
    public void testWriteConfigurationBean() throws Exception
    {
        IConfigurations config = this.getImplementation();
        ConfigsMockupPojo bean = new ConfigsMockupPojo();
        bean.setName("Abraham Lincoln");
        bean.setAddress("Washington DC - President's Street 1");
        bean.setAge(300);
        bean.setBirthDate(sampleDate);

        assertTrue(config.writeConfiguration("TestConfigurations", "bean", bean));
    }

    /**
     * Testing write configuration from map.
     * 
     * @throws ConfigurationException
     */
    public void testWriteConfigurationFromMap() throws ConfigurationException
    {
        IConfigurations config = this.getImplementation();
        Map<String, String> propsMap = new HashMap<String, String>();

        for (int i = 0; i < 10; i++)
            propsMap.put(Integer.toString(i), "value " + i);

        assertTrue(config.writeConfigurationFromMap("TestConfigurations", "properties", propsMap));
    }

    /**
     * Saves a new configuration from a properties file.
     * 
     * @throws ConfigurationException
     */
    public void testWriteConfigurationProperties() throws ConfigurationException
    {
        IConfigurations config = this.getImplementation();
        Properties props = new Properties();

        for (int i = 0; i < 10; i++)
            props.put(Integer.toString(i), "value " + i);

        assertTrue(config.writeConfiguration("TestConfigurations", "properties", props));

        props = new Properties();
        props.setProperty("FullName", "NameForTesting");
        props.setProperty("Address", "AddressForTesting");
        props.setProperty("Age", "3");
        assertTrue(config.writeConfiguration(ConfigsMockupAnnotatedPojo.class, props));

        ConfigsMockupAnnotatedPojo annPojo = config.readConfiguration(ConfigsMockupAnnotatedPojo.class);
        assertNotNull(annPojo);
        assertEquals(annPojo.getName(), "NameForTesting");
        assertEquals(annPojo.getAddress(), "AddressForTesting");
        assertEquals(annPojo.getAge(), 3);
    }

    /**
     * Testing write configuration from map.
     * 
     * @throws ConfigurationException
     */
    public void testWriteConfigurationWithConcurrency() throws ConfigurationException
    {
        final IConfigurations config = this.getImplementation();
        Map<String, String> propsMap = new HashMap<String, String>();
        int propsTotal = 10;

        for (int i = 0; i < propsTotal; i++)
            propsMap.put(Integer.toString(i), "value " + i);

        final Map<String, String> finalPropsMap = new HashMap<String, String>(propsMap);

        config.removeConfiguration("TestConfigurations", "propertiesWithConcurrency");

        Runnable runnable = new Runnable() {

            public void run()
            {
                try
                {
                    config.writeConfigurationFromMap("TestConfigurations", "propertiesWithConcurrency", finalPropsMap);
                }
                catch (ConfigurationException e)
                {
                    e.printStackTrace();
                }
            }
        };

        new Thread(runnable).start();
        new Thread(runnable).start();
        new Thread(runnable).start();
        new Thread(runnable).start();

        assertTrue(config.writeConfigurationFromMap("TestConfigurations", "propertiesWithConcurrency", propsMap));

        assertEquals(propsTotal, config.readConfiguration("TestConfigurations", "propertiesWithConcurrency").size());

    }
}