/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.config;

import pt.digitalis.log.Logger;
import pt.digitalis.utils.common.Chronometer;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.DateUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.annotations.ConfigAlias;
import pt.digitalis.utils.config.annotations.ConfigDefault;
import pt.digitalis.utils.config.annotations.ConfigID;
import pt.digitalis.utils.config.annotations.ConfigIgnore;
import pt.digitalis.utils.config.annotations.ConfigKeyID;
import pt.digitalis.utils.config.annotations.ConfigLOVAjaxEvent;
import pt.digitalis.utils.config.annotations.ConfigLOVValues;
import pt.digitalis.utils.config.annotations.ConfigPrivate;
import pt.digitalis.utils.config.annotations.ConfigSectionID;
import pt.digitalis.utils.config.annotations.ConfigSectionIDGetter;
import pt.digitalis.utils.config.annotations.ConfigTestClass;
import pt.digitalis.utils.config.annotations.ConfigVirtualPathForNode;
import pt.digitalis.utils.inspection.ReflectionUtils;
import pt.digitalis.utils.inspection.ResourceUtils;
import pt.digitalis.utils.inspection.exception.AuxiliaryOperationException;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;

/**
 * A facilitator base implementation for IConfigurations. It implements the recurrent needs of these implementations
 * allowing the descending classes to focus only on what is specific to each implementation.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Sep 27, 2007
 */
abstract public class AbstractConfigurationsImpl implements IConfigurations
{

    /** The value used to save the value when the default value is active. */
    public static final String DEFAULT_VALUE_KEYWORK = "#DefaultValue#";

    /** The common configuration prefix to apply to all configuration paths. */
    static public String generalConfigurationPrefix = "";

    /** The cached configurations. */
    static protected Map<String, Class<?>> cachedConfigurations = null;

    /** The cached configurations indexed by virtual path. */
    static protected Map<String, Class<?>> cachedConfigurationsByVirtualPath = null;

    /** The cached configurations packages searched so far. */
    static protected List<String> cachedConfigurationsPackages = new ArrayList<String>();

    /** The ignore test configurations. */
    private boolean ignoreTestConfigurations;

    /**
     * @param ignoreTestConfigurations
     */
    public AbstractConfigurationsImpl(boolean ignoreTestConfigurations)
    {
        super();
        this.ignoreTestConfigurations = ignoreTestConfigurations;
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#getCacheConfigurationPoints()
     */
    @Override
    public Map<String, Class<?>> getCacheConfigurationPoints() throws ConfigurationException
    {
        if (cachedConfigurations == null)
        {
            this.readAllConfigurationsPoints();
        }

        return cachedConfigurations;
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#getCacheConfigurationPointsByVirtualPath()
     */
    @Override
    public Map<String, Class<?>> getCacheConfigurationPointsByVirtualPath() throws ConfigurationException
    {
        if (cachedConfigurationsByVirtualPath == null)
        {
            this.readAllConfigurationsPoints();
        }

        return cachedConfigurationsByVirtualPath;
    }

    /**
     * Returns the config of an annotated configuration object.
     *
     * @param clazz the class to parse
     *
     * @return the Config value if present
     */
    public String getConfigID(Class<?> clazz)
    {

        // Search for the ConfigID annotation in the object class
        ConfigID configAnnotation = clazz.getAnnotation(ConfigID.class);

        if (configAnnotation == null)
        // If we did not find it return null
        {
            return null;
        }
        else
        // ...else return the annotation value
        {
            return configAnnotation.value();
        }
    }

    /**
     * Gets the config items map.
     *
     * @param clazz the clazz
     *
     * @return the config items map
     *
     * @see pt.digitalis.utils.config.IConfigurations#getConfigItemsMap(java.lang.Class)
     */
    @Override
    public List<ConfigItem> getConfigItemsMap(Class<?> clazz)
    {

        List<ConfigItem> configItems = new ArrayList<ConfigItem>();
        boolean booleanItem;

        // Lets parse all methods
        for (Method method : clazz.getMethods())
        {

            // Exclude inherited methods && if it's a getter method not configIgnore annotated
            if (method.getDeclaringClass() == clazz &&
                ((method.getName().startsWith("get") || method.getName().startsWith("is")) &&
                 !method.isAnnotationPresent(ConfigIgnore.class)))
            {
                booleanItem = method.getName().startsWith("is");

                boolean isPrivate = method.isAnnotationPresent(ConfigPrivate.class);

                // This is a valid item. Let's see if it has a supported Class getter/setter.
                // Only classes that have a constructor with a String argument may be used

                String key;
                Method getter = method;
                Method setter;

                try
                {
                    if (booleanItem)
                    // Remove the "is" and add "set" prefix to determine the setter method
                    {
                        setter = ReflectionUtils.getMethod(clazz, "set" + method.getName().substring(2));
                    }
                    else
                    // Remove the "get" and add "set" prefix to determine the setter method
                    {
                        setter = ReflectionUtils.getMethod(clazz, "set" + method.getName().substring(3));
                    }

                    if (setter == null)
                    {
                        throw new ResourceNotFoundException(
                                "Could not find setter method: " + "set" + method.getName().substring(3) + " for " +
                                clazz.getSimpleName());
                    }

                    if (setter.getParameterTypes().length != 1)
                    {
                        Logger.getLogger().error("More than one parameter: method=" + setter.toGenericString());
                    }
                    else
                    {
                        Class<?> parameterClass = setter.getParameterTypes()[0];
                        Constructor<?> constructor;

                        if (booleanItem && (parameterClass != boolean.class && parameterClass != Boolean.class))
                        {
                            Logger.getLogger()
                                    .error("Prefix \"is\" for getter methods only allows for boolean attributes: method=" +
                                           setter.toGenericString());
                        }
                        else
                        {
                            if (parameterClass == int.class)
                            {
                                parameterClass = Integer.class;
                            }
                            else if (parameterClass == long.class)
                            {
                                parameterClass = Long.class;
                            }
                            else if (parameterClass == double.class)
                            {
                                parameterClass = Double.class;
                            }
                            else if (parameterClass == boolean.class)
                            {
                                parameterClass = Boolean.class;
                            }

                            try
                            {
                                constructor = parameterClass.getConstructor(String.class);
                            }
                            catch (SecurityException e)
                            {
                                constructor = null;
                            }
                            catch (NoSuchMethodException e)
                            {
                                constructor = null;
                            }

                            if (constructor == null)
                            {
                                System.err.println(method.getDeclaringClass() + "#" + method.getName());
                                Logger.getLogger()
                                        .error("No constructor in the argument class for String value: method=" +
                                               setter.toGenericString());
                            }

                            else
                            {
                                if (method.isAnnotationPresent(ConfigKeyID.class))
                                {

                                    // ConfigKeyID annotated getter method
                                    ConfigKeyID keyAnnotation = method.getAnnotation(ConfigKeyID.class);

                                    key = keyAnnotation.value();
                                }
                                else
                                {
                                    String methodName = method.getName();

                                    // Simple getter method not ConfigIgnore annotated
                                    if (methodName.startsWith("is"))
                                    {
                                        key = method.getName().substring(2);
                                    }
                                    else
                                    {
                                        key = method.getName().substring(3);
                                    }
                                }

                                // Read default value if annotation present
                                ConfigDefault defaultValueAnnotation = method.getAnnotation(ConfigDefault.class);

                                String defaultValue;

                                if (defaultValueAnnotation == null)
                                {
                                    defaultValue = null;
                                }
                                else
                                {
                                    defaultValue = defaultValueAnnotation.value();
                                }

                                ConfigItem configItem = new ConfigItem(key, getter, setter, constructor, defaultValue,
                                        getter.getReturnType());
                                configItem.setPrivate(isPrivate);

                                ConfigAlias alias = method.getAnnotation(ConfigAlias.class);

                                String name = (alias != null && StringUtils.isNotBlank(alias.name())) ? alias.name()
                                                                                                      : StringUtils
                                                      .camelCaseToString(key);
                                String description =
                                        (alias != null && StringUtils.isNotBlank(alias.name())) ? alias.description()
                                                                                                : null;

                                configItem.setName(name);
                                configItem.setDescription(description);

                                ConfigLOVValues lovValuesAnnotation = method.getAnnotation(ConfigLOVValues.class);
                                if (lovValuesAnnotation != null && StringUtils.isNotBlank(lovValuesAnnotation.value()))
                                {
                                    configItem.setLovValues(
                                            CollectionUtils.keyValueStringToMap(lovValuesAnnotation.value()));
                                }

                                ConfigLOVAjaxEvent lovAjaxAnnotation = method.getAnnotation(ConfigLOVAjaxEvent.class);
                                if (lovAjaxAnnotation != null && StringUtils.isNotBlank(lovAjaxAnnotation.value()))
                                {
                                    configItem.setLovAjaxEvent(lovAjaxAnnotation.value());
                                }

                                // Add the item record with the read values
                                configItems.add(configItem);
                            }
                        }
                    }
                }
                catch (ResourceNotFoundException e)
                {
                    Logger.getLogger().error(e.getMessage());
                }

                // }

            }
        }

        return configItems;
    }

    /**
     * Returns the config of an annotated configuration object.
     *
     * @param clazz the class to parse
     *
     * @return the Config value if present
     */
    private String getConfigPath(Class<?> clazz)
    {
        String configID = this.getConfigID(clazz);
        String configSectionID = this.getConfigSectionID(clazz);

        if (configID != null)
        {
            if (configSectionID == null)
            {
                return configID;
            }
            else
            {
                return configID + "/" + configSectionID;
            }
        }
        else
        {
            return null;
        }
    }

    /**
     * Returns the configSection of an annotated configuration class.
     *
     * @param clazz the class to parse
     *
     * @return the ConfigSection value if present
     */
    public String getConfigSectionID(Class<?> clazz)
    {

        ConfigSectionID sectionAnnotation = clazz.getAnnotation(ConfigSectionID.class);

        if (sectionAnnotation == null)
        {
            return null;
        }
        else
        {
            return sectionAnnotation.value();
        }
    }

    /**
     * Returns the configSection of an annotated configuration object.
     *
     * @param obj the object to parse
     *
     * @return the ConfigSection value if present
     */
    public String getConfigSectionID(Object obj)
    {

        // Search for the ConfigSectionID annotation in the object class
        ConfigSectionID sectionAnnotation = obj.getClass().getAnnotation(ConfigSectionID.class);

        // If it was not found...
        if (sectionAnnotation == null)
        {

            // ConfigSectionID annotation was not used. Search for the ConfigSectionIDGetter
            ConfigSectionIDGetter sectionMethodAnnotation = null;
            Method sectionMethod = null;

            // Search ConfigSessionIDGetter annotated methods
            for (Method method : obj.getClass().getMethods())
            {
                sectionMethodAnnotation = method.getAnnotation(ConfigSectionIDGetter.class);

                // If found save it and exit the loop...
                if (sectionMethodAnnotation != null)
                {
                    sectionMethod = method;
                    break;
                }
            }

            if (sectionMethod == null)
            // If we did not found the annotation return null...
            {
                return null;
            }
            else
            {
                // ...else we must execute the method to get the value to use as the ID

                try
                {
                    return ReflectionUtils.invokeMethod(sectionMethod, obj).toString();
                }
                catch (Exception e)
                {

                    Logger.getLogger().error(e.getMessage());
                    return null;
                }
            }
        }
        else
        // ...else use the annotation value as the ID
        {
            return sectionAnnotation.value();
        }
    }

    /**
     * Returns the config virtual path of an annotated configuration object.
     *
     * @param clazz the class to parse
     *
     * @return the Config virtual path value if present or the default one if not
     */
    private String getConfigVirtualPath(Class<?> clazz)
    {
        ConfigVirtualPathForNode annotation = clazz.getAnnotation(ConfigVirtualPathForNode.class);

        if (annotation == null)
        {
            return this.getConfigPath(clazz);
        }
        else
        {
            return annotation.value();
        }
    }

    /**
     * Gets the general prefix.
     *
     * @return the general prefix
     *
     * @see pt.digitalis.utils.config.IConfigurations#getGeneralPrefix()
     */
    @Override
    public String getGeneralPrefix()
    {
        return AbstractConfigurationsImpl.generalConfigurationPrefix;
    }

    /**
     * Configures the general prefix.
     *
     * @param configurationPrefix the new general prefix
     */
    static public void setGeneralPrefix(String configurationPrefix)
    {
        AbstractConfigurationsImpl.generalConfigurationPrefix = configurationPrefix;
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#getRealPathForVirtualOne(java.lang.String)
     */
    @Override
    public String getRealPathForVirtualOne(String virtualPath) throws ConfigurationException
    {
        Class<?> clazz = this.getCacheConfigurationPointsByVirtualPath().get(virtualPath);

        if (clazz == null)
        {
            return null;
        }
        else
        {
            return this.getConfigPath(clazz);
        }
    }

    /**
     * Returns T if the class is a configuration test class {@link ConfigTestClass}
     *
     * @param clazz
     *
     * @return the ConfigSection value if present
     */
    protected boolean isTestConfigurationClass(Class<?> clazz)
    {

        ConfigTestClass testClassAnnotation = clazz.getAnnotation(ConfigTestClass.class);

        return testClassAnnotation != null;
    }

    /**
     * Migrate from another {@link IConfigurations} implementation.
     *
     * @param oldConfigs     the old configs
     * @param stopIfNotEmpty the stop if not empty
     *
     * @return true, if successful
     *
     * @exception ConfigurationException the configuration exception
     */
    protected boolean migrateFromOtherImplementation(IConfigurations oldConfigs, boolean stopIfNotEmpty)
            throws ConfigurationException
    {
        IConfigurations newConfigs = this;

        Logger.getLogger()
                .warn("Migrating Configurations repository " + oldConfigs.getClass().getSimpleName() + " => " +
                      newConfigs.getClass().getSimpleName());

        Chronometer crono = new Chronometer();
        crono.start();

        if (stopIfNotEmpty && !newConfigs.isPersistentRepositoryEmpty())
        {
            Logger.getLogger().warn("Destination repository is not empty. Stoped migration process!");
            return false;
        }
        else
        {
            int migratedConfigs = 0;

            Logger.getLogger().warn("Parsing old configuration nodes...");
            Map<String, Properties> nodeNames = oldConfigs.readAllConfigurations("", "");

            Logger.getLogger().warn("Initiating migration...");
            for (Entry<String, Properties> node : nodeNames.entrySet())
            {
                migratedConfigs += this.migrateNodeTree("", node, oldConfigs);
            }

            crono.end();

            Logger.getLogger().warn("Migration complete! (" + migratedConfigs + " configuration nodes migrated in " +
                                    crono.getTimePassedAsFormattedString() + ")");

            return true;
        }
    }

    /**
     * Migrate node tree.
     *
     * @param previousPath the previous path
     * @param node         the node
     * @param oldConfigs   the old configs
     *
     * @return the int
     *
     * @exception ConfigurationException the configuration exception
     */
    private int migrateNodeTree(String previousPath, Entry<String, Properties> node, IConfigurations oldConfigs)
            throws ConfigurationException
    {
        IConfigurations newConfigs = this;
        int migratedConfigs = 0;

        Properties nodeProps = oldConfigs.readConfiguration(previousPath, node.getKey());

        if (!nodeProps.isEmpty())
        {

            Logger.getLogger().info("   - Node: " + (StringUtils.isNotBlank(previousPath) ? previousPath + "/" : "") +
                                    node.getKey());

            newConfigs.writeConfiguration(previousPath, node.getKey(), nodeProps);
            migratedConfigs++;
        }

        Map<String, Properties> children;

        if (StringUtils.isBlank(previousPath))
        {
            children = oldConfigs.readAllConfigurations(node.getKey(), "");
        }
        else
        {
            children = oldConfigs.readAllConfigurations(previousPath, node.getKey());
        }

        if (StringUtils.isNotBlank(previousPath))
        {
            previousPath += "/";
        }

        for (Entry<String, Properties> childNode : children.entrySet())
        {
            migratedConfigs += this.migrateNodeTree(previousPath + node.getKey(), childNode, oldConfigs);
        }

        return migratedConfigs;
    }

    /**
     * Read all configurations points.
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#readAllConfigurationsPoints()
     */
    @Override
    public void readAllConfigurationsPoints() throws ConfigurationException
    {
        cachedConfigurationsPackages.clear();

        cachedConfigurations = new HashMap<String, Class<?>>();
        cachedConfigurationsByVirtualPath = new HashMap<String, Class<?>>();

        List<String> basePackages = new ArrayList<String>();

        for (Package pck : Package.getPackages())
        {
            String basePackage = pck.getName().split("\\.", 2)[0];

            if (!basePackages.contains(basePackage))
            {
                basePackages.add(basePackage);
            }
        }

        for (String basePackage : basePackages)
        {
            this.readConfigurationsPointsForPackage(basePackage);
        }
    }

    /**
     * Read configuration.
     *
     * @param <T>   the generic type
     * @param clazz the clazz
     *
     * @return the t
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#readConfiguration(java.lang.Class)
     */
    @Override
    public <T> T readConfiguration(Class<T> clazz) throws ConfigurationException
    {

        if (clazz == null)
        {
            Logger.getLogger().error("The class cannot be null. Parse error!");
            return null;
        }
        else
        {
            String configID = this.getConfigID(clazz);
            String sectionID = this.getConfigSectionID(clazz);

            if ((configID != null) && (sectionID != null))
            {
                return this.readConfiguration(configID, sectionID, clazz);
            }
            else
            {
                return null;
            }
        }
    }

    /**
     * Read configuration.
     *
     * @param <T>       the generic type
     * @param configID  the config ID
     * @param sectionID the section ID
     * @param clazz     the clazz
     *
     * @return the t
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#readConfiguration(java.lang.String, java.lang.String,
     *         java.lang.Class)
     */
    @Override
    public <T> T readConfiguration(String configID, String sectionID, Class<T> clazz) throws ConfigurationException
    {

        // Will define if the configurations should be updated
        boolean mustUpdate = false;

        if (clazz == null)
        {
            Logger.getLogger().error("The class cannot be null. Parse error!");
            return null;
        }
        else
        {

            Properties props = this.readConfiguration(configID, sectionID);

            T obj;
            try
            {
                obj = ReflectionUtils.instantiateObjectFromClass(clazz);
            }
            catch (AuxiliaryOperationException e)
            {
                Logger.getLogger().error("Could not create the Configuration Instance:\n" + e.getMessage());
                return null;
            }

            List<ConfigItem> configItems = this.getConfigItemsMap(clazz);
            Object valueAsObject;
            String value;

            for (ConfigItem item : configItems)
            {
                if (props == null)
                {
                    value = null; // Will force the default value to be set if available
                }
                else
                {
                    value = props.getProperty(item.getKey()); // Read the value from the properties object
                }

                if ((props.isEmpty() || !props.containsKey(item.getKey())) && item.getDefaultValue() != null)
                {
                    mustUpdate = true;
                    value = item.getDefaultValue();
                }

                // If not found, or defined as default, set it with the default value
                if (value != null && value.equalsIgnoreCase(DEFAULT_VALUE_KEYWORK))
                {
                    value = item.getDefaultValue();
                }

                if (!StringUtils.isBlank(value))
                {
                    // The argument will be created with the previously determined compatible class constructor that can
                    // receive a String argument. This will then be passed to the setter method

                    try
                    {
                        if ("null".equals(value))
                        {
                            valueAsObject = null;
                        }
                        else if (item.getItemClass() == Date.class)
                        {
                            valueAsObject = DateUtils.stringToDate(value);
                        }
                        else
                        {
                            if (item.getItemClass() == Double.class)
                            {
                                value = value.replace(',', '.');
                            }
                            valueAsObject = ReflectionUtils.instantiateObjectFromClass(item.getConstructor(), value);
                        }

                        ReflectionUtils.invokeMethod(item.getSetter(), obj, valueAsObject);
                    }
                    catch (ParseException e)
                    {
                        Logger.getLogger().error("Could not convert to Date: \"" + value + "\"\n" + e.getMessage());
                        return null;
                    }
                    catch (AuxiliaryOperationException e)
                    {
                        Logger.getLogger()
                                .error("Could not call setter: " + item.getSetter().getName() + "\n" + e.getMessage());
                        return null;
                    }
                }
            }

            // Update the configurations if any value was missing...
            if (mustUpdate)
            {
                this.writeConfiguration(configID, sectionID, obj);
            }

            return obj;
        }
    }

    /**
     * Read configuration as map.
     *
     * @param configID  the config ID
     * @param sectionID the section ID
     *
     * @return the map
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#readConfigurationAsMap(java.lang.String, java.lang.String)
     */
    @Override
    public Map<String, String> readConfigurationAsMap(String configID, String sectionID) throws ConfigurationException
    {
        Properties properties = this.readConfiguration(configID, sectionID);
        Map<String, String> propObj = new HashMap<String, String>();
        Iterator<Object> iteratorProps = properties.keySet().iterator();

        while (iteratorProps.hasNext())
        {
            Object object = iteratorProps.next();
            propObj.put(object.toString(), properties.getProperty(object.toString()));
        }
        return propObj;
    }

    /**
     * Read configuration as properties.
     *
     * @param clazz the clazz
     *
     * @return the properties
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#readConfigurationAsProperties(java.lang.Class)
     */
    @Override
    public Properties readConfigurationAsProperties(Class<?> clazz) throws ConfigurationException
    {
        if (clazz == null)
        {
            Logger.getLogger().error("The class cannot be null. Parse error!");
            return null;
        }
        else
        {
            String configID = this.getConfigID(clazz);
            String sectionID = this.getConfigSectionID(clazz);

            return this.readConfiguration(configID, sectionID);
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readConfigurationFromVirtualPath(java.lang.String)
     */
    @Override
    public Properties readConfigurationFromVirtualPath(String configVirtualPath) throws ConfigurationException
    {
        Class<?> clazz = AbstractConfigurationsImpl.cachedConfigurationsByVirtualPath.get(configVirtualPath);

        return this.readConfiguration(this.getConfigID(clazz), this.getConfigSectionID(clazz));
    }

    /**
     * Read configurations points for package.
     *
     * @param basePackage the base package
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#readConfigurationsPointsForPackage(java.lang.String)
     */
    @Override
    public void readConfigurationsPointsForPackage(String basePackage) throws ConfigurationException
    {
        if (basePackage == null || "".equals(basePackage))
        {
            this.readAllConfigurationsPoints();
        }
        else if (cachedConfigurations == null || cachedConfigurationsByVirtualPath == null)
        {
            cachedConfigurations = new HashMap<String, Class<?>>();
            cachedConfigurationsByVirtualPath = new HashMap<String, Class<?>>();
            cachedConfigurationsPackages.clear();
        }

        if (!cachedConfigurationsPackages.contains(basePackage))
        {
            Logger.getLogger().debug("Reading configurations for package: \"" + basePackage + "\"");

            try
            {
                String configPath;
                String configVirtualPath;

                List<String> classes = ResourceUtils.getClassesInDeepPackage(basePackage);

                for (String className : classes)
                {
                    try
                    {
                        Class<?> clazz = Class.forName(className);

                        if (!this.ignoreTestConfigurations || !this.isTestConfigurationClass(clazz))
                        {
                            configPath = this.getConfigPath(clazz);

                            if (configPath != null)
                            {
                                cachedConfigurations.put(configPath, clazz);
                            }

                            configVirtualPath = this.getConfigVirtualPath(clazz);

                            if (configVirtualPath != null)
                            {
                                cachedConfigurationsByVirtualPath.put(configVirtualPath, clazz);
                            }
                        }
                    }
                    catch (Throwable e)
                    {
                        e.printStackTrace();
                        // Do nothing. Will discard all classes it cannot load
                    }
                }
            }
            catch (Throwable e)
            {
                // Do nothing. Will discard all classes it cannot load
            }

            cachedConfigurationsPackages.add(basePackage);
        }
    }

    /**
     * Removes the configuration.
     *
     * @param bean the bean
     *
     * @return true, if successful
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#removeConfiguration(java.lang.Object)
     */
    @Override
    public boolean removeConfiguration(Object bean) throws ConfigurationException
    {

        if (bean == null)
        {
            Logger.getLogger().error("The bean cannot be null. Parse error!");
            return false;
        }
        else
        {

            String configID = this.getConfigID(bean.getClass());
            String sectionID = this.getConfigSectionID(bean);

            if ((configID != null) && (sectionID != null))
            {
                return this.removeConfiguration(configID, sectionID);
            }
            else
            {
                return false;
            }
        }
    }

    /**
     * Write configuration.
     *
     * @param clazz  the clazz
     * @param values the values
     *
     * @return true, if successful
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#writeConfiguration(java.lang.Class, java.util.Properties)
     */
    @Override
    public boolean writeConfiguration(Class<?> clazz, Properties values) throws ConfigurationException
    {
        if (clazz == null)
        {
            Logger.getLogger().error("The class cannot be null. Parse error!");
            return false;
        }
        else
        {
            String configID = this.getConfigID(clazz);
            String sectionID = this.getConfigSectionID(clazz);

            return this.writeConfiguration(configID, sectionID, values);
        }
    }

    /**
     * Write configuration.
     *
     * @param annotatedPojo the annotated pojo
     *
     * @return true, if successful
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#writeConfiguration(java.lang.Object)
     */
    @Override
    public boolean writeConfiguration(Object annotatedPojo) throws ConfigurationException
    {

        if (annotatedPojo == null)
        {
            Logger.getLogger().error("The pojo cannot be null. Parse error!");
            return false;
        }
        else
        {

            String configID = this.getConfigID(annotatedPojo.getClass());
            String sectionID = this.getConfigSectionID(annotatedPojo);

            if ((configID != null) && (sectionID != null))
            {
                return this.writeConfiguration(configID, sectionID, annotatedPojo);
            }
            else
            {
                return false;
            }
        }
    }

    /**
     * Write configuration.
     *
     * @param configID  the config ID
     * @param sectionID the section ID
     * @param bean      the bean
     *
     * @return true, if successful
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#writeConfiguration(java.lang.String, java.lang.String,
     *         java.lang.Object)
     */
    @Override
    public boolean writeConfiguration(String configID, String sectionID, Object bean) throws ConfigurationException
    {

        if (bean == null)
        {
            Logger.getLogger().error("The bean cannot be null. Parse error!");
            return false;
        }
        else
        {
            Properties props = new Properties();
            Object value;
            String valueString;

            List<ConfigItem> configItems = this.getConfigItemsMap(bean.getClass());

            try
            {
                for (ConfigItem item : configItems)
                {
                    value = ReflectionUtils.invokeMethod(item.getGetter(), bean);
                    valueString = "";

                    if (value != null)
                    {
                        if (item.getItemClass() == Date.class)
                        {
                            valueString = DateUtils.dateToString((Date) value);
                        }
                        else
                        {
                            valueString = value.toString();
                        }
                    }

                    if (valueString.equals(item.getDefaultValue()))
                    {
                        valueString = DEFAULT_VALUE_KEYWORK;
                    }

                    props.setProperty(item.getKey(), valueString);
                }
            }
            catch (AuxiliaryOperationException e)
            {
                return false;
            }

            return this.writeConfiguration(configID, sectionID, props);
        }
    }

    /**
     * Write configuration from map.
     *
     * @param configID  the config ID
     * @param sectionID the section ID
     * @param values    the values
     *
     * @return true, if successful
     *
     * @exception ConfigurationException the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#writeConfigurationFromMap(java.lang.String, java.lang.String,
     *         java.util.Map)
     */
    @Override
    public boolean writeConfigurationFromMap(String configID, String sectionID, Map<String, String> values)
            throws ConfigurationException
    {
        if (values == null)
        {
            Logger.getLogger().error("The values map cannot be null. Parse error!");
            return false;
        }
        Properties props = new Properties();
        for (Entry<String, String> entrie : values.entrySet())
        {
            props.put(entrie.getKey(), StringUtils.nvl(entrie.getValue(), ""));
        }
        return this.writeConfiguration(configID, sectionID, props);
    }
}
