/**
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.utils.config;

import pt.digitalis.log.Logger;
import pt.digitalis.utils.common.StringUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

/**
 * The Class ConfigurationsPreferencesImpl.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Sep 27, 2007
 */
public class ConfigurationsPreferencesImpl extends AbstractConfigurationsImpl
{

    /** if T, prefixed configurations will be active */
    static private boolean prefixedConfigurationsActive = false;

    /**
     * The prefixed specific configuration exception. The path specified here will be read from the common configuration
     * node, instead of the specific one
     */
    static private Map<String, List<String>> prefixException = new HashMap<String, List<String>>();

    /**
     * Instantiates a new configurations preferences impl.
     *
     * @exception ConfigurationException
     */
    public ConfigurationsPreferencesImpl() throws ConfigurationException
    {
        this(true, false);
    }

    /**
     * Instantiates a new configurations preferences impl.
     *
     * @param ignoreTestConfigurations the ignore test configurations
     *
     * @exception ConfigurationException the configuration exception
     */
    public ConfigurationsPreferencesImpl(boolean ignoreTestConfigurations) throws ConfigurationException
    {
        this(ignoreTestConfigurations, false);
    }

    /**
     * Instantiates a new configurations preferences impl.
     *
     * @param ignoreTestConfigurations          the ignore test configurations
     * @param forcePrefixedConfigurationsActive the force prefixed configurations active
     *
     * @exception ConfigurationException the configuration exception
     */
    public ConfigurationsPreferencesImpl(boolean ignoreTestConfigurations, boolean forcePrefixedConfigurationsActive)
            throws ConfigurationException
    {
        super(ignoreTestConfigurations);

        // Read general configurations
        Properties props = readConfiguration("Configurations", "General", false);

        // General configurations
        prefixedConfigurationsActive = Boolean.parseBoolean(props.getProperty("PrefixedConfigurationsActive", "false"));
        props.put("PrefixedConfigurationsActive", prefixedConfigurationsActive);

        // Persist, thus creating them if it is the first time
        // TODO: ComQuest: Viegas: 1. Place code to prevent writing if the value has not changed
        writeConfiguration("Configurations", "General", props, false);

        // Custom exclusions for registered prefixes
        props = readConfiguration("Configurations", "PrefixExceptions", false);

        for (Entry<Object, Object> property : props.entrySet())
        {
            if (property.getKey() != null && property.getValue() != null)
            {
                String prefix = property.getKey().toString();
                String[] exceptions = property.getValue().toString().split(",");

                prefixException.put(prefix, Arrays.asList(exceptions));
            }
        }

        if (forcePrefixedConfigurationsActive)
            prefixedConfigurationsActive = true;
    }

    /**
     * Gets the childrens.
     *
     * @param configID  the config ID
     * @param sectionID the section ID
     *
     * @return the childrens
     *
     * @exception BackingStoreException the backing store exception
     */
    private String[] getChildrens(String configID, String sectionID) throws BackingStoreException
    {
        if (sectionID != null && !"".equals(sectionID))
            configID += "/" + sectionID;

        boolean usePrefix = isPrefixedConfiguration(configID);

        if (usePrefix)
            configID = getGeneralPrefix() + configID;

        if (StringUtils.isNotBlank(configID) && !configID.startsWith("/"))
            configID = "/" + configID;

        if ("/".equals(configID))
            configID = "";

        if (configID.endsWith("/"))
            configID = configID.substring(0, configID.length() - 1);

        Preferences systemPrefs = getPreferences(configID);

        return systemPrefs.childrenNames();
    }

    /**
     * Gets the general prefix.
     *
     * @return the general prefix
     *
     * @see pt.digitalis.utils.config.AbstractConfigurationsImpl#getGeneralPrefix()
     */
    @Override
    public String getGeneralPrefix()
    {
        return super.getGeneralPrefix() + "/";
    }

    /**
     * Gets the preferences.
     *
     * @param configID the config id
     *
     * @return the preferences
     */
    private Preferences getPreferences(String configID)
    {

        // WORKAROUND - Nodes with "_" char have the name replaced by Base64 full name encryption
        // SOURCE: Javadoc for FileSystemPreferences.isDirChar
        /**
         * Returns true if the specified character is appropriate for use in Unix directory names. A character is
         * appropriate if it's a printable ASCII character (> 0x1f && < 0x7f) and unequal to slash ('/', 0x2f), dot
         * ('.', 0x2e), or underscore ('_', 0x5f).
         */

        Preferences result;

        if (System.getProperty("os.name").equalsIgnoreCase("Mac OS X"))
            result = Preferences.userRoot();
        else
            result = Preferences.systemRoot();

        if (StringUtils.isNotBlank(configID))
        {
            String node = configID.replaceAll("_", "-");
            result = result.node(node);
        }

        return result;
    }

    /**
     * Initialize.
     *
     * @param upgradeModelManagers
     *
     * @see IConfigurations#initialize(boolean)
     */
    @Override
    public void initialize(boolean upgradeModelManagers)
    {
        // Nothing to do
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#isPersistentRepositoryEmpty()
     */
    @Override
    public boolean isPersistentRepositoryEmpty()
    {
        // Will always return not empty. The repository is created on first read.
        return false;
    }

    /**
     * @param configPath the config path to validate
     *
     * @return T if the specific configuration path should be perfixed for custom (prefixed) or common (not prefixed)
     *         configurations
     */
    protected boolean isPrefixedConfiguration(String configPath)
    {
        if (!isPrefixedConfigurationsActive())
            return false;
        else
        {
            List<String> exceptions = prefixException.get(generalConfigurationPrefix);

            if (exceptions == null)
                return true;
            else
            {
                for (String exceptionPath : exceptions)
                {
                    if (configPath.startsWith(exceptionPath))
                        return false;
                }

                return true;
            }
        }
    }

    /**
     * Inspector for the 'prefixedConfigurationsActive' attribute.
     *
     * @return the prefixedConfigurationsActive value
     */
    public boolean isPrefixedConfigurationsActive()
    {
        return prefixedConfigurationsActive;
    }

    /**
     * Read all configurations.
     *
     * @param configID  the config ID
     * @param sectionID the section ID
     *
     * @return the map
     *
     * @see pt.digitalis.utils.config.IConfigurations#readAllConfigurations(java.lang.String, java.lang.String)
     */
    @Override
    public Map<String, Properties> readAllConfigurations(String configID, String sectionID)
            throws ConfigurationException
    {
        Map<String, Properties> result = new HashMap<String, Properties>();

        boolean usePrefix =
                isPrefixedConfiguration(configID + (StringUtils.isNotBlank(sectionID) ? "/" + sectionID : ""));
        try
        {
            for (String name : getChildrens(configID, sectionID))
            {
                if (StringUtils.isBlank(configID))
                {
                    configID = sectionID;
                    sectionID = "";
                }

                result.put(name, this.readConfiguration(configID, name, usePrefix));
            }
        }
        catch (BackingStoreException e)
        {
            throw new ConfigurationException(e);
        }

        return result;
    }

    /**
     * Read all configurations.
     *
     * @param <T>       the generic type
     * @param configID  the config ID
     * @param sectionID the section ID
     * @param clazz     the clazz
     *
     * @return the map
     *
     * @see pt.digitalis.utils.config.IConfigurations#readAllConfigurations(java.lang.String, java.lang.String,
     *         java.lang.Class)
     */
    @Override
    public <T> Map<String, T> readAllConfigurations(String configID, String sectionID, Class<T> clazz)
            throws ConfigurationException
    {
        Map<String, T> result = new HashMap<String, T>();

        try
        {
            String basePath = StringUtils.isBlank(sectionID) ? "" : sectionID + "/";

            for (String name : getChildrens(configID, sectionID))
            {
                result.put(name, this.readConfiguration(configID, basePath + name, clazz));
            }
        }
        catch (BackingStoreException e)
        {
            throw new ConfigurationException(e);
        }

        return result;
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#readConfiguration(java.lang.String, java.lang.String)
     */
    @Override
    public Properties readConfiguration(String configID, String sectionID) throws ConfigurationException
    {
        return readConfiguration(configID, sectionID, isPrefixedConfiguration(configID + "/" + sectionID));
    }

    /**
     * Reads configurations from the persistence layer
     *
     * @param configID  the identifier of the configuration group
     * @param sectionID the identifier of the section within the group
     * @param usePrefix if the prefix should be used for reading
     *
     * @return a Properties object with the key value pairs
     *
     * @exception ConfigurationException
     */
    private Properties readConfiguration(String configID, String sectionID, boolean usePrefix)
            throws ConfigurationException
    {
        if (StringUtils.isBlank(configID))
        {
            configID = sectionID;
            sectionID = "";
        }
        if (usePrefix)
            configID = getGeneralPrefix() + configID;

        if (sectionID != null && !"".equals(sectionID))
            configID += "/" + sectionID;

        if (!configID.startsWith("/"))
            configID = "/" + configID;

        if (configID.startsWith("//"))
            configID = configID.substring(1);

        if (configID.endsWith("/"))
            configID = configID.substring(0, configID.length() - 1);

        Preferences systemPrefs = getPreferences(configID);
        Properties propObj = new Properties();

        try
        {
            systemPrefs.sync();
            StringBuffer out = new StringBuffer();
            for (String key : systemPrefs.keys())
            {
                if (StringUtils.isNotBlank(key) && systemPrefs.get(key, null) != null)
                {
                    propObj.put(key, systemPrefs.get(key, null));
                    if (key.toLowerCase().contains("password"))
                    {
                        out.append("  " + key + "=[*******]\n");
                    }
                    else
                    {
                        out.append("  " + key + "=[" + propObj.getProperty(key) + "]\n");
                    }
                }
            }
            if (out.length() > 0)
            {
                Logger.getLogger().debug("Configuration " + configID + ":\n" + out.toString());
            }

            return propObj;
        }
        catch (BackingStoreException e)
        {
            throw new ConfigurationException(e);
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#removeConfiguration(java.lang.String, java.lang.String)
     */
    @Override
    public boolean removeConfiguration(String configID, String sectionID) throws ConfigurationException
    {
        return removeConfiguration(configID, sectionID, isPrefixedConfiguration(configID + "/" + sectionID));
    }

    /**
     * Removes configurations from the persistence layer
     *
     * @param configID  the identifier of the configuration group
     * @param sectionID the identifier of the section within the group
     * @param usePrefix if the prefix should be used for reading
     *
     * @return T if successful
     *
     * @exception ConfigurationException
     */
    public boolean removeConfiguration(String configID, String sectionID, boolean usePrefix)
            throws ConfigurationException
    {
        if (StringUtils.isBlank(configID))
        {
            configID = sectionID;
            sectionID = "";
        }
        if (usePrefix)
            configID = getGeneralPrefix() + configID;

        if (sectionID != null && !"".equals(sectionID))
            configID += "/" + sectionID;

        if (!configID.startsWith("/"))
            configID = "/" + configID;

        try
        {
            Preferences systemPrefs = getPreferences(configID);
            systemPrefs.removeNode();
            systemPrefs.flush();

            return true;
        }
        catch (BackingStoreException e)
        {
            throw new ConfigurationException(e);
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#removeConfigurationParameter(java.lang.String,
     *         java.lang.String,
     *         java.lang.String)
     */
    @Override
    public boolean removeConfigurationParameter(String configID, String sectionID, String paramName)
            throws ConfigurationException
    {
        return removeConfigurationParameter(configID, sectionID, paramName,
                isPrefixedConfiguration(configID + "/" + sectionID));
    }

    /**
     * Removes the configuration parameter.
     *
     * @param configID  the config ID
     * @param sectionID the section ID
     * @param paramName the param name
     * @param usePrefix the use prefix
     *
     * @return true, if successful
     *
     * @exception ConfigurationException the configuration exception
     */
    public boolean removeConfigurationParameter(String configID, String sectionID, String paramName, boolean usePrefix)
            throws ConfigurationException
    {
        if (StringUtils.isBlank(configID))
        {
            configID = sectionID;
            sectionID = "";
        }
        if (usePrefix)
            configID = getGeneralPrefix() + configID;

        if (sectionID != null && !"".equals(sectionID))
            configID += "/" + sectionID;

        if (!configID.startsWith("/"))
            configID = "/" + configID;

        try
        {
            Preferences systemPrefs = getPreferences(configID);
            systemPrefs.remove(paramName);
            systemPrefs.flush();

            return true;
        }
        catch (BackingStoreException e)
        {
            throw new ConfigurationException(e);
        }
    }

    /**
     * @see pt.digitalis.utils.config.IConfigurations#writeConfiguration(java.lang.String, java.lang.String,
     *         java.util.Properties)
     */
    @Override
    public boolean writeConfiguration(String configID, String sectionID, Properties values)
            throws ConfigurationException
    {
        return writeConfiguration(configID, sectionID, values, isPrefixedConfiguration(configID + "/" + sectionID));
    }

    /**
     * Writes configurations to the persistence layer
     *
     * @param configID  the identifier of the configuration group
     * @param sectionID the identifier of the section within the group
     * @param values    a Properties object with the key value pairs
     * @param usePrefix if the prefix should be used for writing
     *
     * @return T if the operation was successful
     *
     * @exception ConfigurationException
     */
    public boolean writeConfiguration(String configID, String sectionID, Properties values, boolean usePrefix)
            throws ConfigurationException
    {
        if (StringUtils.isBlank(configID))
        {
            configID = sectionID;
            sectionID = "";
        }
        if (usePrefix)
            configID = getGeneralPrefix() + configID;

        if (sectionID != null && !"".equals(sectionID))
            configID += "/" + sectionID;

        if (!configID.startsWith("/"))
            configID = "/" + configID;

        Preferences systemPrefs = getPreferences(configID);

        for (Entry<Object, Object> entry : values.entrySet())
        {
            if (entry.getValue() != null)
                systemPrefs.put(entry.getKey().toString(), entry.getValue().toString());
        }

        try
        {
            systemPrefs.flush();

            return true;
        }
        catch (BackingStoreException e)
        {
            throw new ConfigurationException(e);
        }
    }
}
